package ru.yandex.canvas.service.rtbhost.helpers.creative;

import java.io.IOException;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.canvas.Html5Constants;
import ru.yandex.canvas.exceptions.InternalServerError;
import ru.yandex.canvas.model.Size;
import ru.yandex.canvas.model.html5.Creative;
import ru.yandex.canvas.model.html5.Source;
import ru.yandex.canvas.service.DirectService;
import ru.yandex.canvas.service.SessionParams;
import ru.yandex.canvas.service.html5.Html5Preset;
import ru.yandex.canvas.service.html5.Html5Zip;
import ru.yandex.canvas.service.html5.RtbhostedHtml;
import ru.yandex.canvas.service.rtbhost.RtbHostUploadHelper;
import ru.yandex.direct.bs.dspcreative.model.DspCreativeExportEntry;
import ru.yandex.direct.feature.FeatureName;

import static ru.yandex.canvas.Html5Constants.EXTRA_STRETCHABLE_PRESETS_CPM_PRICE;
import static ru.yandex.canvas.service.SessionParams.SessionTag.PLAYABLE;

public class Html5DspUploadHelper implements RtbHostUploadHelper<Creative> {
    private static final Logger logger = LoggerFactory.getLogger(Html5DspUploadHelper.class);
    private final DirectService directService;

    public Html5DspUploadHelper(DirectService directService) {
        this.directService = directService;
    }

    private String constructorDataByCreative(Creative creative, ObjectMapper objectMapper, Set<String> features, Boolean isAdaptive) {
        try {
            return objectMapper.writeValueAsString(new Html5ConstructorData(creative,
                    features.contains(FeatureName.SOCIAL_ADVERTISING.getName()) ? true : null, isAdaptive));
        } catch (JsonProcessingException e) {
            logger.error("can't build creative data json", e);
            throw new InternalServerError("can't build creative data json");
        }
    }

    @Override
    public DspCreativeExportEntry toImportDspCreativeEntry(Creative creative,
                                                           ObjectMapper objectMapper) {
        Source src = creative.getSource();

        Html5Zip html5Zip;
        try {
            html5Zip = src.unzipArchiveContent();
        } catch (IOException e) {
            throw new InternalServerError("Couldn't get archive");
        }

        var features = directService.getFeatures(src.getClientId(), null);

        String htmlContent = html5Zip.getFileAsUtf8String(src.getHtmlFilename());
        RtbhostedHtml rtbhostedHtml = new RtbhostedHtml(htmlContent, src.getHtmlReplacements(),
                features, SessionParams.getHtml5SessionTag(creative.getProductType()));
        rtbhostedHtml.setRtbHost(true);

        Boolean isAdaptive = EXTRA_STRETCHABLE_PRESETS_CPM_PRICE.stream()
                .anyMatch(s -> s.getId().equals(creative.getSource().getPresetId()));

        try {
            return DspCreativeExportEntry.builder()
                    .setDspId(1L)
                    .setCreativeId(creative.getId())
                    .setCreativeVersionId(creative.getId())
                    .setData(rtbhostedHtml.getPreview(
                            "${AUCTION_DC_PARAMS}")) // ☠️ This is the only value which BS can recognise! Don't
                    // change it, never!
                    .setStaticData(String.format("{\"creative_id\":\"%d\"}", creative.getId()))
                    .setConstructorData(constructorDataByCreative(creative, objectMapper, features, isAdaptive))
                    .setEnabled(true)
                    .setVideo(false)
                    .setAudio(false)
                    .setPostmoderated(true)
                    .setStatic(false)
                    .setTag("yabs")
                    .setWidth(creative.getWidth())
                    .setHeight(creative.getHeight())
                    .setIsAdaptive(isAdaptive)
                    .build();
        } catch (IOException e) {
            logger.error("Couldn't get rtb html template", e);
            throw new InternalServerError("Couldn't get rtb html template");
        } catch (URISyntaxException e) {
            logger.error("Bad URL", e);
            throw new InternalServerError("Bad URL");
        }
    }

    /*
    See https://st.yandex-team.ru/DIRECT-90642
    objects just to generate JSON
   */
    public static class Html5ConstructorData {
        @JsonProperty
        @JsonInclude(JsonInclude.Include.NON_NULL)
        private List<Html5ConstructorDataElement> elements;

        @JsonProperty
        @JsonInclude(JsonInclude.Include.NON_NULL)
        private String product;

        @JsonInclude(JsonInclude.Include.NON_NULL)
        @JsonProperty
        private Boolean isSocialAdvertising;

        @JsonProperty("OnlyClickUrlsMacro")
        private Boolean onlyClickUrlsMacro = true;

        @JsonInclude(JsonInclude.Include.NON_NULL)
        @JsonProperty
        private Boolean isLarge;
        /**
         * !! Pass into json this keys for Adaptive creatives, ONLY !!!
         */
        @JsonInclude(JsonInclude.Include.NON_NULL)
        private Integer minWidth;
        @JsonInclude(JsonInclude.Include.NON_NULL)
        private Integer minHeight;
        @JsonInclude(JsonInclude.Include.NON_NULL)
        private Integer maxWidth;
        @JsonInclude(JsonInclude.Include.NON_NULL)
        private Integer maxHeight;

        Html5ConstructorData(Creative creative, Boolean isSocialAdvertising, Boolean isAdaptive) {
            Source source = creative.getSource();
            this.isSocialAdvertising = isSocialAdvertising;

            this.elements = new ArrayList<>();
            if (source.getHtmlUrl() != null) {
                this.elements.add(new Html5ConstructorDataElement("index", source.getHtmlUrl()));
            }
            if (source.getExpandedHtmlUrl() != null) {
                this.elements.add(new Html5ConstructorDataElement("expand",
                        source.getExpandedHtmlUrl(), source.getBgrcolor()));
            }
            //для креатива из картинки всё-равно нужно image и index
            if (creative.getSourceImageInfo() != null) {
                this.elements.add(new Html5ConstructorDataElement(creative.getSourceImageInfo().getUrl()));
            }
            if (this.elements.isEmpty()) {
                this.elements = null;//так будет чище json в RTB для playable или медийных немордячих
            }
            if (creative.getProductType() != null && creative.getProductType() == PLAYABLE) {
                this.product = "PLAYABLE";
            }

            /**
             * Для адаптивных (===перетяжки) HTML5 баннеров в ConstructorData нужно передавать
             * min/max значения ширины и/или высоты. Они в РТБ хосте являют более приоритетными
             * по отношению к ширине/высоте в отдельных полях DSPCreatives
             */
            if (Boolean.TRUE.equals(isAdaptive)) {
                this.minHeight = creative.getHeight();
                this.maxHeight = creative.getHeight();
                this.minWidth = 320;
                this.maxWidth = 2800;
            }

            // Для премиум креатива передавать флаг isLarge=true
            if (Html5Constants.PREMIUM_FORMATS.stream().map(Html5Preset::getSize).anyMatch(e -> e.equals(new Size(creative.getWidth(), creative.getHeight())))) {
                this.isLarge = true;
            }
        }

        public List<Html5ConstructorDataElement> getElements() {
            return elements;
        }

        public String getProduct() {
            return product;
        }

        public Integer getMinWidth() {
            return minWidth;
        }

        public Integer getMinHeight() {
            return minHeight;
        }

        public Integer getMaxWidth() {
            return maxWidth;
        }

        public Integer getMaxHeight() {
            return maxHeight;
        }
    }

    private static class Html5ConstructorDataElement {
        @JsonProperty
        private String type = "image";
        @JsonProperty
        private String url;
        @JsonProperty
        @JsonInclude(JsonInclude.Include.NON_NULL)
        private String bgcolor;

        public Html5ConstructorDataElement(String url) {
            this.url = url;
        }

        public Html5ConstructorDataElement(String type, String url) {
            this.type = type;
            this.url = url;
        }

        public Html5ConstructorDataElement(String type, String url, String bgrcolor) {
            this.type = type;
            this.url = url;
            bgcolor = bgrcolor;
            if (bgcolor != null && !bgcolor.startsWith("#")) {
                bgcolor = "#" + bgcolor;//такой формат нужен морде чтобы консистентно с awaps работать
            }
        }

        public String getUrl() {
            return url;
        }

        public String getType() {
            return type;
        }

        public String getBgcolor() {
            return bgcolor;
        }
    }

}
