package ru.yandex.canvas.service.rtbhost.helpers.creative;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;

import ru.yandex.adv.direct.CreativeParameters;
import ru.yandex.adv.direct.media.HtmlParameters;
import ru.yandex.adv.direct.video.VideoParameters;
import ru.yandex.canvas.TimeDelta;
import ru.yandex.canvas.exceptions.NotFoundException;
import ru.yandex.canvas.model.CreativeData;
import ru.yandex.canvas.model.CreativeDocument;
import ru.yandex.canvas.model.elements.ElementType;
import ru.yandex.canvas.model.html_builder.InBannerHtmlCreativeWriter;
import ru.yandex.canvas.model.presets.Preset;
import ru.yandex.canvas.model.presets.PresetItem;
import ru.yandex.canvas.model.presets.PresetSelectionCriteria;
import ru.yandex.canvas.model.video.files.InBannerVideo;
import ru.yandex.canvas.repository.video.VideoFilesRepository;
import ru.yandex.canvas.service.PresetsService;
import ru.yandex.canvas.service.rtbhost.CreativeParametersSerializer;
import ru.yandex.canvas.service.rtbhost.HtmlParametersBuilder;
import ru.yandex.canvas.service.rtbhost.RtbHostUploadHelper;
import ru.yandex.canvas.service.video.InBannerVideoFilesService;
import ru.yandex.canvas.service.video.VideoParametersBuilder;
import ru.yandex.direct.bs.dspcreative.model.DspCreativeExportEntry;

import static java.lang.Math.floor;
import static ru.yandex.canvas.VideoConstants.DEFAULT_VPAID_PCODE_URL;
import static ru.yandex.canvas.model.html_builder.InBannerHtmlCreativeWriter.filterBrokenFormats;
import static ru.yandex.canvas.service.video.InBannerVideoFilesService.DEFAULT_PICTURE_URL;

public class InBannerDspUploadHelper implements RtbHostUploadHelper<CreativeDocument> {
    private final InBannerHtmlCreativeWriter inBannerHtmlCreativeWriter;
    private final InBannerVideoFilesService inBannerVideoFilesService;
    private final PresetsService presetsService;

    public InBannerDspUploadHelper(InBannerHtmlCreativeWriter inBannerHtmlCreativeWriter,
                                   InBannerVideoFilesService inBannerVideoFilesService,
                                   PresetsService presetsService) {
        this.inBannerHtmlCreativeWriter = inBannerHtmlCreativeWriter;
        this.inBannerVideoFilesService = inBannerVideoFilesService;
        this.presetsService = presetsService;
    }

    public DspCreativeExportEntry toImportDspCreativeEntry(CreativeDocument creativeDocument,
                                                           ObjectMapper objectMapper) {

        String htmlContent = inBannerHtmlCreativeWriter.composeCreativeHTML(creativeDocument);

        InBannerVideo videoFile = inBannerVideoFilesService.lookupMovie(creativeDocument,
                new VideoFilesRepository.QueryBuilder());
        InBannerConstructorData inBannerConstructorData = new InBannerConstructorData();
        inBannerConstructorData.setDuration((long) floor(videoFile.getDuration() + 0.5));

        var creativeParameters = CreativeParameters.newBuilder()
                .setHtml(buildHtmlParameters(creativeDocument))
                .setVideo(buildVideoParameters(creativeDocument))
                .build();

        inBannerConstructorData.setCreativeParameters(creativeParameters);

        try {
            return DspCreativeExportEntry.builder()
                    .setDspId(1L)
                    .setCreativeId(creativeDocument.getId())
                    .setCreativeVersionId(creativeDocument.getId())
                    .setData(htmlContent)
                    .setConstructorData(objectMapper.writeValueAsString(inBannerConstructorData))
                    .setStaticData(String.format("{\"creative_id\":\"%d\"}", creativeDocument.getId()))
                    .setEnabled(true)
                    .setStatic(false)
                    .setTag("yabs")
                    .setPostmoderated(true)
                    .setVideo(false)
                    .setAudio(false)
                    .setWidth(creativeDocument.getData().getWidth())
                    .setHeight(creativeDocument.getData().getHeight())
                    .setIsAdaptive(false)
                    .setIsGeoPin(false)
                    .build();
        } catch (JsonProcessingException e) {
            throw new RuntimeException(e);
        }
    }

    private HtmlParameters buildHtmlParameters(CreativeDocument creativeDocument) {
        HtmlParametersBuilder builder = new HtmlParametersBuilder();
        var visibleData = creativeDocument.getData().onlyVisibleData();
        patchDefaultPictures(visibleData);
        builder.setCreativeData(visibleData);

        PresetItem presetItem = getPresetItem(creativeDocument);
        builder.setPresetItem(presetItem);

        return builder.build();
    }

    private void patchDefaultPictures(CreativeData visibleData) {
        if (visibleData.getMediaSets() == null) {
            return;
        }
        visibleData.getMediaSets().forEach((k, v) -> {
            v.getItems().stream()
                    .filter(item -> item.getType() == ElementType.IMAGE ||
                            item.getType() == ElementType.SECOND_IMAGE)
                    .forEach(item -> item.getItems().forEach(subItem -> subItem.setUrl(DEFAULT_PICTURE_URL)));
        });
    }

    private PresetItem getPresetItem(CreativeDocument creativeDocument) {
        Preset preset = presetsService.getById(creativeDocument.getPresetId(),
                PresetSelectionCriteria.builder().withCpmCommon(true).build(), null).orElseThrow(NotFoundException::new);

        return preset.getItems().stream()
                .filter(e -> e.getId() == creativeDocument.getItemId())
                .findFirst()
                .orElseThrow(NotFoundException::new);
    }

    private VideoParameters buildVideoParameters(CreativeDocument creativeDocument) {
        InBannerVideo inBannerVideo = inBannerVideoFilesService.lookupMovie(creativeDocument,
                new VideoFilesRepository.QueryBuilder());

        VideoParametersBuilder builder = new VideoParametersBuilder();
        builder.setCreativeId(creativeDocument.getId());
        builder.setDuration(inBannerVideo.getDuration());
        builder.setMediaFiles(filterBrokenFormats(inBannerVideo.getFormats()));
        builder.setPlaybackParameters(false, new TimeDelta(5));
        builder.setAddPixelImpression(false);
        builder.setUseVpaidImpressions(false);
        builder.setUseTrackingEvents(false);
        builder.setVpaidPcodeUrl(DEFAULT_VPAID_PCODE_URL);
        builder.setStrmPrefix(inBannerVideo.getStrmPrefix());
        builder.setShowVideoClicks(true);

        return builder.build();
    }

    public static class InBannerConstructorData {
        @JsonProperty("duration")
        long duration;

        @JsonProperty("isInBanner")
        boolean isInBanner = true;

        @JsonInclude(JsonInclude.Include.NON_NULL)
        @JsonProperty("creative_parameters")
        @JsonSerialize(using = CreativeParametersSerializer.class)
        private CreativeParameters creativeParameters;

        public long getDuration() {
            return duration;
        }

        public void setDuration(long duration) {
            this.duration = duration;
        }

        public void setCreativeParameters(CreativeParameters creativeParameters) {
            this.creativeParameters = creativeParameters;
        }
    }

}
