package ru.yandex.canvas.service.sandbox;

import java.util.Arrays;
import java.util.Collection;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

@ParametersAreNonnullByDefault
public class ClientTagsQueryBuilder {
    private static final String AND = " & ";
    private static final String OR = " | ";
    private static final String NOT = "~";

    private ClientTagsQueryBuilder() {
    }

    public static ClientTagsQuery or(ClientTagsQuery... tags) {
        return new Or(Arrays.asList(tags));
    }

    public static ClientTagsQuery and(ClientTagsQuery... queries) {
        return new And(Arrays.asList(queries));
    }

    public static ClientTagsQuery not(ClientTag tag) {
        return new Not(tag);
    }

    static class Or extends BinaryOperatorQuery {
        Or(Collection<? extends ClientTagsQuery> queries) {
            super(queries);
        }

        @Override
        CharSequence delimiter() {
            return OR;
        }
    }

    static class And extends BinaryOperatorQuery {
        And(Collection<? extends ClientTagsQuery> queries) {
            super(queries);
        }

        @Override
        CharSequence delimiter() {
            return AND;
        }
    }

    abstract static class BinaryOperatorQuery implements ClientTagsQuery {
        private Collection<? extends ClientTagsQuery> tags;

        BinaryOperatorQuery(Collection<? extends ClientTagsQuery> tags) {
            this.tags = tags;
        }

        boolean isMulti() {
            return tags.size() > 1;
        }

        @Override
        public String build() {
            return tags.stream()
                    .map(ClientTagsQuery::buildWrapped)
                    .collect(Collectors.joining(delimiter()));
        }

        @Override
        public String buildWrapped() {
            return isMulti() ? "(" + build() + ")" : build();
        }

        abstract CharSequence delimiter();
    }

    static class Not implements ClientTagsQuery {
        private ClientTagsQuery query;

        Not(ClientTagsQuery query) {
            this.query = query;
        }

        @Override
        public String build() {
            return NOT + query.buildWrapped();
        }

        @Override
        public String buildWrapped() {
            return build();
        }
    }
}
