package ru.yandex.canvas.service.screenshooters;

import javax.annotation.Nullable;

import com.mongodb.client.result.UpdateResult;
import org.springframework.data.mongodb.core.MongoOperations;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;

import ru.yandex.canvas.model.CreativeDocument;
import ru.yandex.canvas.model.CreativeDocumentBatch;
import ru.yandex.canvas.model.direct.DirectUploadableHelperService;
import ru.yandex.canvas.model.html_builder.InBannerHtmlCreativeWriter;
import ru.yandex.canvas.repository.mongo.MongoOperationsWrapper;
import ru.yandex.canvas.service.DirectService;
import ru.yandex.canvas.service.PresetsService;
import ru.yandex.canvas.service.ScreenshooterService;
import ru.yandex.canvas.service.direct.CreativeUploadHelperFacade;
import ru.yandex.direct.screenshooter.client.model.ScreenShooterScreenshot;

import static org.springframework.data.mongodb.core.query.Criteria.where;

public class CreativesScreenshooterHelperService extends AbstractScreenshooterHelperService<CreativeDocument> {

    private final ScreenshooterService screenshooterService;
    private final MongoOperationsWrapper mongoOperation;
    private final CreativeUploadHelperFacade creativeUploadHelperFacade;
    private final InBannerHtmlCreativeWriter inBannerHtmlCreativeWriter;


    public CreativesScreenshooterHelperService(ScreenshooterService screenshooterService,
                                               MongoOperations mongoOperation, DirectService directService,
                                               CreativeUploadHelperFacade creativeUploadHelperFacade,
                                               InBannerHtmlCreativeWriter inBannerHtmlCreativeWriter) {
        super(directService);
        this.screenshooterService = screenshooterService;
        this.mongoOperation = new MongoOperationsWrapper(mongoOperation, "creatives_screenshooter_service");
        this.creativeUploadHelperFacade = creativeUploadHelperFacade;
        this.inBannerHtmlCreativeWriter = inBannerHtmlCreativeWriter;
    }

    @Override
    protected ScreenShooterScreenshot takeScreenshotFromScreenshooter(CreativeDocument creative) {
        if (creative.getPresetId() != null && PresetsService.IN_BANNER_PRESET_IDS.contains(creative.getPresetId())) {
            String html = inBannerHtmlCreativeWriter.composeCreativeHtmlForScreenshot(creative);

            return screenshooterService.getScreenshotFromHtml(html, creative.getData().getWidth(),
                    creative.getData().getHeight());
        }

        return screenshooterService.getScreenshotFromUrl(creative.getCreativeURL(),
                creative.getData().getWidth(), creative.getData().getHeight());
    }

    @Override
    protected UpdateResult updateCreativeScreenshotUrl(CreativeDocument creative, String screenshotUrl,
                                                       Boolean screenshotIsDone, @Nullable Long userId, Long clientId) {
        creative.setScreenshotURL(screenshotUrl);
        creative.setScreenshotIsDone(screenshotIsDone);

        return mongoOperation.updateFirst(Query.query(
                where("items._id").is(creative.getId())),
                new Update()
                        .set("items.$.screenshotURL", screenshotUrl)
                        .set("items.$.screenshotIsDone", screenshotIsDone),
                CreativeDocumentBatch.class);
    }

    @Override
    protected DirectUploadableHelperService<CreativeDocument> getDirectUploadHelper() {
        return creativeUploadHelperFacade;
    }

    @Override
    protected boolean needReuploadToDirect(CreativeDocument creative) {
        return creative.getId() != 0;
    }
}
