package ru.yandex.canvas.service.video;

import java.io.IOException;
import java.net.URI;
import java.sql.Date;
import java.time.Instant;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import org.springframework.data.domain.Sort;

import ru.yandex.canvas.model.stillage.StillageFileInfo;
import ru.yandex.canvas.model.video.AudioFiles;
import ru.yandex.canvas.model.video.files.AudioSource;
import ru.yandex.canvas.repository.ItemsWithTotal;
import ru.yandex.canvas.repository.video.AudioFilesRepository;
import ru.yandex.canvas.service.SandBoxService;
import ru.yandex.canvas.service.StillageService;

import static ru.yandex.canvas.service.SandBoxService.VideoTaskExtraFields.AUDIO_WEB_HOOK_URL;
import static ru.yandex.canvas.service.SandBoxService.VideoTaskExtraFields.URL;

/**
 * Сервис для работы с аудиодорожкой для аудиорекламы.
 */
public class AudioService implements AudioServiceInterface {
    private final AudioFilesRepository audioFilesRepository;
    private final StillageService stillageService;
    private final AudioUploadService fileUploadService;
    private final SandBoxService sandBoxService;
    private final VideoLimitsService videoLimitsService;
    private final String hookSecret;
    private final String canvasVaApiBaseUrl;

    public AudioService(AudioFilesRepository audioFilesRepository, StillageService stillageService,
                        AudioUploadService fileUploadService, SandBoxService sandBoxService,
                        VideoLimitsService videoLimitsService, String hookSecret, String canvasVaApiBaseUrl) {
        this.audioFilesRepository = audioFilesRepository;
        this.stillageService = stillageService;
        this.fileUploadService = fileUploadService;
        this.sandBoxService = sandBoxService;
        this.videoLimitsService = videoLimitsService;
        this.hookSecret = hookSecret;
        this.canvasVaApiBaseUrl = canvasVaApiBaseUrl;
    }

    public AudioSource upload(byte[] content, String filename, Long clientId, VideoCreativeType videoCreativeType,
                              Long presetId) {
        StillageFileInfo info = stillageService.uploadFile(filename, content);
        return upload(info, filename, clientId, videoCreativeType, presetId);
    }

    public AudioSource upload(URI url, String filename, Long clientId, VideoCreativeType videoCreativeType,
                              Long presetId) throws IOException {
        StillageFileInfo info = stillageService.uploadFile(filename, url.toURL());
        return upload(info, filename, clientId, videoCreativeType, presetId);
    }

    public AudioSource upload(StillageFileInfo stillageFileInfo, String filename, Long clientId,
                              VideoCreativeType videoCreativeType, Long presetId) {
        AudioFiles record = makeRecordForUpload(clientId, filename, stillageFileInfo, videoCreativeType);
        return new AudioSource(fileUploadService.upload(record, stillageFileInfo, videoCreativeType, presetId));
    }

    private AudioFiles makeRecordForUpload(Long clientId, String filename, StillageFileInfo info,
                                           VideoCreativeType videoCreativeType) {
        AudioFiles record = new AudioFiles();
        record.setArchive(false);
        record.setClientId(clientId);
        record.setName(filename);
        record.setArchive(false);
        record.setDate(Date.from(Instant.now()));
        record.setUrl(info.getUrl());
        record.setStillageId(info.getId());
        record.setCreativeType(videoCreativeType);
        return record;
    }

    public boolean delete(String fileId, Long clientId) {
        return audioFilesRepository.deleteFile(fileId, clientId);
    }

    public AudioSource makeFromDb(AudioFiles record) {
        return new AudioSource(record);
    }

    public ItemsWithTotal<AudioSource> getUserFiles(Long clientId,
                                                    String nameSubstring,
                                                    Sort.Direction direction,
                                                    VideoCreativeType videoCreativeType,
                                                    Boolean showFeeds,
                                                    int limit,
                                                    int offset) {
        AudioFilesRepository.QueryBuilder queryBuilder = new AudioFilesRepository.QueryBuilder();
        queryBuilder.withClientId(clientId).withArchive(false);

        if (nameSubstring != null) {
            queryBuilder.withNameRegexp(".*" + Pattern.quote(nameSubstring) + ".+", "i");
        }

        // TODO: раскомментировать, когда выедет фронт DIRECT-125964!
        // VideoLimitsInterface videoLimits = videoLimitsService.getLimits(videoCreativeType);
        // queryBuilder.withDurationBetween(videoLimits.getDurationMin(), videoLimits.getAudioDurationMax());

        if (showFeeds != null) {
            if (showFeeds) {
                queryBuilder.withCreativeType(VideoCreativeType.VIDEO_CONSTRUCTOR_FEED);
            } else {
                queryBuilder.withCreativeTypeNe(VideoCreativeType.VIDEO_CONSTRUCTOR_FEED);
            }
        }

        return new ItemsWithTotal<>(
                audioFilesRepository.findByQuery(queryBuilder, direction, limit, offset).stream()
                        .map(this::makeFromDb).collect(Collectors.toList()),
                audioFilesRepository.count(queryBuilder)
        );
    }

    @Override
    public AudioSource lookupAudio(String audioId, Long clientId) {
        if (audioId == null) {
            return null;
        }

        AudioFiles file = audioFilesRepository.findById(audioId, clientId);

        if (file == null) {
            return null;
        }

        return new AudioSource(file);
    }

    public String markFileUsed(AudioSource src, Long clientId) {
        if (src == null || src.getId() == null) {
            return null;
        }

        audioFilesRepository.markFileUsed(src.getId(), clientId);
        return src.getId();
    }

    @Override
    public AudioFiles updateConvertingFile(String fileId, SandBoxService.SandboxFfConversionTaskOutput output) {
        return audioFilesRepository.updateConvertingFile(fileId, output.getOutUrl());
    }

    public Long startConverting(AudioSource audioSource) {
        SandBoxService.SandboxCreateRequest request = makeSandboxRequest(audioSource);
        Long taskId = sandBoxService.createVideoConvertionTask(request);
        sandBoxService.startTask(taskId);
        return taskId;
    }

    private SandBoxService.SandboxCreateRequest makeSandboxRequest(AudioSource audioSource) {
        if (audioSource.getId() == null) {
            throw new IllegalArgumentException(
                    "file must be already inserted before ffConvertTask is called");
        }

        SandBoxService.SandboxCreateRequest request = new SandBoxService.SandboxCreateRequest();

        request.setPriority("SERVICE", "HIGH")
                .addCustomField(URL, audioSource.getStillageUrl())
                .addCustomField(AUDIO_WEB_HOOK_URL, makeHookUrl(audioSource.getId()));

        request.setOwner("CANVAS");
        request.setType("CANVAS_FF_CONVERT");

        return request;
    }

    private String makeHookUrl(String id) {
        return String.format("%s/files/%s/ffhook?secret=%s", canvasVaApiBaseUrl, id, hookSecret);
    }
}
