package ru.yandex.canvas.service.video;

import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.Objects;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.canvas.model.video.VideoFiles;
import ru.yandex.canvas.model.video.files.FileStatus;
import ru.yandex.canvas.model.video.vh.VhVideo;
import ru.yandex.canvas.repository.video.VideoFilesRepository;

public class CmsConversionStatusUpdateService {
    private static final Logger logger = LoggerFactory.getLogger(CmsConversionStatusUpdateService.class);

    // таймаут на получение метрик VMAF от транскодера
    private static final long GET_VMAF_TIMEOUT_MINUTES = 90;

    private final VhService vhClient;
    private final VideoFileUploadServiceInterface fileUploadService;
    private final VideoFilesRepository videoFilesRepository;

    public CmsConversionStatusUpdateService(VhService vhClient, VideoFileUploadServiceInterface fileUploadService,
                                            VideoFilesRepository videoFilesRepository) {
        this.vhClient = vhClient;
        this.fileUploadService = fileUploadService;
        this.videoFilesRepository = videoFilesRepository;
    }

    public VideoFiles updateStatus(VideoFiles record) {
        if (record.getVideoMetaId() == null
                || record.getConvertionTaskId() != null//ждём конвертацию sandbox. Fallback режим. CMS не дёргаем больше
                || record.getStatus() != FileStatus.CONVERTING) {
            return record;
        }

        //если файл конвертируется через CMS и результата нет, то запросить результат
        try {
            VhVideo vhResp = vhClient.getVideo(record.getVideoMetaId());

            if (vhResp.isDone()) {// если завершено успешно, то записать результаты
                List<VideoFiles.VideoFormat> formats = vhClient.getFormatsFromCMSResult(vhResp.getTranscoderInfo(),
                        record.getStillageFileInfo());

                // ожидаем получения VMAF от транскодера, оно подтягивается через какое-то
                // время после завершения конвертации
                if (formats.stream().map(VideoFiles.VideoFormat::getVmafAvg).allMatch(Objects::isNull) &&
                        LocalDateTime.now().isBefore(vhResp.getCreateTime().plus(
                                GET_VMAF_TIMEOUT_MINUTES, ChronoUnit.MINUTES))) {
                    return record;
                }
                var updateBuilder = new VideoFilesRepository.UpdateBuilder()
                        .withFormats(formats)
                        .withStatus(FileStatus.READY)
                        .withPlayerId(vhResp.getPlayerId())
                        .withInputHeight(vhResp.getTranscoderInfo().getInputHeight())
                        .withInputWidth(vhResp.getTranscoderInfo().getInputWidth())
                        .withSignaturesUrl(vhResp.getTranscoderInfo().getSignaturesUrl())
                        .withFirstFrameUrl(vhResp.getTranscoderInfo().getFirstFrameUrl());
                if (vhResp.getThumbnail() != null) {
                    updateBuilder.withThumbnail(fileUploadService.getThumbnailFromUrl(vhResp.getThumbnail()));
                }
                var queryBuilder = new VideoFilesRepository.QueryBuilder()
                        .withStatusNe(FileStatus.READY)
                        .withId(record.getId())
                        .withStockFileId(null);
                videoFilesRepository.update(queryBuilder, updateBuilder);
                return videoFilesRepository.findByIdAndQuery(record.getId(), new VideoFilesRepository.QueryBuilder());
            }

            if (vhResp.isError()) {// в случае ошибки запустить sandbox
                logger.error("CMS fallback to sandbox. Meta_id: {}. File_id: {}. Status: {}", record.getVideoMetaId(),
                        record.getId(), vhResp.getTranscoderStatus());
                Long sandboxTaskId = fileUploadService.startSandboxConverting(record, true);
                var updateBuilder = new VideoFilesRepository.UpdateBuilder()
                        .withConversionId(sandboxTaskId);
                videoFilesRepository.update(record.getId(), updateBuilder);
            }
        } catch (Exception e) {
            logger.error("Error get vh result", e);
        }

        return record;
    }

}
