package ru.yandex.canvas.service.video;

import java.io.IOException;
import java.net.URISyntaxException;
import java.sql.Date;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.collect.Iterables;
import com.google.common.collect.Iterators;
import com.google.common.collect.Streams;
import org.apache.commons.collections4.iterators.IteratorChain;

import ru.yandex.canvas.model.video.Addition;
import ru.yandex.canvas.model.video.CustomVastParams;
import ru.yandex.canvas.model.video.StockAddition;
import ru.yandex.canvas.model.video.files.AudioSource;
import ru.yandex.canvas.model.video.files.Movie;
import ru.yandex.canvas.model.video.files.VideoSource;
import ru.yandex.canvas.repository.video.StockVideoAdditionsRepository;
import ru.yandex.canvas.repository.video.VideoAdditionsRepository;
import ru.yandex.canvas.service.TankerKeySet;
import ru.yandex.canvas.service.video.files.StockMoviesService;
import ru.yandex.canvas.service.video.generation.StockElementFactory;
import ru.yandex.canvas.service.video.presets.PresetTag;
import ru.yandex.canvas.service.video.presets.VideoPreset;

import static com.google.common.collect.Lists.cartesianProduct;
import static org.springframework.context.i18n.LocaleContextHolder.getLocale;
import static org.springframework.context.i18n.LocaleContextHolder.setLocale;
import static ru.yandex.canvas.VideoConstants.DEFAULT_VPAID_PCODE_URL;

public class CreativesGenerationService {

    private StockMoviesService stockMoviesService;
    private VideoPresetsService presetsService;
    private VideoAdditionsService videoAdditionsService;
    private VideoAdditionsRepository videoAdditionsRepository;
    private StockVideoAdditionsRepository stockVideoAdditionsRepository;
    private VideoCreativesService videoCreativesService;

    public CreativesGenerationService(StockMoviesService stockMoviesService,
                                      VideoPresetsService presetsService,
                                      VideoAdditionsService videoAdditionsService,
                                      VideoAdditionsRepository videoAdditionsRepository,
                                      StockVideoAdditionsRepository stockVideoAdditionsRepository,
                                      VideoCreativesService videoCreativesService) {
        this.stockMoviesService = stockMoviesService;
        this.presetsService = presetsService;
        this.videoAdditionsService = videoAdditionsService;
        this.videoAdditionsRepository = videoAdditionsRepository;
        this.stockVideoAdditionsRepository = stockVideoAdditionsRepository;
        this.videoCreativesService = videoCreativesService;
    }

    public List<List<Addition>> generateAdditions(List<GenerateCondition> conditions, Long clientId)
            throws IOException, URISyntaxException {
        Iterator<ConditionsCombination> combinations = getCombinationsIterator(conditions);

        Map<ConditionsCombination, AdditionsComboHolder> holders =
                Streams.stream(combinations).distinct().map(AdditionsComboHolder::new).collect(
                        Collectors.toMap(AdditionsComboHolder::getConditionsCombination,
                                Function.identity())); //LinkedHashMap TODO

        List<StockAddition> stockAdditions = findStockAdditions(
                holders.values().stream().map(AdditionsComboHolder::getConditionsCombination).collect(
                        Collectors.toList()));

        for (StockAddition stockAddition : stockAdditions) {
            holders.get(makeCombinationFromStockAddition(stockAddition)).setStockAddition(stockAddition);
        }

        for (StockAddition stockAddition : createStockAdditions(
                holders.values().stream().filter(e -> e.getStockAddition() == null).map(e -> e.conditionsCombination)
                        .collect(Collectors.toList()))) {
            holders.get(makeCombinationFromStockAddition(stockAddition)).setStockAddition(stockAddition);
        }

        for (AdditionsComboHolder holder : holders.values()) {
            holder.setAddition(buildAdditionFromStock(holder.getStockAddition(), clientId));
        }

        List<Addition> duplicates = videoAdditionsRepository
                .createAdditions(holders.values().stream().map(AdditionsComboHolder::getAddition).collect(
                        Collectors.toList()));

        if (duplicates.size() > 0) {
            loadAdditionsFromDbToHolders(holders, duplicates, clientId);
        }

        return makeAdditionsListFromConditions(conditions, holders);
    }

    void loadAdditionsFromDbToHolders(Map<ConditionsCombination, AdditionsComboHolder> holders,
                                      List<Addition> duplicates, Long clientId) {
        Map<Long, ConditionsCombination> stockAdditionMap = holders.entrySet().stream()
                .collect(Collectors.toMap(e -> e.getValue().getStockAddition().getCreativeId(), Map.Entry::getKey));

        VideoAdditionsRepository.QueryBuilder queryBuilder = new VideoAdditionsRepository.QueryBuilder()
                .withStockCreativeIds(duplicates.stream().map(Addition::getStockCreativeId).collect(Collectors.toList()))
                .withClientId(clientId);

        List<Addition> additions = videoAdditionsRepository.findByQuery(queryBuilder);

        for (Addition addition : additions) {
            ConditionsCombination combo = stockAdditionMap.get(addition.getStockCreativeId());
            holders.get(combo).setAddition(addition);
        }
    }

    private static List<List<Addition>> makeAdditionsListFromConditions(List<GenerateCondition> conditions,
                                                                        Map<ConditionsCombination,
                                                                                AdditionsComboHolder> holders) {
        List<List<Addition>> result = new ArrayList<>();

        Iterator<AdditionsComboHolder> iterator = holders.values().iterator();

        for (GenerateCondition condition : conditions) {
            List<Addition> row = new ArrayList<>();
            for (int i = 0; i < condition.getCount(); i++) {
                row.add(iterator.next().getAddition());
            }

            result.add(row);
        }

        return result;
    }

    private Iterator<ConditionsCombination> getCombinationsIterator(List<GenerateCondition> conditions) {
        return new IteratorChain<>(
                conditions.stream().map(this::generateCombinations).filter(Objects::nonNull)
                        .collect(Collectors.toList())
        );
    }

    //TODO changes his argument
    Addition buildAdditionFromStock(StockAddition stockAddition, Long clientId) {
        Addition copy = stockAddition.makeAddtionFromStock();

        copy.setId(null);
        copy.setClientId(clientId);
        copy.setDate(Date.from(Instant.now()));
        copy.setCreationTime(copy.getDate());
        copy.setCreativeId(videoAdditionsService.getNextAdditionId());

        return copy;
    }

    //надо сначала уникализировать, потом создавать, иначе мы можнм присвоить creativeId
    List<StockAddition> createStockAdditions(List<ConditionsCombination> combinations)
            throws IOException, URISyntaxException {
        if (combinations.isEmpty()) {
            return Collections.emptyList();
        }

        List<StockAddition> additions = new ArrayList<>();

        Locale currentLocale = getLocale();

        try {
            for (ConditionsCombination combination : combinations) {
                StockAddition addition = new StockAddition();

                //Dedicated to all perl-lovers, converts locale format 'ru_RU' to 'ru'
                setLocale(Locale.forLanguageTag(combination.getLocale().replaceAll("_.+$", "")));

                //VideoStream is needed inside conversions
                combination.setMovie(
                        stockMoviesService.getFileByIds(combination.getVideo(), combination.getAudio()));

                addition.setData(StockElementFactory.convert(combination.getPreset(), combination));

                addition.setPresetId(combination.getPreset().getId());
                addition.setLocale(combination.getLocale());
                addition.setAudioId(combination.getAudio());
                addition.setVideoId(combination.getVideo());
                addition.setName(TankerKeySet.VIDEO.key("autogenerated-addition-name"));

                videoAdditionsService.createAdditionWithScreenshot(0L, addition,
                        new CustomVastParams().setVpaidPcodeUrl(DEFAULT_VPAID_PCODE_URL));

                additions.add(addition);
            }
        } finally {
            setLocale(currentLocale);
        }

        videoCreativesService.uploadToRtbHost(additions);

        List<StockAddition> duplicates = stockVideoAdditionsRepository.createStockAdditions(additions);

        if (duplicates.size() > 0) {
            additions = findStockAdditions(combinations);
        }

        return additions;
    }

    List<StockAddition> findStockAdditions(List<ConditionsCombination> combinations) {
        List<StockVideoAdditionsRepository.StockFindQueryBuilder> queries = new ArrayList<>();

        for (ConditionsCombination combination : combinations) {
            StockVideoAdditionsRepository.StockFindQueryBuilder builder =
                    new StockVideoAdditionsRepository.StockFindQueryBuilder();

            builder.withAudioId(combination.getAudio())
                    .withVideoId(combination.getVideo())
                    .withLocale(combination.getLocale())
                    .withPresetId(combination.getPreset().getId());

            queries.add(builder);
        }

        return stockVideoAdditionsRepository.findStockAdditions(queries);
    }

    static class AdditionsComboHolder {
        private StockAddition stockAddition;
        private Addition addition;
        private ConditionsCombination conditionsCombination;

        public AdditionsComboHolder(ConditionsCombination conditionsCombination) {
            this.conditionsCombination = conditionsCombination;
        }

        public StockAddition getStockAddition() {
            return stockAddition;
        }

        public Addition getAddition() {
            return addition;
        }

        public ConditionsCombination getConditionsCombination() {
            return conditionsCombination;
        }

        public AdditionsComboHolder setStockAddition(StockAddition stockAddition) {
            this.stockAddition = stockAddition;
            return this;
        }

        public AdditionsComboHolder setAddition(Addition addition) {
            this.addition = addition;
            return this;
        }
    }

    ConditionsCombination makeCombinationFromStockAddition(
            StockAddition stockAddition) {
        return new ConditionsCombination(
                presetsService.getPreset(stockAddition.getPresetId()), stockAddition.getAudioId(),
                stockAddition.getVideoId(), stockAddition.getLocale());
    }


    Iterator<ConditionsCombination> generateCombinations(
            GenerateCondition condition) {
        List<VideoPreset> presets = presetsService.getPresetsByTag(condition.getPresetTag());

        List<String> audios = stockMoviesService.searchAudio(null).getResult().stream().map(AudioSource::getId)
                .collect(Collectors.toList());

        List<String> knownCategories = null;

        if (condition.getCategories() != null) {
            knownCategories = condition.getCategories().stream()
                    .filter(e -> stockMoviesService.getKnownSubCategories().contains(e))
                    .collect(Collectors.toList());

            if (knownCategories.isEmpty()) {
                knownCategories = null;
            }
        }

        List<String> videos =
                stockMoviesService.searchVideo(knownCategories, null, null)
                        .getResult().stream().map(
                        VideoSource::getId)
                        .collect(Collectors.toList());

        if (presets.isEmpty() || audios.isEmpty() || videos.isEmpty() || condition.getLocales().isEmpty()
                || condition.getCount() == 0) {
            condition.count = 0; //TODO changing incoming arguments is such awful practice.
            return null;
        }

        return generateCombinationsIterator(presets, audios, videos, condition.getLocales(), condition.getCount(),
                condition.doShuffle);
    }

    private static <T> List<T> copyAndShuffle(List<T> source, boolean doShuffle) {

        if (!doShuffle) {
            return source;
        }

        ArrayList<T> copy = new ArrayList<>(source);
        Collections.shuffle(copy);
        return copy;
    }

    Iterator<ConditionsCombination> generateCombinationsIterator(List<VideoPreset> presets, List<String> audios,
                                                                 List<String> videos, List<String> locales, int count,
                                                                 boolean doShuffle) {
        ConditionsCombinationsIterator.Builder builder = ConditionsCombinationsIterator.builder();

        return Iterators.limit(

                builder.withPresets(copyAndShuffle(presets, doShuffle))
                        .withAudios(copyAndShuffle(audios, doShuffle))
                        .withVideos(copyAndShuffle(videos, doShuffle))
                        .withLocales(copyAndShuffle(locales, doShuffle))
                        .build(),

                count);
    }

    public static class GenerateCondition {
        private PresetTag presetTag;
        private List<String> categories;
        private List<String> locales;
        private int count;
        private boolean doShuffle;

        public GenerateCondition(PresetTag presetTag, List<String> categories, List<String> locales, int count,
                                 boolean doShuffle) {
            this.presetTag = presetTag;
            this.categories = categories;
            this.locales = locales;
            this.count = count;
            this.doShuffle = doShuffle;
        }

        public int getCount() {
            return count;
        }

        public PresetTag getPresetTag() {
            return presetTag;
        }

        public List<String> getCategories() {
            return categories;
        }

        public List<String> getLocales() {
            return locales;
        }
    }

    static class ConditionsCombinationsIterator implements Iterator<ConditionsCombination> {
        private Iterator<List<Object>> iterator;

        @Override
        public boolean hasNext() {
            return iterator.hasNext();
        }

        @Override
        public ConditionsCombination next() {
            List<Object> record = iterator.next();
            return new ConditionsCombination((VideoPreset) record.get(0),
                    record.get(1).equals("") ? null : (String) record.get(1),
                    (String) record.get(2),
                    (String) record.get(3));
        }

        @Override
        public void remove() {
            throw new UnsupportedOperationException();
        }

        @Override
        public void forEachRemaining(Consumer<? super ConditionsCombination> action) {
            throw new UnsupportedOperationException();
        }

        public static Builder builder() {
            return new Builder();
        }

        public static class Builder {
            private List<VideoPreset> presets;
            private List<String> audios;
            private List<String> videos;
            private List<String> locales;

            private Builder() {

            }

            public Builder withPresets(List<VideoPreset> presets) {
                this.presets = presets;
                return this;
            }

            public Builder withAudios(List<String> audios) {
                this.audios = audios;
                return this;
            }

            public Builder withVideos(List<String> videos) {
                this.videos = videos;
                return this;
            }

            public Builder withLocales(List<String> locales) {
                this.locales = locales;
                return this;
            }

            public ConditionsCombinationsIterator build() {
                audios = audios.stream().map(e -> (e == null) ? "" : e).collect(Collectors.toList());
                List<List<Object>> combinations = cartesianProduct(presets, audios, videos, locales);

                Iterator<List<Object>> iterator = Iterables.cycle(combinations).iterator();

                return new ConditionsCombinationsIterator(iterator);
            }
        }

        private ConditionsCombinationsIterator(Iterator<List<Object>> iterator) {
            this.iterator = iterator;
        }


    }


    public static class ConditionsCombination {
        private VideoPreset preset;
        private String audio;
        private String video;
        private Movie movie;
        String locale;

        ConditionsCombination(VideoPreset preset, String audio, String video, String locale) {
            this.preset = preset;
            this.audio = audio;
            this.video = video;
            this.locale = locale;
        }

        public Movie getMovie() {
            return movie;
        }

        public ConditionsCombination setMovie(Movie movie) {
            this.movie = movie;
            return this;
        }

        public VideoPreset getPreset() {
            return preset;
        }

        public String getAudio() {
            return audio;
        }

        public String getVideo() {
            return video;
        }

        public String getLocale() {
            return locale;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            ConditionsCombination that = (ConditionsCombination) o;
            return Objects.equals(preset.getId(), that.preset.getId()) &&
                    Objects.equals(audio, that.audio) &&
                    Objects.equals(video, that.video) &&
                    Objects.equals(locale, that.locale);
        }

        @Override
        public int hashCode() {
            return Objects.hash(preset.getId(), audio, video, locale);
        }
    }
}
