package ru.yandex.canvas.service.video;

import java.io.IOException;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.stream.Stream;

import com.google.common.base.Strings;

import ru.yandex.canvas.JSONObjectHelper;
import ru.yandex.canvas.model.video.addition.AdditionData;
import ru.yandex.canvas.model.video.addition.AdditionElement;
import ru.yandex.common.util.collections.Pair;

public class DCParams {
    private static final Set<AdditionElement.ElementType> DC_ELEMENT_TYPES = Set.of(
            AdditionElement.ElementType.TITLE, AdditionElement.ElementType.BODY,
            AdditionElement.ElementType.AGE, AdditionElement.ElementType.DOMAIN
    );
    private PreviewData previewData;
    private AdditionData additionData;
    private boolean skipAbuse;

    public DCParams(PreviewData previewData, AdditionData additionData, boolean skipAbuse) {
        this.previewData = previewData;
        this.skipAbuse = skipAbuse;
        this.additionData = additionData;
    }

    private boolean hasSomethingForModerate() {
        return previewData.getImages() != null || previewData.getSecondTitle() != null
                || previewData.getSitelinks() != null;
    }

    public String toJSON() throws IOException {
        List<AdditionElement> elementList;
        if (additionData != null && additionData.getElements() != null) {
            elementList = additionData.getElements();
        } else {
            elementList = Collections.emptyList();
        }

        return new JSONObjectHelper()
                .add("data_params", new JSONObjectHelper()
                        .add("misc", new JSONObjectHelper()
                                .add("object_id", "")
                                .add("target_url", "")
                                .add("click_url", new JSONObjectHelper()
                                        .addif(() -> !skipAbuse, "abuse", "abuselink.com"))
                        )
                        .add("offer123", new JSONObjectHelper()
                                .add("object_id", "offer123")
                                .add("target_url", previewData.getUrl())
                                .add("image", new JSONObjectHelper())
                                .add("text", new JSONObjectHelper()
                                        .add("lang", previewData.getLang())
                                        .add("titleHyph", "")
                                        .add("bodyHyph", "")
                                        .add("domain", previewData.getDomain())
                                        .add("punyDomain", previewData.getPunyDomain())
                                        .add("geoDistance", "")
                                        .add("geoDistanceRegular", "")
                                        .add("title", previewData.getTitle())
                                        .add("body", previewData.getBody())
                                        .add("age", previewData.getAge())
                                        .add("warning", previewData.getWarning())
                                        .add("dynamic_disclaimer", previewData.getDynamicDisclaimer())
                                        .add("banner_flags", previewData.getBannerFlags())
                                        .add("region", "")
                                        .addStream(jsonAdditionData(elementList), k -> k.getFirst(), v -> v.getSecond())
                                )
                                .add("count", "")
                                .add("click_url", new JSONObjectHelper()
                                        .addIf(()->previewData.getButtonUrl() != null, "action_button", previewData.getButtonUrl())
                                ).addIf(() -> hasSomethingForModerate(), "unmoderated",
                                new JSONObjectHelper()
                                        .addIf(() -> previewData.getImages() != null, "images", previewData.getImages())
                                        .addIf(() -> previewData.getSitelinks() != null, "sitelinks", previewData.getSitelinks())
                                        .addIf(() -> previewData.getSecondTitle() != null, "secondTitle", previewData.getSecondTitle()))
                        )
                ).toJSON();
    }

    private static Stream<Pair<String,String>> jsonAdditionData(List<AdditionElement> elementList) {
        return elementList.stream()
                .filter(e -> DC_ELEMENT_TYPES.contains(e.getType()))
                .filter(e -> e.getOptions().getText() != null && e.getOptions().getText().length() != 0)
                .map(e -> new Pair<>(e.getType().name().toLowerCase(), e.getOptions().getText()))
                .map(p -> {
                    if (p.getFirst().equals("age")) {
                        return new Pair<>(p.getFirst(), agePlus(p.getSecond()));
                    }
                    return p;
                });
    }

    private static String agePlus(String age) {
        if (!Strings.isNullOrEmpty(age) && !age.endsWith("+")) {
            return age + "+";
        }
        return age;
    }
}
