package ru.yandex.canvas.service.video;

import javax.annotation.ParametersAreNonnullByDefault;

/**
 * Битрейты скопированы с DEFAULT_RESOLUTIONS отсюда
 * https://a.yandex-team.ru/arc/trunk/arcadia/sandbox/projects/Strm/StrmVideoConvert/constants/__init__.py?rev=4161760
 * Далеко не все битрейты совпадают в точности, тогда мы берём, что-нибудь ближайшее.
 * Например по указанной выше ссылке есть такая конфигурация:
 * ("169_720p", ("1280", "720", "3200", "128")),
 * Мы используем значения битрейтов 3200 и 128 для всех 720p разрешений - не важно 2:1, 16:9, 4:3
 *
 * <p>Стоит заметить, что bitrate зависит от требуемого fps, данный enum расчитан на стандартный fps=25
 */
@ParametersAreNonnullByDefault
public enum FfmpegResolution {
    // Правила именования:
    // suffix:
    // для стандартных разрешений: {соотношениеСторон}_{количествоПикселейПоВысоте}p
    // для нестандартных:          {соотношениеСторон}_{количествоПикселейПоВысоте}p_{количествоПикселейПоШирине}
    // имя enum: R_{suffix}
    // Сортировка элементов enum: сначала по ratio, потом по width по возрастанию, потом по height по возрастанию
    R_21_320p_576("2:1", "21_320p_576", 576, 320, 700, 64),
    R_21_320p("2:1", "21_320p_640", 640, 320, 800, 64),
    R_21_360p("2:1", "21_360p", 720, 360, 800, 64),
    R_21_720p("2:1", "21_720p", 1440, 720, 3200, 128),
    R_21_900p("2:1", "21_900p", 1800, 900, 5000, 128),

    R_31_288p("3:1", "31_288p", 864, 288, 1200, 64),
    R_31_300p_940("3:1", "31_300p_940", 940, 300, 2000, 64),
    R_31_320p_940("3:1", "31_320p_940", 940, 320, 2000, 64),
    R_31_400p_1216("3:1", "31_400p_1216", 1216, 400, 2000, 64),
    R_31_416p("3:1", "31_416p", 1248, 416, 2000, 64),
    R_31_576p("3:1", "31_576p", 1728, 576, 3200, 128),
    R_31_600p_1920("3:1", "31_600p_1920", 1920, 600, 4200, 128),

    R_103_576p("10:3", "103_576p", 1920, 576, 3200, 128),

    R_43_1152p("4:3", "43_1152p", 1536, 1152, 3200, 128),

    // для 1080 * 1920 стоит videoBitrate в 2500, хотя всё указывает на то, что должно быть в 2 раза больше - 5000
    // но так было сделано до этого и пока что никто не жаловался, поэтому оставляю 2500
    R_916_1920p("9:16", "916_1920p", 1080, 1920, 2500, 128),

    R_2318_576p("23:18", "2318_576p", 736, 576, 2000, 64),

    R_7855_880p("78:55", "7855_880p", 1248, 880, 5000, 128),

    R_9425_400p("94:25", "9425_400p", 1504, 400, 2000, 64),

    R_169_234p("16:9", "169_234p", 416, 234, 350, 64),
    R_169_360p("16:9", "169_360p", 640, 360, 700, 64),
    R_169_486p("16:9", "169_486p", 864, 486, 1200, 64),

    /*
       ("169_240p", ("426", "240", "500", "64")),
    ("169_360p", ("640", "360", "1000", "64")),
    ("169_480p", ("854", "480", "2500", "64")),
    ("169_720p", ("1280", "720", "5000", "128")),
    ("169_1080p", ("1920", "1080", "8000", "128")),
     */;


    private final String ratio;
    // суффикс добавляется к имени файла при конвертации
    // поэтому суффикс должен быть уникальным - иначе два разрешения попадут в один файл
    private final String suffix;
    private final int width;
    private final int height;
    private final int videoBitrate;
    private final int audioBitrate;
    private final String stringValue;

    FfmpegResolution(String ratio, String suffix, int width, int height, int videoBitrate, int audioBitrate) {
        this.ratio = ratio;
        this.suffix = suffix;
        this.width = width;
        this.height = height;
        this.videoBitrate = videoBitrate;
        this.audioBitrate = audioBitrate;
        this.stringValue = String.format("[\"%s\", [\"%d\", \"%d\", \"%d\", \"%d\"]]",
                suffix, width, height, videoBitrate, audioBitrate);
    }

    public String getRatio() {
        return ratio;
    }

    public String getSuffix() {
        return suffix;
    }

    public int getWidth() {
        return width;
    }

    public int getHeight() {
        return height;
    }

    public int getVideoBitrate() {
        return videoBitrate;
    }

    public int getAudioBitrate() {
        return audioBitrate;
    }

    @Override
    public String toString() {
        return stringValue;
    }

}
