package ru.yandex.canvas.service.video;

import java.util.Arrays;
import java.util.Collection;
import java.util.EnumSet;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

@ParametersAreNonnullByDefault
public class FfmpegResolutions {

    private final Set<FfmpegResolution> values;

    private FfmpegResolutions(Set<FfmpegResolution> values) {
        this.values = values;
    }

    public Set<FfmpegResolution> getAllResolutions() {
        return values;
    }

    public static FfmpegResolutions of(FfmpegResolution... resolutions) {
        return new FfmpegResolutions(resolutions.length == 0 ?
                EnumSet.noneOf(FfmpegResolution.class) :
                EnumSet.copyOf(Arrays.asList(resolutions)));
    }

    @Nullable
    public String getConvertibleResolutionsAsString(String ratio, int originWidth, int originHeight) {
        Collection<FfmpegResolution> resolutions = getConvertibleResolutions(ratio, originWidth, originHeight);
        if (resolutions.isEmpty()) {
            return null;
        } else {
            return resolutions.stream()
                    .map(FfmpegResolution::toString)
                    .collect(Collectors.joining(",", "[", "]"));
        }
    }

    /**
     * Для данных ratio, originWidth и originHeight возвращает список форматов в которые можно сконвертировать.
     * Это форматы у которых:
     * - ratio такой же
     * - либо width либо height (либо и то и другое) меньше либо равны соответсвенно originWidth и originHeight
     *
     * <p>Идея в том, что мы только сжимаем, а не растягиваем видео. При необходимости видео добивается чёрными
     * полосами либо по вертикали либо по горизонтали. Мы всегда это можем сделать, если формат, в который мы сжимаем
     * хотя бы по одной стороне не больше исходного.
     *
     * <p>Пример1: нам загружают 1800x600 - конвертируем в 1920x600, добавляя слева и справа полосы.
     * <p>Пример2: нам загружают 720x360  - конвертируем в 640x360, масштабируя и добавляя сверху и снизу чёрные полосы.
     * <p>Пример3: нам загружают 1797x599 - не конвертируем в 1920x600, с двух сторон чёрными полосами не добиваем,
     * растягивать нельзя.
     */
    public Collection<FfmpegResolution> getConvertibleResolutions(String ratio, int originWidth, int originHeight) {
        return values.stream()
                .filter(v -> v.getRatio().equals(ratio))
                .filter(v -> v.getWidth() <= originWidth || v.getHeight() <= originHeight)
                .collect(Collectors.toList());
    }

    @Nullable
    public String getAllResolutionsAsString(String ratio) {
        Collection<FfmpegResolution> resolutions = getAllResolutions(ratio);
        if (resolutions.isEmpty()) {
            return null;
        } else {
            return resolutions.stream()
                    .map(FfmpegResolution::toString)
                    .collect(Collectors.joining(",", "[", "]"));
        }
    }

    public Collection<FfmpegResolution> getAllResolutions(String ratio) {
        return values.stream()
                .filter(v -> v.getRatio().equals(ratio))
                .collect(Collectors.toList());
    }

    public Optional<String> getResolutionRatio(int width, int height) {
        return values.stream()
                .filter(v -> v.getWidth() == width && v.getHeight() == height)
                .map(FfmpegResolution::getRatio)
                .findAny();
    }

}
