package ru.yandex.canvas.service.video;

import java.io.IOException;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.data.domain.Sort;

import ru.yandex.canvas.exceptions.BadRequestException;
import ru.yandex.canvas.exceptions.NotFoundException;
import ru.yandex.canvas.model.CreativeData;
import ru.yandex.canvas.model.CreativeDocument;
import ru.yandex.canvas.model.MediaSetItem;
import ru.yandex.canvas.model.MediaSetSubItem;
import ru.yandex.canvas.model.elements.Element;
import ru.yandex.canvas.model.elements.ElementType;
import ru.yandex.canvas.model.elements.Image;
import ru.yandex.canvas.model.elements.SecondImage;
import ru.yandex.canvas.model.elements.Video;
import ru.yandex.canvas.model.presets.Preset;
import ru.yandex.canvas.model.presets.PresetItem;
import ru.yandex.canvas.model.presets.PresetSelectionCriteria;
import ru.yandex.canvas.model.stillage.StillageFileInfo;
import ru.yandex.canvas.model.video.VideoFiles;
import ru.yandex.canvas.model.video.files.FileStatus;
import ru.yandex.canvas.model.video.files.FileType;
import ru.yandex.canvas.model.video.files.InBannerVideo;
import ru.yandex.canvas.model.video.files.Movie;
import ru.yandex.canvas.repository.ItemsWithTotal;
import ru.yandex.canvas.repository.video.QueryBuilderBase;
import ru.yandex.canvas.repository.video.VideoFilesRepository;
import ru.yandex.canvas.service.DateTimeService;
import ru.yandex.canvas.service.PresetsService;
import ru.yandex.canvas.service.StillageService;
import ru.yandex.canvas.service.video.files.StockMoviesService;

@ParametersAreNonnullByDefault
public class InBannerVideoFilesService implements VideoFilesServiceInterface<InBannerVideo, String> {

    private final VideoFileUploadServiceInterface fileUploadService;
    private final VideoFilesRepository videoFilesRepository;

    private final StillageService stillageService;
    private final DateTimeService dateTimeService;
    private final PresetsService presetsService;
    private final StockMoviesService stockMoviesService;
    private final CmsConversionStatusUpdateService cmsConversionStatusUpdateService;

    public static final String DEFAULT_PICTURE_URL = "https://storage.mds.yandex.net/get-bstor/42187/349acc21-2de5-4e3a-8111-9888169b92c0.png";

    public InBannerVideoFilesService(VideoFileUploadServiceInterface fileUploadService,
                                     VideoFilesRepository videoFilesRepository, StillageService stillageService,
                                     DateTimeService dateTimeService, PresetsService presetsService,
                                     StockMoviesService stockMoviesService,
                                     CmsConversionStatusUpdateService cmsConversionStatusUpdateService) {
        this.fileUploadService = fileUploadService;
        this.videoFilesRepository = videoFilesRepository;
        this.stillageService = stillageService;
        this.dateTimeService = dateTimeService;
        this.presetsService = presetsService;
        this.stockMoviesService = stockMoviesService;
        this.cmsConversionStatusUpdateService = cmsConversionStatusUpdateService;
    }

    public String lookupPicture(CreativeDocument creativeDocument) {
        return lookupPicture(getImageMediaItem(creativeDocument));
    }

    public String lookupSecondPicture(@Nullable CreativeDocument creativeDocument) {
        return lookupPicture(getSecondImageMediaItem(creativeDocument));
    }

    public String lookupPicture(@Nullable MediaSetSubItem mediaSetSubItem) {
        if (mediaSetSubItem == null) {
            return null;
        }

        if (mediaSetSubItem.isDefaultValue()) {
            return DEFAULT_PICTURE_URL;
        }

        return mediaSetSubItem.getUrl();
    }

    @Nullable
    public InBannerVideo lookupMovie(String videoId, VideoFilesRepository.QueryBuilder queryBuilder) {
        var fileId = getFileByIdForCreativeType(videoId, queryBuilder);

        if (fileId == null) {
            return null;
        }

        return makeFromDb(fileId);
    }

    @Nullable
    public MediaSetSubItem getUniqSubItem(@Nullable CreativeData creativeData, @Nullable Element element) {
        if (element == null) {
            return null;
        }

        String mediaSetName = element.getMediaSet();

        if (creativeData == null
                || creativeData.getMediaSets() == null
                || creativeData.getMediaSets().get(mediaSetName) == null
                || creativeData.getMediaSets().get(mediaSetName).getItems().size() != 1) {
            return null;
        }

        MediaSetItem items = creativeData.getMediaSets().get(mediaSetName).getItems().get(0);

        if (items.getItems().size() != 1) {
            throw new RuntimeException("Wrong amount of subItems");

        }

        return items.getItems().get(0);
    }

    @Nullable
    private MediaSetSubItem getUniqSubItem(CreativeDocument creativeDocument, Element element) {
        return getUniqSubItem(creativeDocument.getData(), element);
    }

    public Video getVideoElement(@Nullable CreativeDocument creativeDocument) {
        return (Video) getElementByType(creativeDocument, ElementType.VIDEO);
    }

    public Image getImageElement(@Nullable CreativeDocument creativeDocument) {
        return (Image) getElementByType(creativeDocument, ElementType.IMAGE);
    }

    @Nullable
    public SecondImage getSecondImageElement(@Nullable CreativeDocument creativeDocument) {
        return (SecondImage) getElementByType(creativeDocument, ElementType.SECOND_IMAGE);
    }

    private Element getElementByType(@Nullable CreativeDocument creativeDocument, String type) {
        if (creativeDocument.getPresetId() == null) {
            throw new BadRequestException("No presetId");
        }

        Preset preset = presetsService.getById(creativeDocument.getPresetId(),
                PresetSelectionCriteria.builder().withCpmCommon(true).build(), null).orElseThrow(NotFoundException::new);

        PresetItem item =
                preset.getItems().stream()
                        .filter(e -> e.getId() == creativeDocument.getItemId())
                        .findFirst()
                        .orElseThrow(NotFoundException::new);

        return item.getElements().stream().filter(e -> e.getType().equals(type)).findFirst().orElse(null);
    }

    public MediaSetSubItem getVideoMediaItem(CreativeDocument creativeDocument) {
        Video video = getVideoElement(creativeDocument);
        return getUniqSubItem(creativeDocument, video);
    }

    public MediaSetSubItem getImageMediaItem(CreativeDocument creativeDocument) {
        Image image = getImageElement(creativeDocument);
        return getUniqSubItem(creativeDocument, image);
    }

    @Nullable
    public MediaSetSubItem getSecondImageMediaItem(@Nullable CreativeDocument creativeDocument) {
        SecondImage image = getSecondImageElement(creativeDocument);
        return getUniqSubItem(creativeDocument, image);
    }

    private String getVideoFileId(CreativeDocument creativeDocument) {
        return getVideoMediaItem(creativeDocument).getFileId();
    }

    public InBannerVideo lookupMovie(CreativeDocument creativeDocument,
                                     VideoFilesRepository.QueryBuilder queryBuilder) {
        MediaSetSubItem mediaSetSubItem = getVideoMediaItem(creativeDocument);

        if (mediaSetSubItem == null) {
            return null;
        }

        if (mediaSetSubItem.isDefaultValue()) {
            Movie movie = stockMoviesService.getFileByIds("new_0_0-077.mov", "3891311");
            return new InBannerVideo(movie);
        }

        var record = getFileByIdForCreativeType(getVideoFileId(creativeDocument), queryBuilder);

        if (record == null) {
            return null;
        }

        return makeFromDb(record);
    }

    public VideoFiles getFileByIdForCreativeType(String id, VideoFilesRepository.QueryBuilder queryBuilder) {

        queryBuilder.withType(FileType.IN_BANNER)
                .withStatus(FileStatus.READY)
                .and(buildMovieSearchQuery())
                .withCreativeType(VideoCreativeType.IN_BANNER);

        return videoFilesRepository.findByIdAndQuery(id, queryBuilder);
    }

    @Override
    public String markFileUsed(InBannerVideo file, Long clientId) {
        return null;
    }

    @Override
    public InBannerVideo upload(byte[] content, String filename, Long clientId, VideoCreativeType videoCreativeType,
                                Long presetId) throws IOException {
        StillageFileInfo info = stillageService.uploadFile(filename, content);
        return upload(info, filename, clientId, videoCreativeType, presetId);
    }

    @Override
    public InBannerVideo upload(URI url, String filename, Long clientId, VideoCreativeType videoCreativeType,
                                Long presetId) throws IOException {
        StillageFileInfo info = stillageService.uploadFile(filename, url.toURL());
        return upload(info, filename, clientId, videoCreativeType, presetId);
    }

    @Override
    public InBannerVideo upload(StillageFileInfo info, String filename, Long clientId,
                                VideoCreativeType videoCreativeType, Long presetId) {
        VideoFiles record = makeVideoFilesRecordForUpload(clientId, filename, videoCreativeType);
        //presetId =null потому что это VideoPreset id, а у нас на самом деле не видео, а ин-баннер.
        // А от номера пресета меняется валидация, и начинают срабатывать проверки предназначенные для видео креативов
        return makeFromDb(fileUploadService.upload(record, info, null));
    }

    @Override
    public StillageFileInfo processFileInfo(StillageFileInfo info, String filename,
                                            VideoCreativeType videoCreativeType) {
        return info;
    }

    private VideoFiles makeVideoFilesRecordForUpload(Long clientId, String filename,
                                                     VideoCreativeType videoCreativeType) {
        VideoFiles record = new VideoFiles();
        record.setStatus(FileStatus.NEW);
        record.setClientId(clientId);
        record.setName(filename);
        record.setType(FileType.IN_BANNER);
        record.setArchive(false);
        record.setDate(dateTimeService.getCurrentDate());
        record.setCreativeType(videoCreativeType);

        return record;
    }

    @Override
    public boolean delete(InBannerVideo file, Long clientId) {
        boolean deleted = videoFilesRepository.deleteFile(file.getId(), clientId, FileType.IN_BANNER);

        if (!deleted) {
            fileUploadService.stopSandboxConverting(file.getId());
        }

        return deleted;
    }

    @Override
    public InBannerVideo makeFromDb(VideoFiles record) {
        return new InBannerVideo(cmsConversionStatusUpdateService.updateStatus(record));
    }

    @Override
    public ItemsWithTotal<InBannerVideo> getUserFiles(Long clientId, String nameSubstring, Sort.Direction direction,
                                                      int limit, int offset, VideoCreativeType videoCreativeType,
                                                      Long presetId, Boolean showFeeds) {

        VideoFilesRepository.QueryBuilder queryBuilder = new VideoFilesRepository.QueryBuilder();
        queryBuilder
                .withType(FileType.IN_BANNER)
                .withClientId(clientId)
                .withStatus(FileStatus.READY)
                .and(buildMovieSearchQuery())
                .withArchive(false);

        if (nameSubstring != null) {
            queryBuilder.withNameRegexp(".*" + Pattern.quote(nameSubstring) + ".+", "i");
        }

        return new ItemsWithTotal<>(
                runQuery(queryBuilder, direction, limit, offset).stream().map(this::makeFromDb)
                        .collect(Collectors.toList()),
                videoFilesRepository.count(queryBuilder)
        );
    }

    List<VideoFiles> runQuery(VideoFilesRepository.QueryBuilder queryBuilder, Sort.Direction direction, int limit,
                              int offset) {
        return videoFilesRepository.findByQuery(queryBuilder, direction, limit, offset);
    }

    protected List<QueryBuilderBase<?>> buildMovieSearchQuery() {
        List<QueryBuilderBase<?>> queryBuilders = new ArrayList<>();

        var ratioQueryFilter = new VideoFilesRepository.QueryBuilder()
                .withRatios(List.of("16:9"));

        //у старого видео нет ratio. Они подходят для широких пресетов
        ratioQueryFilter = new VideoFilesRepository.QueryBuilder().or(
                ratioQueryFilter,
                new VideoFilesRepository.QueryBuilder().withNullRatio()
        );

        queryBuilders.add(ratioQueryFilter);
        return queryBuilders;
    }

}
