package ru.yandex.canvas.service.video;

import java.io.IOException;
import java.net.URI;
import java.sql.Date;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Set;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.domain.Sort;

import ru.yandex.canvas.model.stillage.StillageFileInfo;
import ru.yandex.canvas.model.video.CleanupOldConversionsResult;
import ru.yandex.canvas.model.video.VideoFiles;
import ru.yandex.canvas.model.video.files.FileStatus;
import ru.yandex.canvas.model.video.files.FileType;
import ru.yandex.canvas.model.video.files.MediaDataSource;
import ru.yandex.canvas.model.video.files.Movie;
import ru.yandex.canvas.model.video.files.MovieAndVideoSourceFactory;
import ru.yandex.canvas.repository.ItemsWithTotal;
import ru.yandex.canvas.repository.video.QueryBuilderBase;
import ru.yandex.canvas.repository.video.VideoFilesRepository;
import ru.yandex.canvas.service.DateTimeService;
import ru.yandex.canvas.service.DirectService;
import ru.yandex.canvas.service.SandBoxService;
import ru.yandex.canvas.service.StillageService;
import ru.yandex.canvas.service.VideoLimitsInterface;
import ru.yandex.canvas.service.video.files.StockMoviesService;
import ru.yandex.canvas.service.video.presets.VideoPreset;

import static ru.yandex.canvas.VideoConstants.CANVAS_VIDEO_ANY_SIZE_ALLOWED_FEATURE;
import static ru.yandex.canvas.model.video.files.FileType.IN_BANNER;
import static ru.yandex.canvas.model.video.files.FileType.VIDEO;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class MovieService implements MovieServiceInterface {
    private static final Logger logger = LoggerFactory.getLogger(MovieService.class);

    private final VideoFilesRepository videoFilesRepository;
    private final StockMoviesService stockMoviesService;
    private final VideoFileUploadServiceInterface fileUploadService;
    private final StillageService stillageService;
    private final VideoLimitsService videoLimitsService;
    private final DateTimeService dateTimeService;
    private final DirectService directService;
    private final VideoPresetsService videoPresetsService;
    private final CmsConversionStatusUpdateService cmsConversionStatusUpdateService;
    private final MovieAndVideoSourceFactory movieAndVideoSourceFactory;
    private final VideoGeometryService videoGeometryService;


    public MovieService(VideoFilesRepository videoFilesRepository,
                        StockMoviesService stockMoviesService, VideoFileUploadServiceInterface fileUploadService,
                        StillageService stillageService, VideoLimitsService videoLimitsService,
                        DateTimeService dateTimeService, DirectService directService,
                        VideoPresetsService videoPresetsService,
                        CmsConversionStatusUpdateService cmsConversionStatusUpdateService,
                        MovieAndVideoSourceFactory movieAndVideoSourceFactory,
                        VideoGeometryService videoGeometryService) {
        this.videoFilesRepository = videoFilesRepository;
        this.stockMoviesService = stockMoviesService;
        this.fileUploadService = fileUploadService;
        this.stillageService = stillageService;
        this.videoLimitsService = videoLimitsService;
        this.dateTimeService = dateTimeService;
        this.directService = directService;
        this.videoPresetsService = videoPresetsService;
        this.cmsConversionStatusUpdateService = cmsConversionStatusUpdateService;
        this.movieAndVideoSourceFactory = movieAndVideoSourceFactory;
        this.videoGeometryService = videoGeometryService;
    }

    private static String getStockId(String fileId, VideoFiles file) {
        if (file == null) {
            return fileId;
        } else {
            return file.getStockFileId();
        }
    }

    @Override
    public Movie lookupMovie(String videoId, String audioId, Long clientId, Long presetId) {
        String videoStockId;
        String audioStockId;
        boolean isAudioStock;
        boolean isVideoStock;

        logger.info("Looking for movie with videoId: {} undef client: {}", videoId, clientId);
        VideoFiles audioFile;
        VideoCreativeType creativeType = videoPresetsService.fromPresetId(presetId);

        if (audioId == null) { //We have special stock audio with id == null (silence) (bloody hell..)
            isAudioStock = true;
            audioStockId = null;
            audioFile = null;
        } else {
            audioFile = isAudioStock(audioId) ? null : getFileByIdForCreativeType(audioId, FileType.AUDIO, clientId,
                    creativeType, videoPresetsService.getPresetSafe(presetId));
            audioStockId = getStockId(audioId, audioFile);
            isAudioStock = audioStockId != null;
        }

        VideoFiles videoFile = isVideoStock(videoId) ? null : getFileByIdForCreativeType(videoId, VIDEO,
                clientId,
                creativeType, videoPresetsService.getPresetSafe(presetId));
        videoStockId = getStockId(videoId, videoFile);

        isVideoStock = videoStockId != null;

        if (!isVideoStock && audioId == null) {
            isAudioStock = false;
        }

        if (!isAudioStock && !isVideoStock) {
            if (videoFile == null) {
                return null;
            }
            return movieAndVideoSourceFactory.movieFromVideoFile(videoFile);
        } else if (isAudioStock != isVideoStock) {
            throw new IllegalArgumentException("Both of files must be stock or non stock simultaneously");
        } else {
            Movie stockMovie = stockMoviesService.getFileByIds(videoStockId, audioStockId);

            //If it was a ref, we are supposed to build movie from it.
            if (videoFile != null || audioFile != null) {
                return movieAndVideoSourceFactory.movieFromStockMovieAndDbRefs(videoFile, audioFile, stockMovie);
            }

            return stockMovie;
        }
    }

    private boolean isAudioStock(String id) {
        return stockMoviesService.getAudio(id) != null;
    }

    private boolean isVideoStock(String id) {
        return stockMoviesService.getVideo(id) != null;
    }

    public VideoFiles getFileById(String id, FileType type, Long clientId) {
        VideoFilesRepository.QueryBuilder queryBuilder = new VideoFilesRepository.QueryBuilder();
        queryBuilder.withType(type)
                .withClientId(clientId);

        return videoFilesRepository.findByIdAndQuery(id, queryBuilder);
    }

    public VideoFiles getFileByIdInternal(String id) {
        return videoFilesRepository.findById(id);
    }

    public VideoFiles getFileByIdForCreativeType(String id, FileType type, Long clientId,
                                                 VideoCreativeType videoCreativeType, VideoPreset preset) {
        VideoFilesRepository.QueryBuilder queryBuilder = new VideoFilesRepository.QueryBuilder();
        queryBuilder.withType(type)
                .withClientId(clientId)
                .and(buildMovieSearchQuery(videoCreativeType, directService.getFeatures(clientId, null),
                        preset, null, id, type));

        return videoFilesRepository.findByIdAndQuery(id, queryBuilder);
    }

    public static class MarkedMovieIds {
        private String videoMarkId = null;
        private String audioMarkId = null;

        public String getVideoMarkId() {
            return videoMarkId;
        }

        public void setVideoMarkId(String videoMarkId) {
            this.videoMarkId = videoMarkId;
        }

        public String getAudioMarkId() {
            return audioMarkId;
        }

        public void setAudioMarkId(String audioMarkId) {
            this.audioMarkId = audioMarkId;
        }
    }

    @Override
    public MarkedMovieIds markFileUsed(Movie movie, Long clientId) {
        MarkedMovieIds ids = new MarkedMovieIds();

        for (MediaDataSource file : Arrays
                .asList(movie.getVideoSource(), movie.getAudioSource())) {

            /*
            869     # Do not mark file_id of None as used
            870     # We have `Silence` audio with id=None
            871     # FIXME: remove kostyl when proper templating is implemented

            Couple of years passed.

            - What a hell is proper templating?
            */

            if (file == null || file.getId() == null) {
                continue;
            }

            String newId = videoFilesRepository.markFileUsed(file, clientId);

            if (file.getSourceType() == VIDEO) {
                ids.setVideoMarkId(newId);
            } else {
                ids.setAudioMarkId(newId);
            }
        }

        return ids;
    }

    @Override
    public Movie upload(byte[] content, String filename, Long clientId,
                        VideoCreativeType videoCreativeType, Long presetId) {
        StillageFileInfo info = stillageService.uploadFile(filename, content);
        return upload(info, filename, clientId, videoCreativeType, presetId);
    }

    @Override
    public Movie upload(URI url, String filename, Long clientId,
                        VideoCreativeType videoCreativeType, Long presetId) throws IOException {
        StillageFileInfo info = stillageService.uploadFile(filename, url.toURL());
        return upload(info, filename, clientId, videoCreativeType, presetId);
    }

    public Movie upload(StillageFileInfo info, String filename, Long clientId,
                        VideoCreativeType videoCreativeType, Long presetId) {
        info = processFileInfo(info, filename, videoCreativeType);

        VideoFiles record = makeVideoFilesRecordForUpload(clientId, filename, videoCreativeType);

        return movieAndVideoSourceFactory.movieFromVideoFile(fileUploadService.upload(record, info, presetId));
    }

    public StillageFileInfo processFileInfo(StillageFileInfo info, String filename,
                                            VideoCreativeType videoCreativeType) {
        // для видеоконструктора проверяем наличие аудио, если есть, то дергаем специальную ручку для удаления аудио
        // и сохраняем уже новое полученное видео
        if (VideoCreativeType.isVideoConstructor(videoCreativeType)) {
            VideoMetaData videoMetaData = new ObjectMapper().convertValue(info.getMetadataInfo(), VideoMetaData.class);
            if (videoMetaData.getAudioStreams() != null && !videoMetaData.getAudioStreams().isEmpty()) {
                return stillageService.uploadVideoWithoutAudioUrlInternal(filename, info.getUrl());
            }
        }
        return info;
    }

    @Override
    public boolean delete(Movie file, Long clientId) {
        boolean removed = videoFilesRepository.deleteFile(file.getVideoSource().getId(), clientId, VIDEO);

        if (!removed) {
            return false;
        }

        fileUploadService.stopSandboxConverting(file.getVideoSource().getId());
        return true;
    }

    private VideoFiles makeVideoFilesRecordForUpload(Long clientId, String filename,
                                                     VideoCreativeType videoCreativeType) {
        VideoFiles record = new VideoFiles();
        record.setStatus(FileStatus.NEW);
        record.setClientId(clientId);
        record.setName(filename);
        record.setType(VIDEO);
        //для html5 файл сразу заархивирован чтобы не появлялся в видеокреативах в ранее загруженных
        record.setArchive(videoCreativeType == VideoCreativeType.HTML5);
        record.setDate(dateTimeService.getCurrentDate());
        record.setCreativeType(videoCreativeType);

        return record;
    }

    @Override
    public void updateConvertingFile(String fileId, SandBoxService.SandboxConversionTaskOutput output)
            throws IOException {
        fileUploadService.updateConvertingFile(fileId, output);
    }

    @Override
    public CleanupOldConversionsResult cleanupOldConversions(long ageSeconds) {
        return fileUploadService.cleanupOldConversions(
                Date.from(dateTimeService.getCurrentInstant().minus(24, ChronoUnit.DAYS)),
                Date.from(dateTimeService.getCurrentInstant().minusSeconds(ageSeconds))
        );
    }

    @Override
    public Movie makeFromDb(VideoFiles record) {
        Movie movie;

        if (record.getStockFileId() != null) {
            movie = stockMoviesService.getFileByIds(record.getStockFileId(), null);

            if (movie == null) {
                return null;
            }

            movie = movieAndVideoSourceFactory.movieFromStockMovieAndDbRefs(record, null, movie);

        } else {
            record = cmsConversionStatusUpdateService.updateStatus(record);
            movie = movieAndVideoSourceFactory.movieFromVideoFile(record);
        }

        return movie;
    }

    protected List<QueryBuilderBase<?>> buildMovieSearchQuery(VideoCreativeType videoCreativeType,
                                                              Set<String> features,
                                                              VideoPreset preset,
                                                              @Nullable Boolean showFeeds,
                                                              String id,/*признак что фильтруем для конкретного файла*/
                                                              FileType fileType) {
        //Нужно знать фичи клиента и пресет
        boolean anyRatios = features.contains(CANVAS_VIDEO_ANY_SIZE_ALLOWED_FEATURE);
        List<QueryBuilderBase<?>> queryBuilders = new ArrayList<>();

        VideoLimitsInterface videoLimits = videoLimitsService.getLimits(videoCreativeType, features,
                ifNotNull(preset, VideoPreset::getId));
        queryBuilders.add(
                new VideoFilesRepository.QueryBuilder().or(
                        new VideoFilesRepository.QueryBuilder().withDuration(null),
                        new VideoFilesRepository.QueryBuilder()
                                .withDurationBetween(videoLimits.getDurationMin(), videoLimits.getDurationCaptureStop())
                )
        );

        if ((fileType == VIDEO || fileType == IN_BANNER) && !anyRatios) {
            if (preset != null && videoGeometryService.hasAllowedRatiosInterval(preset.getDescription().getGeometry(),
                    features, videoCreativeType)) {

                var geometry = preset.getDescription().getGeometry();
                var ratios = videoGeometryService.getRatiosByPreset(geometry, features, videoCreativeType);
                var ratioQueryFilter = new VideoFilesRepository.QueryBuilder()
                        .withRatiosBetween(ratios.getFrom(), ratios.getTo());

                if (geometry == Geometry.UNIVERSAL || geometry == Geometry.WIDE) {
                    //у старого видео нет ratio. Они подходят для широких пресетов
                    ratioQueryFilter = new VideoFilesRepository.QueryBuilder().or(
                            ratioQueryFilter,
                            new VideoFilesRepository.QueryBuilder().withNullRatio()
                    );
                }

                queryBuilders.add(ratioQueryFilter);
            } else {
                queryBuilders.add(new VideoFilesRepository.QueryBuilder()
                        .withRatios(
                                mapList(videoGeometryService.getAllowedRatios(videoCreativeType,
                                        ifNotNull(preset, VideoPreset::getId)), e -> ifNotNull(e, Ratio::toString))));
            }

            if (videoCreativeType == VideoCreativeType.CPM_INDOOR) {
                queryBuilders.add(new VideoFilesRepository.QueryBuilder().withHeightGreaterThan(
                        videoLimits.getVideoHeightMin().longValue()));
            }
        }

        if (id == null &&
                (videoCreativeType == VideoCreativeType.CPM
                || videoCreativeType == VideoCreativeType.NON_SKIPPABLE_CPM
                || videoCreativeType == VideoCreativeType.PRICE_CAMPS_CPM
                || videoCreativeType == VideoCreativeType.CPM_YNDX_FRONTPAGE
                || videoCreativeType == VideoCreativeType.PRICE_CAMPS_NON_SKIPPABLE_CPM)
        ) {// для охватных нельзя выбирать стоковое видео. Откуда пошло требование нет информации
            queryBuilders.add(new VideoFilesRepository.QueryBuilder().withStockFileId(null));
        }
        if (showFeeds != null) {
            if (showFeeds) {
                queryBuilders.add(new VideoFilesRepository.QueryBuilder()
                        .withCreativeType(VideoCreativeType.VIDEO_CONSTRUCTOR_FEED));
            } else {
                queryBuilders.add(new VideoFilesRepository.QueryBuilder()
                        .withCreativeTypeNe(VideoCreativeType.VIDEO_CONSTRUCTOR_FEED));
            }
        }
        return queryBuilders;
    }

    @Override
    public ItemsWithTotal<Movie> getUserFiles(Long clientId, String nameSubstring,
                                              Sort.Direction direction, int limit,
                                              int offset, VideoCreativeType videoCreativeType, Long presetId,
                                              @Nullable Boolean showFeeds) {
        VideoFilesRepository.QueryBuilder queryBuilder = new VideoFilesRepository.QueryBuilder();
        queryBuilder
                .withType(VIDEO)
                .withClientId(clientId)
                .withArchive(false);

        if (nameSubstring != null) {
            queryBuilder.withNameRegexp(".*" + Pattern.quote(nameSubstring) + ".+", "i");
        }

        List<QueryBuilderBase<?>> additionalCriteries = buildMovieSearchQuery(videoCreativeType,
                directService.getFeatures(clientId, null), videoPresetsService.getPresetSafe(presetId), showFeeds,
                null, VIDEO);

        if (!VideoCreativeType.isVideoConstructor(videoCreativeType)) {
            additionalCriteries.add(
                    new VideoFilesRepository.QueryBuilder().or(
                            new VideoFilesRepository.QueryBuilder().withStatus(FileStatus.READY),
                            new VideoFilesRepository.QueryBuilder().withStockFile()
                    )
            );
        }

        queryBuilder.and(additionalCriteries);

        return new ItemsWithTotal<>(
                runQuery(queryBuilder, direction, limit, offset).stream().map(this::makeFromDb)
                        .collect(Collectors.toList()),
                videoFilesRepository.count(queryBuilder));
    }

    List<VideoFiles> runQuery(VideoFilesRepository.QueryBuilder queryBuilder, Sort.Direction direction, int limit,
                              int offset) {
        return videoFilesRepository.findByQuery(queryBuilder, direction, limit, offset);
    }

    /**
     * Ищет файлы в статусе "идёт конвертация", где конвертация через CMS. Старше ageSeconds
     */
    @Override
    public List<VideoFiles> findOldConvertingCmsFiles(long ageSeconds) {
        VideoFilesRepository.QueryBuilder queryBuilder = new VideoFilesRepository.QueryBuilder()
                .withStatus(FileStatus.CONVERTING);
        queryBuilder.withConversionTaskId(null);
        queryBuilder.withNonNullVideoMetaId();
        queryBuilder.withDateBefore(Date.from(dateTimeService.getCurrentInstant().minusSeconds(ageSeconds)));
        return videoFilesRepository.findByQuery(queryBuilder);
    }
}

