package ru.yandex.canvas.service.video;

import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.collect.Sets;

import ru.yandex.canvas.model.stillage.StillageFileInfo;
import ru.yandex.canvas.service.FileValidator;
import ru.yandex.canvas.service.TankerKeySet;
import ru.yandex.canvas.service.VideoLimitsInterface;
import ru.yandex.canvas.service.video.presets.VideoPreset;

import static ru.yandex.canvas.VideoConstants.CANVAS_VIDEO_ANY_SIZE_ALLOWED_FEATURE;
import static ru.yandex.canvas.VideoConstants.VIDEO_FILE_RATE_LIMIT;
import static ru.yandex.direct.feature.FeatureName.SKIP_VIDEO_FILE_RATE_LIMIT;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Валидатор видеофайла. Зависит от VideoCreativeType.
 * limits зависят от VideoCreativeType.
 */
public class MovieValidator extends FileValidator {

    public static final String INVALID_VIDEO_FILE_FORMAT = "invalid_video_file_format";

    private final VideoMetaData videoMetaData;
    private final StillageFileInfo fileInfo;
    private final VideoLimitsInterface limits;
    private final VideoCreativeType videoCreativeType;
    private final Set<String> features;
    private final Long clientConvertingFilesCount;
    private final VideoPreset preset;
    private final VideoGeometryService videoGeometryService;

    public MovieValidator(VideoMetaData videoMetaData, StillageFileInfo fileInfo,
                          VideoLimitsInterface limits, VideoCreativeType videoCreativeType,
                          Set<String> features, VideoPreset preset, VideoGeometryService videoGeometryService,
                          Long clientConvertingFilesCount) {
        super(TankerKeySet.VIDEO_VALIDATION_MESSAGES);
        this.videoMetaData = videoMetaData;
        this.fileInfo = fileInfo;
        this.limits = limits;
        this.videoCreativeType = videoCreativeType;
        this.features = features;
        this.clientConvertingFilesCount = clientConvertingFilesCount;
        this.preset = preset;
        this.videoGeometryService = videoGeometryService;
    }

    @Override
    public void validate() {
        if (features.contains(CANVAS_VIDEO_ANY_SIZE_ALLOWED_FEATURE)
                || videoCreativeType == VideoCreativeType.HTML5) {
            return;
        }

        if (fileInfo.getFileSize() > limits.getVideoSizeMbMax() * 1024 * 1024) {
            addErrorMessage("video-is-too-large", limits.getVideoSizeMbMax());
        }
        if (clientConvertingFilesCount >= VIDEO_FILE_RATE_LIMIT
                && !features.contains(SKIP_VIDEO_FILE_RATE_LIMIT.getName())) {
            addErrorMessageAndThrowValidationErrors("converting-limit-exceeded", VIDEO_FILE_RATE_LIMIT);
        }

        if (!validateMimeType()) {
            throwValidationErrors();
        }

        ArrayList<Double> durations = new ArrayList<>();
        durations.add(videoMetaData.getDuration());

        if (videoMetaData.getVideoStreams() == null || videoMetaData.getVideoStreams().isEmpty()) {
            //нет смысла дальше валидировать, если видеопотоков 0 штук
            addErrorMessageAndThrowValidationErrors(INVALID_VIDEO_FILE_FORMAT);
        }
        if (videoMetaData.getVideoStreams().size() > 1) {
            addErrorMessageAndThrowValidationErrors("only_one_video_file_allowed");
        }

        VideoMetaData.VideoStreamInfo videoStreamInfo = videoMetaData.getVideoStreams().get(0);

        boolean isDurationByRationOk = true;
        if (validateFrameRate(videoStreamInfo.getFrameRate())) {
            if (validateCodecs(videoStreamInfo.getCodec())) {
                validateWidthAndHeight(videoStreamInfo.getWidth(), videoStreamInfo.getHeight());
                isDurationByRationOk = validateLimitsByRatios(videoStreamInfo);
                durations.add(videoStreamInfo.getDuration());
            }
        }

        // some formats may not provide duration for streams -- rely on global duration validation
        // see https://st.yandex-team.ru/CANVAS-720
        List<VideoMetaData.AudioStreamInfo> audioStreams = videoMetaData.getAudioStreams();
        if (limits.getAudioRequired() && (audioStreams == null || audioStreams.size() > 1)) {
            addErrorMessage("audio_required");
        } else if (!limits.getAudioRequired() && (audioStreams != null && audioStreams.size() > 0)) {
            addErrorMessage("audio_not_allowed");
        } else if (limits.getAudioRequired() && audioStreams != null && audioStreams.size() == 1) {
            VideoMetaData.AudioStreamInfo audioStreamInfo = audioStreams.get(0);
            validateAudioStream(audioStreamInfo);
            if (audioStreamInfo.getBaseInfo() != null) {
                durations.add(audioStreamInfo.getBaseInfo().getDuration());
            }
        }

        if (isDurationByRationOk) { // since errors are basically the same
            for (Double d : durations) {
                if (d != null && !validateDuration(d)) { // if present and improper
                    break; // one error of this type is enough
                }
            }
        }

        throwValidationErrors();
    }

    private boolean validateAudioStream(VideoMetaData.AudioStreamInfo audioStreamInfo) {
        Set<String> allowedAudioCodecs = Sets.newHashSet(limits.getAllowedAudioCodecs());
        if (audioStreamInfo.getBaseInfo() == null) {
            addErrorMessage("invalid_audio_file_format");
        } else if (!allowedAudioCodecs.contains(audioStreamInfo.getBaseInfo().getCodec())) {
            return addErrorMessage("invalid_audio_codec", joinArgs(allowedAudioCodecs));
        }

        return true;
    }

    private boolean validateLimitsByRatios(VideoMetaData.VideoStreamInfo videoStreamInfo) {
        Map<String, Object> limitsByRatio = limits.getDurationLimitsByRatio();

        if (limitsByRatio == null) {
            return true;
        }
        Ratio ratio = new Ratio(videoStreamInfo.getWidth(), videoStreamInfo.getHeight());

        String ratioString = ratio.toString();

        Map<String, Object> map = (Map) limitsByRatio.get(ratioString);

        if (map == null) {
            return true;
        }

        boolean isLimitsByRatioOk = true;
        if (map.containsKey("durations")) {
            List<Number> availableDurations = (List) map.get("durations");
            Double videoDuration = videoStreamInfo.getDuration();

            isLimitsByRatioOk = availableDurations.stream().anyMatch(duration -> {
                double min = duration.doubleValue() - limits.getDurationDelta();
                double max = duration.doubleValue() + limits.getDurationDelta();
                return in(videoDuration, min, max);
            });

            if (!isLimitsByRatioOk) {
                DecimalFormat durationFormat = new DecimalFormat("#.##");
                String durationsString = joinArgs(mapList(availableDurations, durationFormat::format));
                addErrorMessage("incompatible_duration_for_ratio", ratioString, durationsString);
            }
        }

        if (map.containsKey("minWidth")) {
            Integer minWidth = (Integer) map.get("minWidth");
            if (videoStreamInfo.getWidth() < minWidth) {
                isLimitsByRatioOk = false;
                addErrorMessage("incompatible_width_for_ratio", ratioString, minWidth);
            }
        }

        if (map.containsKey("minHeight")) {
            Integer minHeight = (Integer) map.get("minHeight");
            if (videoStreamInfo.getHeight() < minHeight) {
                isLimitsByRatioOk = false;
                addErrorMessage("incompatible_height_for_ratio", ratioString, minHeight);
            }
        }

        return isLimitsByRatioOk;
    }

    private boolean validateMimeType() {
        if (!limits.getKnownMimeTypes().contains(fileInfo.getMimeType())) {
            return addErrorMessage("unknown_mime_type", fileInfo.getMimeType());
        }
        return true;
    }

    private void validateWidthAndHeight(Integer width, Integer height) {
        boolean widthAndHeightCorrect = true;

        if (width == null || notIn(width, limits.getVideoWidthMin(), limits.getVideoWidthMax())) {
            widthAndHeightCorrect = false;
            addErrorMessage("invalid_video_width", width, limits.getVideoWidthMin(), limits.getVideoWidthMax());
        }

        if (height == null || notIn(height, limits.getVideoHeightMin(), limits.getVideoHeightMax())) {
            widthAndHeightCorrect = false;
            addErrorMessage("invalid_video_height", height, limits.getVideoHeightMin(), limits.getVideoHeightMax());
        }

        if (!widthAndHeightCorrect) {
            return;// если длина или ширина не валидны, нет смысла проверять соотношение сторон
        }

        validateVideoRatio(new Ratio(width, height));
    }

    private void validateVideoRatio(Ratio ratio) {
        if (preset != null && videoGeometryService.hasAllowedRatiosInterval(preset.getDescription().getGeometry(),
                features, videoCreativeType)) {
            var ratios = videoGeometryService.getRatiosByPreset(preset.getDescription().getGeometry(), features,
                    videoCreativeType);

            var ratioFrom = ratios.getFrom();
            var ratioTo = ratios.getTo();

            if (ratioFrom.compareTo(ratio) < 0 || ratioTo.compareTo(ratio) > 0) {
                if (preset.getDescription().getGeometry() == Geometry.UNIVERSAL) {
                    addErrorMessage("incompatible_video_ratio", ratio, ratioFrom, ratioTo);
                } else {
                    if (ratio.isWideVideo()) {
                        if (preset.getDescription().getGeometry() == Geometry.WIDE) {
                            addErrorMessage("incompatible_video_ratio", ratio, ratioFrom, ratioTo);
                        } else {
                            addErrorMessage("incorrect_video_ratio_horizontal", ratio, ratioFrom, ratioTo);
                        }
                    } else if (ratio.isTallVideo()) {
                        if (preset.getDescription().getGeometry() == Geometry.TALL) {
                            addErrorMessage("incompatible_video_ratio", ratio, ratioFrom, ratioTo);
                        } else {
                            addErrorMessage("incorrect_video_ratio_vertical", ratio, ratioFrom, ratioTo);
                        }
                    } else {
                        addErrorMessage("incorrect_video_ratio_square", ratio, ratioFrom, ratioTo);
                    }
                }
            }
        } else {
            List<Ratio> allowedVideoHwRatio = videoGeometryService.getAllowedRatios(videoCreativeType,
                    ifNotNull(preset, VideoPreset::getId));
            if (!allowedVideoHwRatio.isEmpty() && !allowedVideoHwRatio.contains(ratio)) {
                addErrorMessage("incompatible_video_ratio_list", ratio.toString(),
                        allowedVideoHwRatio.stream()
                                .filter(Objects::nonNull)
                                .map(Ratio::toString)
                                .collect(Collectors.joining(","))
                );
            }
        }
    }

    private boolean notIn(Number value, Number min, Number max) {
        return !in(value, min, max);
    }

    private boolean in(Number value, Number min, Number max) {
        return value.doubleValue() >= min.doubleValue() && value.doubleValue() <= max.doubleValue();
    }

    private boolean validateCodecs(String codec) {
        Set<String> allowedVideoCodecs = Sets.newHashSet(limits.getAllowedVideoCodecs());
        if (!allowedVideoCodecs.contains(codec)) {
            return "unknown codec".equals(codec)
                    ? addErrorMessage("unknown_video_codec", joinArgs(allowedVideoCodecs))
                    : addErrorMessage("invalid_video_codec", codec, joinArgs(allowedVideoCodecs));
        }
        return true;
    }

    private boolean validateFrameRate(Integer frameRate) {
        if (notIn(frameRate, limits.getVideoFrameRateMin(), limits.getVideoFrameRateMax())) {
            return addErrorMessage("invalid_video_framerate", frameRate, limits.getVideoFrameRateMin(),
                    limits.getVideoFrameRateMax());
        }

        return true;
    }

    private boolean validateDuration(Double duration) {
        double min = limits.getDurationMin() - limits.getDurationDelta();
        double max = limits.getDurationMax() + limits.getDurationDelta();

        if (duration != null && notIn(duration, min, max)) {
            return addErrorMessage("invalid_video_duration", duration,
                    limits.getDurationMin(), limits.getDurationMax());
        }
        return true;
    }

    private String joinArgs(Iterable<String> args) {
        return String.join(", ", args);
    }

}
