package ru.yandex.canvas.service.video;

import java.util.Set;

import com.google.common.collect.ImmutableSet;

import ru.yandex.canvas.model.stillage.StillageFileInfo;
import ru.yandex.canvas.service.FileValidator;
import ru.yandex.canvas.service.TankerKeySet;

import static ru.yandex.canvas.VideoConstants.IMAGE_CPM_AUDIO_HEIGHT_MAX;
import static ru.yandex.canvas.VideoConstants.IMAGE_CPM_AUDIO_HEIGHT_MIN;
import static ru.yandex.canvas.VideoConstants.IMAGE_CPM_AUDIO_WIDTH_MAX;
import static ru.yandex.canvas.VideoConstants.IMAGE_CPM_AUDIO_WIDTH_MIN;

public class PackshotCpmAudioValidator extends FileValidator {
    private Set<String> allowedMimeTypes = ImmutableSet.of("image/jpeg", "image/gif", "image/png");
    private final static Integer FILESIZE_MAX = 90_000;

    private StillageFileInfo stillageFileInfo;
    private PackshotUploadingService.ImageMetaDataInfo metaDataInfo;

    public PackshotCpmAudioValidator(StillageFileInfo stillageFileInfo,
                             PackshotUploadingService.ImageMetaDataInfo metaDataInfo) {
        super(TankerKeySet.VIDEO_VALIDATION_MESSAGES);
        this.stillageFileInfo = stillageFileInfo;
        this.metaDataInfo = metaDataInfo;
    }
    @Override
    public void validate() {
        if (!allowedMimeTypes.contains(stillageFileInfo.getMimeType())) {
            addErrorMessage("mime_type_not_supported_audio_packshot", "jpeg, gif, png");
        }

        if (metaDataInfo.getWidth() == null || metaDataInfo.getHeight() == null) {
            throwValidationErrors();
        }

        if (metaDataInfo.getWidth() < IMAGE_CPM_AUDIO_WIDTH_MIN
                || metaDataInfo.getWidth() > IMAGE_CPM_AUDIO_WIDTH_MAX) {
            addErrorMessage("incorrect_audio_image_width", metaDataInfo.getWidth(),
                    IMAGE_CPM_AUDIO_WIDTH_MIN, IMAGE_CPM_AUDIO_WIDTH_MAX);
        }

        if (metaDataInfo.getHeight() < IMAGE_CPM_AUDIO_HEIGHT_MIN
                || metaDataInfo.getHeight() > IMAGE_CPM_AUDIO_HEIGHT_MAX) {
            addErrorMessage("incorrect_audio_image_height", metaDataInfo.getHeight(),
                    IMAGE_CPM_AUDIO_HEIGHT_MIN, IMAGE_CPM_AUDIO_HEIGHT_MAX);
        }

        if (stillageFileInfo.getFileSize() > FILESIZE_MAX) {
            addErrorMessage("invalid_audio_packshot_filesize", stillageFileInfo.getFileSize(), FILESIZE_MAX);
        }

        if (stillageFileInfo.getMimeType().equals("image/gif")
                && metaDataInfo.getDuration() != null && metaDataInfo.getDuration() > 0) {
            addErrorMessage("animated_gif_not_supported");
        }
        throwValidationErrors();
    }
}
