package ru.yandex.canvas.service.video;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.canvas.exceptions.InternalServerError;
import ru.yandex.canvas.model.avatars.AvatarsPutCanvasResult;
import ru.yandex.canvas.model.stillage.StillageFileInfo;
import ru.yandex.canvas.model.video.VideoFiles;
import ru.yandex.canvas.model.video.files.FileStatus;
import ru.yandex.canvas.repository.video.VideoFilesRepository;
import ru.yandex.canvas.service.AvatarsService;
import ru.yandex.canvas.service.DirectService;
import ru.yandex.canvas.service.FileValidator;
import ru.yandex.canvas.service.SessionParams;
import ru.yandex.canvas.service.TankerKeySet;

public class PackshotUploadingService extends FileUploadService<PackshotUploadingService.ImageMetaDataInfo> {
    private static final Logger logger = LoggerFactory.getLogger(PackshotUploadingService.class);
    private final AvatarsService avatarsService;
    private final SessionParams sessionParams;
    private final VideoPresetsService videoPresetsService;

    private static final Map<String, Function<AvatarsPutCanvasResult, AvatarsPutCanvasResult.SizeInfo>> SIZE_GETTERS =
            Map.of("cropSource", e -> e.getSizes().getCropSource(),
                    "largePreview", e -> e.getSizes().getLargePreview(),
                    "optimize", e -> e.getSizes().getOptimize(),
                    "orig", e -> e.getSizes().getOrig(),
                    "preview", e -> e.getSizes().getPreview(),
                    "thumbnail", e -> e.getSizes().getThumbnail()
            );

    private final DirectService directService;
    private final VideoGeometryService videoGeometryService;

    public PackshotUploadingService(VideoFilesRepository videoFilesRepository, AvatarsService avatarsService,
                                    SessionParams sessionParams, VideoPresetsService videoPresetsService,
                                    DirectService directService, VideoGeometryService videoGeometryService) {
        super(videoFilesRepository);
        this.avatarsService = avatarsService;
        this.sessionParams = sessionParams;
        this.videoPresetsService = videoPresetsService;
        this.directService = directService;
        this.videoGeometryService = videoGeometryService;
    }

    @JsonIgnoreProperties(ignoreUnknown = true)
    public static class ImageMetaDataInfo {
        Double duration;
        Integer width;
        Integer height;

        public Double getDuration() {
            return duration;
        }

        public ImageMetaDataInfo setDuration(Double duration) {
            this.duration = duration;
            return this;
        }

        public Integer getWidth() {
            return width;
        }

        public ImageMetaDataInfo setWidth(Integer width) {
            this.width = width;
            return this;
        }

        public Integer getHeight() {
            return height;
        }

        public ImageMetaDataInfo setHeight(Integer height) {
            this.height = height;
            return this;
        }
    }

    @Override
    public void validate(ImageMetaDataInfo metaDataInfo, StillageFileInfo stillageFileInfo,
                         VideoCreativeType videoCreativeType, Long presetId,
                         Long clientId) {
        FileValidator validator = createFileValidator(metaDataInfo, stillageFileInfo, presetId, clientId,
                videoCreativeType);
        validator.validate();
    }

    private FileValidator createFileValidator(ImageMetaDataInfo metaDataInfo, StillageFileInfo stillageFileInfo,
                                              Long presetId, Long clientId, VideoCreativeType videoCreativeType) {
        if (sessionParams.isPresent() && sessionParams.getSessionType() == SessionParams.SessionTag.CPM_AUDIO) {
            return new PackshotCpmAudioValidator(stillageFileInfo, metaDataInfo);
        } else {
            var features = directService.getFeatures(clientId, null);
            return new PackshotValidator(stillageFileInfo, metaDataInfo, videoPresetsService.getPreset(presetId),
                    features, videoCreativeType, videoGeometryService);
        }
    }

    @Override
    public void beforeInsert(VideoFiles record, StillageFileInfo stillageFileInfo, ImageMetaDataInfo metaDataInfo,
                             Long presetId) {

        AvatarsPutCanvasResult result;
        try {
            result = avatarsService.upload(stillageFileInfo.getUrl());
        } catch (Exception e) {
            logger.error("avatarsService.upload failed", e);
            throw new InternalServerError(TankerKeySet.VIDEO.key("failed_to_save_file"));
        }

        List<VideoFiles.VideoFormat> formats = new ArrayList<>();

        for (Map.Entry<String, Function<AvatarsPutCanvasResult, AvatarsPutCanvasResult.SizeInfo>> entry : SIZE_GETTERS
                .entrySet()) {

            AvatarsPutCanvasResult.SizeInfo info = entry.getValue().apply(result);

            if (info == null) {
                continue;
            }

            VideoFiles.VideoFormat format = new VideoFiles.VideoFormat();
            format.setSize(entry.getKey());
            format.setUrl(info.getUrl());
            format.setWidth(info.getWidth() + "");
            format.setHeight(info.getHeight() + "");
            format.setPath(info.getPath());
            formats.add(format);
        }

        record.setFormats(formats);
        record.setThumbnailUrl(result.getSizes().getThumbnail().getUrl());
        record.setLargePreviewUrl(result.getSizes().getLargePreview().getUrl());
        record.setPreviewUrl(result.getSizes().getPreview().getUrl());
        record.setStatus(FileStatus.READY);

        AvatarsPutCanvasResult.SizeInfo origInfo = SIZE_GETTERS.get("orig").apply(result);
        if (origInfo != null) {
            Ratio ratio = new Ratio(origInfo.getWidth(), origInfo.getHeight());
            record.setRatio(ratio.toString());
            record.setRatioPercent(ratio.ratioPercent());
        }
    }

    @Override
    public void afterInsert(VideoFiles record) {

    }

    @Override
    protected ImageMetaDataInfo parseMetaData(StillageFileInfo info) {
        return new ObjectMapper().convertValue(info.getMetadataInfo(), ImageMetaDataInfo.class);
    }

}
