package ru.yandex.canvas.service.video;

import java.util.Set;

import com.google.common.collect.ImmutableSet;

import ru.yandex.canvas.model.stillage.StillageFileInfo;
import ru.yandex.canvas.service.FileValidator;
import ru.yandex.canvas.service.TankerKeySet;
import ru.yandex.canvas.service.video.presets.VideoPreset;

import static ru.yandex.canvas.VideoConstants.IMAGE_HEIGHT_MAX;
import static ru.yandex.canvas.VideoConstants.IMAGE_HEIGHT_MIN;
import static ru.yandex.canvas.VideoConstants.IMAGE_WIDTH_MAX;
import static ru.yandex.canvas.VideoConstants.IMAGE_WIDTH_MIN;

public class PackshotValidator extends FileValidator {
    private final Set<String> allowedMimeTypes = ImmutableSet.of("image/jpeg"); //image/gif, image/png
    private final StillageFileInfo stillageFileInfo;
    private final PackshotUploadingService.ImageMetaDataInfo metaDataInfo;
    private final VideoPreset preset;
    private final Set<String> clientFeatures;
    private final VideoCreativeType videoCreativeType;
    private final VideoGeometryService videoGeometryService;


    private final static Ratio DEFAULT_RATIO = new Ratio("16:9");

    public PackshotValidator(StillageFileInfo stillageFileInfo,
                             PackshotUploadingService.ImageMetaDataInfo metaDataInfo, VideoPreset preset,
                             Set<String> clientFeatures, VideoCreativeType videoCreativeType,
                             VideoGeometryService videoGeometryService) {
        super(TankerKeySet.VIDEO_VALIDATION_MESSAGES);
        this.stillageFileInfo = stillageFileInfo;
        this.metaDataInfo = metaDataInfo;
        this.preset = preset;
        this.clientFeatures = clientFeatures;
        this.videoCreativeType = videoCreativeType;
        this.videoGeometryService = videoGeometryService;
    }

    private void validateImageRatio() {
        Ratio ratio = new Ratio(metaDataInfo.getWidth(), metaDataInfo.getHeight());

        Ratio ratioFrom = DEFAULT_RATIO;
        Ratio ratioTo = DEFAULT_RATIO;

        if (preset != null && videoGeometryService.hasAllowedRatiosInterval(preset.getDescription().getGeometry(),
                clientFeatures, videoCreativeType)) {

            var ratiosInterval = videoGeometryService.getRatiosByPreset(preset.getDescription().getGeometry(),
                    clientFeatures, videoCreativeType);
            ratioFrom = ratiosInterval.getFrom();
            ratioTo = ratiosInterval.getTo();
        }

        if (ratioFrom.compareTo(ratio) < 0 || ratioTo.compareTo(ratio) > 0) {
            if (preset.getDescription().getGeometry() == Geometry.UNIVERSAL) {
                addErrorMessage("incorrect_image_ratio", ratio, ratioFrom, ratioTo);
            } else {
                int dr = ratio.getWidth() - ratio.getHeight();
                if (dr > 0) { //x:y > 1, горизонтального)
                    if (preset.getDescription().getGeometry() == Geometry.WIDE) {
                        addErrorMessage("incorrect_image_ratio", ratio, ratioFrom, ratioTo);
                    } else {
                        addErrorMessage("incorrect_image_ratio_horizontal", ratio, ratioFrom, ratioTo);
                    }
                } else if (dr < 0) { //x/y < 1, вертикального
                    if (preset.getDescription().getGeometry() == Geometry.TALL) {
                        addErrorMessage("incorrect_image_ratio", ratio, ratioFrom, ratioTo);
                    } else {
                        addErrorMessage("incorrect_image_ratio_vertical", ratio, ratioFrom, ratioTo);
                    }
                } else {
                    addErrorMessage("incorrect_image_ratio_squared", ratio, ratioFrom, ratioTo);
                }
            }
        }

    }

    @Override
    public void validate() {
        if (!allowedMimeTypes.contains(stillageFileInfo.getMimeType())) {
            addErrorMessage("mime_type_not_supported", "image/jpeg");
        }

        if (metaDataInfo.getWidth() == null || metaDataInfo.getHeight() == null) {
            throwValidationErrors();
        }

        if (metaDataInfo.getWidth() == null || metaDataInfo.getWidth() < IMAGE_WIDTH_MIN
                || metaDataInfo.getWidth() > IMAGE_WIDTH_MAX) {
            addErrorMessage("incorrect_image_width", metaDataInfo.getWidth(), IMAGE_WIDTH_MIN, IMAGE_WIDTH_MAX);
        }

        if (metaDataInfo.getHeight() == null || metaDataInfo.getHeight() < IMAGE_HEIGHT_MIN
                || metaDataInfo.getHeight() > IMAGE_HEIGHT_MAX) {
            addErrorMessage("incorrect_image_height", metaDataInfo.getHeight(), IMAGE_HEIGHT_MIN, IMAGE_WIDTH_MAX);
        }

        validateImageRatio();

        throwValidationErrors();
    }

}
