package ru.yandex.canvas.service.video;

import java.util.Objects;
import java.util.regex.Pattern;

import org.jetbrains.annotations.NotNull;

import static com.google.common.math.IntMath.gcd;

public class Ratio implements Comparable<Ratio> {
    private static final Pattern RATIO_PATTERN = Pattern.compile("^(\\d*):(\\d*)$");
    private int width;
    private int height;

    public Ratio(int width, int height) {
        int g = gcd(width, height);
        if (g == 0) {
            g = 1;
        }
        this.height = height / g;
        this.width = width / g;
        //есть требования директа и модерации что числа соотношения сторон не могут быть больше 255
        //CreativeValidationService.validateModerationInfoAspect
        //если не удалось подобрать точное соотношение сторон, то найдём примерное
        //такая ситуация возможно в рамках эксперимента с фичей canvas_video_any_size_allowed
        //у боевых креативов будет ограничено явно соотношение сторон чтобы инвентарь был к нему готов
        while (this.height > 255 || this.width > 255) {
            this.height = this.height / 2;
            this.width = this.width / 2;
        }
    }

    /**
     * Соотношение в формате width:height, например 3:1
     *
     * @param ratio
     */
    public Ratio(String ratio) {
        var matcher = RATIO_PATTERN.matcher(ratio);
        if (matcher.find()) {
            this.height = Integer.parseInt(matcher.group(2));
            this.width = Integer.parseInt(matcher.group(1));
        }
    }

    @Override
    public int compareTo(@NotNull Ratio o) {
        Double thisRatio = (double) width / (double) height;
        Double otherRatio = (double) o.width / (double) o.height;
        return thisRatio.compareTo(otherRatio);
    }

    /**
     * Возвращает соотношение ширины к высоте в процентах. Целое число. Нужно для фильтрации и сортировки.
     *
     * @return
     */
    public Integer ratioPercent() {
        if (height == 0) {
            return 0;
        }
        return width * 100 / height;
    }

    public boolean isWideVideo() {
        return getWidth() > getHeight();
    }

    public boolean isTallVideo() {
        return getWidth() < getHeight();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        Ratio ratio = (Ratio) o;
        return width == ratio.width &&
                height == ratio.height;
    }

    @Override
    public int hashCode() {
        return Objects.hash(width, height);
    }

    @Override
    public String toString() {
        return width + ":" + height;
    }

    public int getWidth() {
        return width;
    }

    public int getHeight() {
        return height;
    }
}
