package ru.yandex.canvas.service.video;

import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.github.mustachejava.DefaultMustacheFactory;
import com.github.mustachejava.Mustache;
import com.google.common.io.Resources;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.canvas.TimeDelta;
import ru.yandex.canvas.model.video.addition.AdditionElement;
import ru.yandex.canvas.model.video.files.Movie;

import static ru.yandex.canvas.service.video.VAST.injectMediaFiles;

public class StaticPreview {
    private static final Logger logger = LoggerFactory.getLogger(StaticPreview.class);
    private static final String TEMPLATE_NAME = "video_static_preview_template.html";

    private Movie movie;
    private String bundleName;
    private List<AdditionElement> elements;
    private DCParams dcParams;
    private TimeDelta skipOffset;
    private Long width;
    private Long height;
    private boolean isControlAllowed;

    private Mustache loadTemplater() throws IOException, URISyntaxException {
        URI presetsUri = Resources.getResource(TEMPLATE_NAME).toURI();
        String injectionHtml = new String(Files.readAllBytes(Paths.get(presetsUri)), StandardCharsets.UTF_8);
        return new DefaultMustacheFactory().compile(new StringReader(injectionHtml), "data_params.json");
    }

    public String makeStaticTemplate() throws IOException, URISyntaxException {
        String adParamsJson = new AdParams()
                .setDuration(movie.getDuration())
                .setElements(elements)
                .setSkipDelay(skipOffset)
                .setTheme(bundleName)
                .setHasAbuseButton(isControlAllowed)
                .setShowSkipButton(isControlAllowed)
                .toJson();

        Mustache mustache = loadTemplater();

        Map<String, Object> vars = new HashMap<>();
        vars.put("ad_parameters", adParamsJson);
        vars.put("width", width);
        vars.put("height", height);
        if(movie.getVideoSource().getThumbnail() != null) {
            vars.put("background_url", movie.getVideoSource().getThumbnail().getUrl());
        } else {
            logger.warn("Thumbnail is null, video source id: {}", movie.getVideoSource().getId());
        }

        StringWriter writer = new StringWriter();
        mustache.execute(writer, vars);

        String result = injectMediaFiles(writer.toString(), movie.getFormats());

        return result.replace(AdParams.RTB_HOST_DC_PARAMS_MACRO, dcParams.toJSON());
    }

    public StaticPreview setBundleName(String bundleName) {
        this.bundleName = bundleName;
        return this;
    }

    public StaticPreview setElements(List<AdditionElement> elements) {
        this.elements = elements;
        return this;
    }

    public StaticPreview setMovie(Movie movie) {
        this.movie = movie;
        return this;
    }

    public StaticPreview setDcParams(DCParams dcParams) {
        this.dcParams = dcParams;
        return this;
    }

    public StaticPreview setSkipOffset(TimeDelta skipOffset) {
        this.skipOffset = skipOffset;
        return this;
    }

    public StaticPreview setWidth(Long width) {
        this.width = width;
        return this;
    }


    public StaticPreview setHeight(Long height) {
        this.height = height;
        return this;
    }

    public StaticPreview setControlAllowed(boolean controlAllowed) {
        isControlAllowed = controlAllowed;
        return this;
    }

    public Movie getMovie() {
        return movie;
    }
}
