package ru.yandex.canvas.service.video;

import java.math.BigInteger;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Strings;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.canvas.exceptions.InternalServerError;
import ru.yandex.canvas.model.stillage.StillageFileInfo;
import ru.yandex.canvas.model.video.VideoFiles;
import ru.yandex.canvas.model.video.vh.TranscoderInfo;
import ru.yandex.canvas.model.video.vh.VhFormat;
import ru.yandex.canvas.model.video.vh.VhQualityMetrics;
import ru.yandex.canvas.model.video.vh.VhTranscoderParams;
import ru.yandex.canvas.model.video.vh.VhUploadPayload;
import ru.yandex.canvas.model.video.vh.VhUploadResponse;
import ru.yandex.canvas.model.video.vh.VhVideo;
import ru.yandex.direct.asynchttp.ErrorResponseWrapperException;
import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.asynchttp.Result;
import ru.yandex.direct.http.smart.annotations.Json;
import ru.yandex.direct.http.smart.core.Call;
import ru.yandex.direct.http.smart.core.Smart;
import ru.yandex.direct.http.smart.http.Body;
import ru.yandex.direct.http.smart.http.GET;
import ru.yandex.direct.http.smart.http.Headers;
import ru.yandex.direct.http.smart.http.POST;
import ru.yandex.direct.http.smart.http.Path;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmService;

import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;

/**
 * Сервис, работающий с CMS видеохостинга для конвертации видеофайлов.
 * Описание API https://wiki.yandex-team.ru/video/VideoPoisk/devops/vh-ugc/external_services/
 *
 * Примеры вызова
 * заказать конвертацию
 * curl -i -H "X-Ya-Service-Ticket: $(cat ~/.tvm)" -H 'Content-Type: application/json' -d '{"SourceUrl":
 * "https://storage.mds.yandex.net/get-bstor/2267090/3e2f55d0-8de2-4200-9fe2-cddd0742b7d5.mp4"}' 'https://vh.test
 * .yandex.ru/v1/upload'
 *
 * получить результат
 * curl -i -H "X-Ya-Service-Ticket: $(cat ~/.tvm)" -H 'Content-Type: application/json' 'https://vh.test.yandex.ru/v1/admin/video/10213176894360891132'
 */
public class VhService {
    private static final Logger logger = LoggerFactory.getLogger(VhService.class);

    private static final Map<String, String> FORMAT_NAME_TO_MIME_TYPE = Map.of(
            "EMp4", "video/mp4",
            "EHls", "application/vnd.apple.mpegurl",
            "EKaltura", "application/dash+xml",
            "EWebm", "video/webm");

    private final Api api;

    public VhService(String vhBaseUrl, TvmIntegration tvmIntegration, TvmService tvmService,
                     ParallelFetcherFactory fetcherFactory) {
        this.api = Smart.builder()
                .withParallelFetcherFactory(fetcherFactory)
                .useTvm(tvmIntegration, tvmService)
                .withProfileName("canvas_vh_client")
                .withBaseUrl(vhBaseUrl)
                .build()
                .create(Api.class);
    }

    public interface Api {
        @POST("upload")
        @Json
        @Headers("Content-Type: application/json")
        Call<VhUploadResponse> upload(@Body @Json VhUploadPayload payload);

        @GET("admin/video/{meta_id}")
        @Json
        @Headers("Content-Type: application/json")
        Call<VhVideo> getVideo(@Path("meta_id") BigInteger metaId);
    }

    public VhUploadResponse startEncoding(String url) {
        VhUploadPayload uploadPayload = new VhUploadPayload()
                .setUrl(url)
                .setTranscoderParams(new VhTranscoderParams().setGraph("ad"));
        return executeCall(api.upload(uploadPayload)).getSuccess();
    }

    public VhVideo getVideo(BigInteger metaId) {
        return executeCall(api.getVideo(metaId)).getSuccess();
    }

    private <T> Result<T> executeCall(Call<T> call) {
        logger.info("executing request \"{}\"", call.getRequest());
        Result<T> result = call.execute();
        if (result.getSuccess() == null && result.getErrors() != null) {
            logErrors(result.getErrors());
            throw new InternalServerError("request was failed");
        }
        return result;
    }

    private void logErrors(List<Throwable> errors) {
        logger.error("request failed ({})", errors
                .stream()
                .map(error -> {
                    var msg = error.toString();
                    if (error instanceof ErrorResponseWrapperException &&
                                    ((ErrorResponseWrapperException) error).getResponse() != null) {
                        ErrorResponseWrapperException e = (ErrorResponseWrapperException) error;
                        msg += " (" + e.getResponse().toString()
                                + " : " + e.getMessage()
                                + " : " + e.getResponse().getResponseBody()
                                + ")";
                    }
                    return msg;
                })
                .collect(Collectors.toList()));
    }

    // Поддерживаем 2 формата данных:
    // 1. Если есть GraphMeta, то это старый формат данных, берем инфу из него
    // 2. Иначе берем инфу из Streams
    public List<VideoFiles.VideoFormat> getFormatsFromCMSResult(TranscoderInfo info, StillageFileInfo originFileInfo) {
        //Фреймрейт транскодер не меняет, можно брать у оригинала. Обсуждали в DIRECT-103050
        Integer framerate = extractFramerate(originFileInfo);

        if (info.getGraphMeta() != null && info.getGraphMeta().getFormats() != null) {
            List<VideoFiles.VideoFormat> formats = new ArrayList<>();
            if (!Strings.isNullOrEmpty(info.getOutputUrl())) {
                VideoFiles.VideoFormat streamFormat = new VideoFiles.VideoFormat();
                String streamUrl = info.getOutputUrl();
                streamFormat.setId(Paths.get(streamUrl).getParent().toString());
                streamFormat.setMimeType("application/vnd.apple.mpegurl");
                streamFormat.setUrl(streamUrl);
                formats.add(streamFormat);
            }
            for (VhFormat format : info.getGraphMeta().getFormats()) {
                formats.add(new VideoFiles.VideoFormat()
                        .setDelivery("progressive")
                        .setUrl(format.getUrl())
                        .setBitrate(format.getBitrate())
                        .setMimeType(format.getType())
                        .setId(format.getId())
                        .setWidth(String.valueOf(format.getWidth()))
                        .setHeight(String.valueOf(format.getHeight()))
                        .setCodec(format.getCodecs())
                        .setFramerate(framerate)
                );
            }
            return formats;
        }

        return info.getStreams().stream()
                .filter(vhStream -> vhStream.getFormatName() != null &&
                        FORMAT_NAME_TO_MIME_TYPE.containsKey(vhStream.getFormatName()))
                .map(vhStream -> new VideoFiles.VideoFormat()
                        .setDelivery("progressive")
                        .setUrl(vhStream.getUrl())
                        .setBitrate(ifNotNull(vhStream.getBitrate(), VhService::bitrateFromBitToKbit))
                        .setMimeType(FORMAT_NAME_TO_MIME_TYPE.get(vhStream.getFormatName()))
                        .setFormatName(vhStream.getFormatName())
                        .setTagsStr(vhStream.getTagsStr())
                        .setId(Paths.get(vhStream.getUrl()).getFileName().toString())
                        .setWidth(vhStream.getWidth() == null ? null : String.valueOf(vhStream.getWidth()))
                        .setHeight(vhStream.getHeight() == null ? null : String.valueOf(vhStream.getHeight()))
                        .setCodec(vhStream.getCodecs())
                        .setFramerate(framerate)
                        .setFileSize(vhStream.getByteSize())
                        .setHasAudio(nvl(info.getHasAudioStream(), false))
                        .setVmafAvg(vhStream.getQualityMetrics() == null ? null :
                                vhStream.getQualityMetrics()
                                        .stream()
                                        .map(VhQualityMetrics::getVmafAvg)
                                        .filter(Objects::nonNull)
                                        .max(Double::compareTo)
                                        .orElse(null)))
                .collect(Collectors.toList());
    }

    private static Long bitrateFromBitToKbit(Long bit) {
        return bit / 1000;
    }

    private static Integer extractFramerate(StillageFileInfo originFileInfo) {
        if (originFileInfo == null || originFileInfo.getMetadataInfo() == null) {
            return null;
        }
        try {
            var videoMetaData = new ObjectMapper().convertValue(originFileInfo.getMetadataInfo(), VideoMetaData.class);
            if (videoMetaData.getVideoStreams() != null) {
                return videoMetaData.getVideoStreams()
                        .stream()
                        .map(VideoMetaData.VideoStreamInfo::getFrameRate)
                        .findAny()
                        .orElse(null);
            }
        } catch (Exception e) {
            logger.warn("parse videoMetaData error: ", e);
        }
        return null;
    }
}
