package ru.yandex.canvas.service.video;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;

import com.google.common.base.Strings;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Sets;
import org.slf4j.LoggerFactory;
import org.springframework.validation.BindingResult;

import ru.yandex.canvas.LocalizedBindingResultBuilder;
import ru.yandex.canvas.exceptions.BadRequestException;
import ru.yandex.canvas.model.validation.DifferentlyColoredOptions;
import ru.yandex.canvas.model.validation.Validator;
import ru.yandex.canvas.model.video.Addition;
import ru.yandex.canvas.model.video.addition.AdditionElement;
import ru.yandex.canvas.model.video.addition.Options;
import ru.yandex.canvas.model.video.addition.options.AdditionElementOptions;
import ru.yandex.canvas.model.video.addition.options.AgeElementOptions;
import ru.yandex.canvas.model.video.addition.options.BodyElementOptions;
import ru.yandex.canvas.model.video.addition.options.ButtonElementOptions;
import ru.yandex.canvas.model.video.addition.options.DisclaimerElementOptions;
import ru.yandex.canvas.model.video.addition.options.DomainElementOptions;
import ru.yandex.canvas.model.video.addition.options.LegalElementOptions;
import ru.yandex.canvas.model.video.addition.options.OptionName;
import ru.yandex.canvas.model.video.addition.options.SubtitlesElementOptions;
import ru.yandex.canvas.model.video.addition.options.TitleElementOptions;
import ru.yandex.canvas.model.video.files.AudioSource;
import ru.yandex.canvas.model.video.files.Movie;
import ru.yandex.canvas.model.video.files.PackShot;
import ru.yandex.canvas.service.DirectService;
import ru.yandex.canvas.service.TankerKeySet;
import ru.yandex.canvas.service.video.presets.VideoPreset;
import ru.yandex.canvas.service.video.presets.configs.AdditionConfig;
import ru.yandex.canvas.service.video.presets.configs.AgeConfig;
import ru.yandex.canvas.service.video.presets.configs.BaseConfig;
import ru.yandex.canvas.service.video.presets.configs.BodyConfig;
import ru.yandex.canvas.service.video.presets.configs.ButtonConfig;
import ru.yandex.canvas.service.video.presets.configs.ConfigType;
import ru.yandex.canvas.service.video.presets.configs.DisclaimerConfig;
import ru.yandex.canvas.service.video.presets.configs.DomainConfig;
import ru.yandex.canvas.service.video.presets.configs.LegalConfig;
import ru.yandex.canvas.service.video.presets.configs.SubtitlesConfig;
import ru.yandex.canvas.service.video.presets.configs.TitleConfig;
import ru.yandex.canvas.service.video.presets.configs.options.OptionConfig;
import ru.yandex.canvas.service.video.presets.configs.options.TextOptionConfig;

import static java.lang.Math.toIntExact;
import static ru.yandex.canvas.model.validation.Checkers.checkColor;
import static ru.yandex.canvas.model.validation.Checkers.checkDomain;
import static ru.yandex.canvas.model.validation.Checkers.checkLength;
import static ru.yandex.canvas.model.validation.Checkers.checkSubtitles;
import static ru.yandex.canvas.model.validation.Checkers.checkUnicodeText;
import static ru.yandex.direct.utils.CommonUtils.nvl;

/**
 * Валидация дополнения, его элементов и их опций в рамках пресета (VideoPreset)
 * Пресет состоит из описания данных элементов и конфигураций (Config) для отрисовки контролов элемента на фронте
 * Т.к. Config - это информация необходимая для отрисовки на виджета элемента дополнения на фронте, то
 * нет никакой гарантии, что информация из конфига будет достаточно для валидации
 * приходящих в элементе данных, поэтому при валидации доступен престет целиком
 * <p>
 * Существующие тексты ошибок см. в src/main/resources/localized/VideoValidationMessages_ru.properties
 */
public class VideoAdditionValidationService implements Validator<VideoAdditionValidationService.AdditionValidationObject> {
    private static final org.slf4j.Logger logger = LoggerFactory.getLogger(VideoAdditionValidationService.class);

    private static final double COLORS_MIN_DISTANCE = 9.8;

    private final TankerKeySet keyset = TankerKeySet.VIDEO_VALIDATION_MESSAGES;

    private static final Map<ConfigType, AdditionElement.ElementType> configTypeToElementTypeMap =
            ImmutableMap.<ConfigType, AdditionElement.ElementType>builder()
                    .put(ConfigType.ADDITION, AdditionElement.ElementType.ADDITION)
                    .put(ConfigType.AGE, AdditionElement.ElementType.AGE)
                    .put(ConfigType.BODY, AdditionElement.ElementType.BODY)
                    .put(ConfigType.BUTTON, AdditionElement.ElementType.BUTTON)
                    .put(ConfigType.DISCLAIMER, AdditionElement.ElementType.DISCLAIMER)
                    .put(ConfigType.DOMAIN, AdditionElement.ElementType.DOMAIN)
                    .put(ConfigType.LEGAL, AdditionElement.ElementType.LEGAL)
                    .put(ConfigType.TITLE, AdditionElement.ElementType.TITLE)
                    .put(ConfigType.SUBTITLES, AdditionElement.ElementType.SUBTITLES)
                    .build();

    private final VideoPresetsService videoPresetsService;
    private final MovieServiceInterface movieService;
    private final PackshotServiceInterface packshotService;
    private final AudioService audioService;
    private final DirectService directService;

    public VideoAdditionValidationService(VideoPresetsService videoPresetsService,
                                          MovieServiceInterface movieService,
                                          PackshotServiceInterface packshotService,
                                          AudioService audioService,
                                          DirectService directService) {
        this.videoPresetsService = videoPresetsService;
        this.movieService = movieService;
        this.packshotService = packshotService;
        this.audioService = audioService;
        this.directService = directService;
    }

    public BindingResult validate(Addition addition) {
        return validate(new AdditionValidationObject(addition, null));
    }

    public static class ValidationOptions {
        private boolean forPreview;

        private boolean emptyTitleOrBodyAllowed;

        public boolean isForPreview() {
            return forPreview;
        }

        public ValidationOptions setForPreview(boolean forPreview) {
            this.forPreview = forPreview;
            return this;
        }

        public boolean isEmptyTitleOrBodyAllowed() {
            return emptyTitleOrBodyAllowed;
        }

        public void setEmptyTitleOrBodyAllowed(boolean emptyTitleOrBodyAllowed) {
            this.emptyTitleOrBodyAllowed = emptyTitleOrBodyAllowed;
        }
    }

    public static class AdditionValidationObject {
        private Addition addition;
        private ValidationOptions options;

        public AdditionValidationObject(Addition addition,
                                        ValidationOptions options) {
            this.addition = addition;
            this.options = options;
        }

        public Addition getAddition() {
            return addition;
        }

        public ValidationOptions getOptions() {
            return options;
        }
    }

    public BindingResult validate(AdditionValidationObject validationObject) {
        return validateAdditionByPreset(validationObject.getAddition(), getVideoPreset(validationObject.getAddition()),
                validationObject.getOptions());
    }

    private VideoPreset getVideoPreset(Addition addition) {
        if (addition.getPresetId() == null || !videoPresetsService.contains(addition.getPresetId())) {
            throw new BadRequestException("{preset_id missed}");
        }

        return videoPresetsService.getPreset(addition.getPresetId());
    }

    public BindingResult validateAdditionByPreset(Addition addition, VideoPreset preset,
                                                  @Nullable ValidationOptions options) {
        Optional<AdditionElement> additionElement =
                addition.getData().getElements()
                        .stream()
                        .filter(e -> AdditionElement.ElementType.ADDITION.equals(e.getType()))
                        .findFirst();

        // обязательный параметр, независимо от пресета/конфига
        if (additionElement.isEmpty()) {
            throw new BadRequestException("{missed_mandatory_addition_element}");
        }

        // для нестоковых видео разрешаем пустые title и body
        if (!isStockMovie(addition, additionElement.get())) {
            // в некоторых случаях в options в явном виде передают null
            options = nvl(options, new ValidationOptions());
            options.setEmptyTitleOrBodyAllowed(true);
        }

        LocalizedBindingResultBuilder bindingResult =
                new LocalizedBindingResultBuilder(addition.getData(), "AdditionData", keyset);

        OptionsValidator optionsValidator =
                new OptionsValidator(preset, addition, movieService, bindingResult, options, packshotService,
                        audioService, directService);

        bindingResult.validate("elements",
                (e, evr) -> evr.validate("options",
                        (o, ovr) -> optionsValidator.validate((AdditionElement) e)));
        return bindingResult.build();
    }

    private boolean isStockMovie(Addition addition, AdditionElement additionElement) {
        try {
            var options = additionElement.getOptions();
            var movie = movieService.lookupMovie(
                    options.getVideoId(), options.getAudioId(), addition.getClientId(), addition.getPresetId());
            return movie == null || movie.isStock();
        } catch (IllegalArgumentException ignored) {
            // тут интересует только флаг сток / не сток, поэтому исключение игнорим,
            // оно будет учтено в additionElementOptionsValidation
            return false;
        }
    }

    public List<AdditionElement.ElementType> getAllowedElements(Addition addition) {
        VideoPreset videoPreset = getVideoPreset(addition);
        Map<ConfigType, BaseConfig> configs = videoPreset.getConfig().getConfigs();
        List<AdditionElement.ElementType> elementsInPreset =
                configs.values().stream().map(c -> getAdditionElementTypeByConfigType(c.getConfigType()))
                        .collect(Collectors.toList());
        return elementsInPreset;
    }

    /**
     * @return AdditionElement which is validated by validator built upon this config
     */

    public static AdditionElement.ElementType getAdditionElementTypeByConfigType(ConfigType configType) {
        return getConfigToElementTypeMap().get(configType);
    }

    public static Map<ConfigType, AdditionElement.ElementType> getConfigToElementTypeMap() {
        return configTypeToElementTypeMap;
    }

    /**
     * Валидируем опции элемента по конфигу элемента (полный VideoPreset все еще доступен)
     */
    public static class OptionsValidator implements Validator<AdditionElement> {
        private VideoPreset videoPreset;
        private Addition addition;
        private MovieServiceInterface mediaFilesService;
        private PackshotServiceInterface packshotService;
        private AudioService audioService;
        private LocalizedBindingResultBuilder bindingResult;
        private ValidationOptions options;
        private DirectService directService;

        private Map<AdditionElement.ElementType, BaseConfig> additionTypeToConfig;

        // map for testing options regarding element type it is in
        Map<AdditionElement.ElementType, ElementOptionValidator> elementTypeElementOptionsValidatorMap =
                ImmutableMap.<AdditionElement.ElementType, ElementOptionValidator>builder()
                        .put(AdditionElement.ElementType.ADDITION,
                                (o, c) -> additionElementOptionsValidation((AdditionElementOptions) o,
                                        (AdditionConfig) c))
                        .put(AdditionElement.ElementType.AGE,
                                (o, c) -> ageElementOptionsValidation((AgeElementOptions) o, (AgeConfig) c))
                        .put(AdditionElement.ElementType.BODY,
                                (o, c) -> bodyElementOptionsValidation((BodyElementOptions) o, (BodyConfig) c))
                        .put(AdditionElement.ElementType.BUTTON,
                                (o, c) -> buttonElementOptionsValidation((ButtonElementOptions) o, (ButtonConfig) c))
                        .put(AdditionElement.ElementType.DISCLAIMER,
                                (o, c) -> disclaimerElementOptionsValidation((DisclaimerElementOptions) o,
                                        (DisclaimerConfig) c))
                        .put(AdditionElement.ElementType.DOMAIN,
                                (o, c) -> domainElementOptionsValidation((DomainElementOptions) o, (DomainConfig) c))
                        .put(AdditionElement.ElementType.LEGAL,
                                (o, c) -> legalElementOptionsValidation((LegalElementOptions) o, (LegalConfig) c))
                        .put(AdditionElement.ElementType.TITLE,
                                (o, c) -> titleElementOptionsValidation((TitleElementOptions) o, (TitleConfig) c))
                        .put(AdditionElement.ElementType.SUBTITLES, (o, c) ->
                                subtitlesElementOptionsValidation((SubtitlesElementOptions) o, (SubtitlesConfig) c))
                        .build();

        @FunctionalInterface
        interface ElementOptionValidator {
            void validate(Options options, BaseConfig t);
        }

        public OptionsValidator(VideoPreset videoPreset, Addition addition, MovieServiceInterface movieService,
                                LocalizedBindingResultBuilder bindingResult, ValidationOptions options,
                                PackshotServiceInterface packshotService, AudioService audioService,
                                DirectService directService) {
            this.videoPreset = videoPreset;
            this.addition = addition;
            this.mediaFilesService = movieService;
            this.bindingResult = bindingResult;
            this.options = options;
            this.packshotService = packshotService;
            this.audioService = audioService;
            this.directService = directService;
            additionTypeToConfig = new HashMap<>();

            for (BaseConfig config : this.videoPreset.getConfig().getConfigs().values()) {
                additionTypeToConfig.put(getAdditionElementTypeByConfigType(config.getConfigType()), config);
            }
        }

        private BaseConfig configByElementType(AdditionElement.ElementType type) {
            if (!additionTypeToConfig.containsKey(type)) {
                rejectValue(OptionName.VIDEO_ID.toFieldName(), "unexpected_element", type.toValue());
                return null;
            }

            return additionTypeToConfig.get(type);
        }

        public BindingResult validate(AdditionElement element) {
            AdditionElement.ElementType type = element.getType();
            BaseConfig config = configByElementType(type);

            if (config == null) {
                return null;
            }

            ElementOptionValidator elementOptionsValidator = elementTypeElementOptionsValidatorMap.get(type);

            if (elementOptionsValidator == null) {
                throw new IllegalStateException("No validator found for element type: " + type.toValue());
            }

            Options options = element.getOptions();

            if (element.getAvailable()) {
                validateOptionsAllRequired(config, options);
                validateOptionsRedundantNotPresent(config, options);
                validateOptionsFieldsByType(config, options, element);
                validateColorsDifferentEnough(options);
                elementOptionsValidator.validate(options, config);
            }

            return null;
        }

        /**
         * Проверки одинаковые для опций определенного типа не зависимо от пресета
         *
         * @param config
         * @param options
         * @param element
         */
        public void validateOptionsFieldsByType(BaseConfig config, Options options, AdditionElement element) {
            for (OptionConfig optionConfig : config.getOptionConfigs()) {

                OptionConfig.OptionType type = optionConfig.getType();
                String field = optionConfig.getOptionName().toFieldName();
                String value = options.optionByName(optionConfig.getName());

                if (value == null) {
                    continue;
                }

                if (OptionConfig.OptionType.COLOR_PICKER.equals(type)) {
                    validateColorPicker(field, value);
                } else if (OptionConfig.OptionType.TEXT == type || OptionConfig.OptionType.TEXT_AREA == type) {
                    validateText(field, value, ((TextOptionConfig) optionConfig).getLimit(), optionConfig,
                            config.getConfigType(), element);
                }
            }
        }

        private void validateColorPicker(String field, String color) {
            if (!checkColor(color)) {
                rejectValue(field, "invalid_color_value");
            }
        }

        private void validateText(String field, String text, Integer limit, OptionConfig optionConfig,
                                  ConfigType configType, AdditionElement element) {
            if (optionConfig.getName().equals(OptionName.PLACEHOLDER.toFieldName()) || optionConfig.getEditable()) {
                if (!checkUnicodeText(text)) {
                    rejectValue(field, "invalid_unicode_symbols");
                }
                if (limit != null && limit > 0) {
                    int min = (configType.equals(ConfigType.BODY) || configType.equals(ConfigType.TITLE))
                            && (this.options != null && this.options.isEmptyTitleOrBodyAllowed()) ? 0 : 1;

                    if (!checkLength(text, limit, min)) {
                        rejectValue(field, "javax.validation.constraints.Size.message", min, limit);
                    }

                    if ((configType.equals(ConfigType.BODY) || configType.equals(ConfigType.TITLE))
                            && (this.options != null && this.options.isEmptyTitleOrBodyAllowed())
                            && Strings.isNullOrEmpty(text)) {
                        element.withAvailable(false);
                    }
                }
            } else if (text != null && text.length() > 0) {
                rejectValue(field, "field_is_not_null");
            }
        }

        public void validateOptionsRedundantNotPresent(BaseConfig config, Options options) {
            for (OptionName optionName : OptionName.values()) {
                String option = options.optionByName(optionName);
                if (option != null && config.getOptionConfigByName(optionName) == null) {
                    rejectValue(optionName.toFieldName(), "field_is_not_null");
                }
            }
        }

        public void validateOptionsAllRequired(BaseConfig config, Options options) {
            for (OptionConfig optionConfig : config.getOptionConfigs()) {
                OptionName optionName = OptionName.fromValue(optionConfig.getName());
                String value = options.optionByName(optionName);

                if (optionConfig.getRequired() && value == null && optionConfig.getEditable()) {
                    rejectValue(optionName.toFieldName(), "org.hibernate.validator.constraints.NotBlank.message");
                }
            }
        }

        private void validateColorsDifferentEnough(Options options) {
            List<OptionName> colorOptions = Stream.of(
                    OptionName.COLOR,
                    OptionName.TEXT_COLOR,
                    OptionName.BACKGROUND_COLOR,
                    OptionName.BORDER_COLOR
            ).filter(n -> options.optionByName(n) != null).collect(Collectors.toList());

            if (colorOptions.size() < 2) {
                return;
            }

            Set<OptionName> matchedColorOptions = Sets.newHashSet();
            try {
                //N square algo, but can be only 6 pairs
                for (int i = 0; i < colorOptions.size(); i++) {
                    for (int j = i + 1; j < colorOptions.size(); j++) {
                        OptionName o1 = colorOptions.get(i);
                        String color1 = options.optionByName(o1);
                        OptionName o2 = colorOptions.get(j);
                        String color2 = options.optionByName(o2);
                        if (colorsAreTooClose(color1, color2)) {
                            // чтобы не добавлять ошибку на одно поле более одного раза
                            matchedColorOptions.add(o1);
                            matchedColorOptions.add(o2);
                        }
                    }
                }
            } catch (NumberFormatException e) {
                logger.warn("can't compute color difference", e);
            }

            for (OptionName o : matchedColorOptions) {
                rejectValue(o.toFieldName(), "colors_are_too_close");
            }
        }

        private static boolean colorsAreTooClose(String color1, String color2) {
            // https://st.yandex-team.ru/BANNERSTORAGE-5422 Желтый на белом читается плохо
            String yellow = "#ffdc00";
            String white = "#ffffff";
            return DifferentlyColoredOptions.Metric.CIEDE2000.almostTheSameColors(color1, color2, COLORS_MIN_DISTANCE)
                    || (yellow.equals(color1.toLowerCase()) && white.equals(color2.toLowerCase()))
                    || (yellow.equals(color2.toLowerCase()) && white.equals(color1.toLowerCase()));
        }

        public PackShot packshotOptionsValidation(AdditionElementOptions options, AdditionConfig config) {
            String packshotId = options.getPackshotId();

            if (packshotId == null) {
                return null;
            }

            if (!videoPreset.getPackshotAllowed()) {
                rejectValue(OptionName.PACKSHOT_ID.toFieldName(), "packshot_not_allowed");
            } else {
                PackShot packShot = packshotService.lookupPackshot(packshotId, addition.getClientId());

                if (packShot == null) {
                    rejectValue(OptionName.PACKSHOT_ID.toFieldName(), "packshot_not_found");
                }

                return packShot;
            }

            return null;
        }

        public void additionElementAudioCreativeOptionsValidation(AdditionElementOptions options,
                                                                  AdditionConfig config) {
            packshotOptionsValidation(options, config);

            if (this.options != null && this.options.isForPreview()) {
                //если на этапе превью ещё не загрузили файл, то не нужно показывать ошибку.
                // Валидировать будем только по кнопке "создать"
                return;
            }
            String audioId = options.getAudioId();
            if (audioId == null) {
                rejectValue(OptionName.AUDIO_ID.toFieldName(), "audio_file_not_present");
                return;
            }
            AudioSource audioSource = audioService.lookupAudio(audioId, addition.getClientId());
            if (audioSource == null) {
                rejectValue(OptionName.AUDIO_ID.toFieldName(), "audio_file_not_present");
                return;
            }
        }

        public void additionElementOptionsValidation(AdditionElementOptions options, AdditionConfig config) {
            if (videoPreset.getDescription().getCpmAudio()) {
                additionElementAudioCreativeOptionsValidation(options, config);
                return;
            }

            final PackShot packShot = packshotOptionsValidation(options, config);

            String videoId = options.getVideoId();
            String audioId = options.getAudioId();

            final Movie movie;

            try {
                movie = mediaFilesService.lookupMovie(videoId, audioId, addition.getClientId(), addition.getPresetId());
            } catch (IllegalArgumentException e) {
                rejectValue(OptionName.VIDEO_ID.toFieldName(), "video_file_invalid");
                return;
            }

            if (movie == null) {
                rejectValue(OptionName.VIDEO_ID.toFieldName(), "video_file_not_found");
            } else if ((movie.getFormats().isEmpty() ||
                    ((this.options == null || !this.options.isForPreview()) && !movie.isReady())) &&
                    !movie.isCreateEarlyCreativeAllowed(videoPreset.videoCreativeType(),
                            directService.getFeatures(addition.getClientId(), null))) {
                rejectValue(OptionName.VIDEO_ID.toFieldName(), "video_file_not_ready");
            } else if (movie.isStock() && !videoPreset.getAllowStockVideo() && (this.options == null || !this.options
                    .isForPreview())) {
                final String message;

                if (!videoPreset.isRecentVideoAllowed()) {
                    message = "stock_video_not_allowed_upload";
                } else {
                    message = "stock_video_not_allowed";
                }

                rejectValue(OptionName.VIDEO_ID.toFieldName(), message);
            }

            if (movie != null && audioId != null) {
                if (!movie.isStock()) {
                    rejectValue(OptionName.AUDIO_ID.toFieldName(), "audio_should_be_empty");
                } else if (movie.getAudioSource() == null) {
                    rejectValue(OptionName.AUDIO_ID.toFieldName(), "audio_file_not_found");
                }
            }

            if (movie != null && packShot != null) {
                Ratio movieRatio = new Ratio(movie.getVideoSource().getRatio());
                PackShot.ImageFormat format = packShot.getFormats().stream()
                        .filter(e -> e.getSize().equals("optimize"))
                        .findFirst().orElse(null);

                if (format != null) {
                    Ratio packshotRatio = new Ratio(toIntExact(format.getWidth()), toIntExact(format.getHeight()));

                    if (packshotRatio.compareTo(movieRatio) != 0) {
                        rejectValue(OptionName.PACKSHOT_ID.toFieldName(), "packshot_ratio_doesnt_match_movie_ratio",
                                packshotRatio.toString());
                    }
                }
            }

        }

        public void ageElementOptionsValidation(AgeElementOptions options, AgeConfig config) {
            // covered in validateOptionsFieldsByType
        }

        public void bodyElementOptionsValidation(BodyElementOptions options, BodyConfig config) {
            // covered in validateOptionsFieldsByType
        }

        public void buttonElementOptionsValidation(ButtonElementOptions options, ButtonConfig config) {
            // covered in validateOptionsFieldsByType
        }

        public void disclaimerElementOptionsValidation(DisclaimerElementOptions options, DisclaimerConfig config) {
            // covered in validateOptionsFieldsByType
        }

        public void domainElementOptionsValidation(DomainElementOptions options, DomainConfig config) {
            if (options.getText() != null && !checkDomain(options.getText())) {
                rejectValue(OptionName.TEXT.toFieldName(), "invalid_domain_value");
            }
        }

        public void legalElementOptionsValidation(LegalElementOptions options, LegalConfig config) {
            // covered in validateOptionsFieldsByType
        }

        public void titleElementOptionsValidation(TitleElementOptions options, TitleConfig config) {
            // covered in validateOptionsFieldsByType
        }

        public void subtitlesElementOptionsValidation(SubtitlesElementOptions options, SubtitlesConfig config) {
            if (options.getText() != null && !checkSubtitles(options.getText())) {
                rejectValue(OptionName.TEXT.toFieldName(), "invalid_subtitles_format");
            }
        }

        private void rejectValue(String field, String message, Object... messageArgs) {
            bindingResult.rejectValue(field, message, messageArgs);
        }
    }
}
