package ru.yandex.canvas.service.video;

import java.io.IOException;
import java.net.URISyntaxException;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import javax.validation.constraints.NotNull;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.ImmutableList;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.domain.Sort;
import org.springframework.util.CollectionUtils;

import ru.yandex.canvas.TimeDelta;
import ru.yandex.canvas.controllers.video.PreviewResponseEntity;
import ru.yandex.canvas.exceptions.InternalServerError;
import ru.yandex.canvas.exceptions.NotFoundException;
import ru.yandex.canvas.model.direct.CreativeCampaignResult;
import ru.yandex.canvas.model.stillage.StillageFileInfo;
import ru.yandex.canvas.model.video.Addition;
import ru.yandex.canvas.model.video.CustomVastParams;
import ru.yandex.canvas.model.video.PythiaParams;
import ru.yandex.canvas.model.video.VideoFiles;
import ru.yandex.canvas.model.video.addition.RtbStatus;
import ru.yandex.canvas.model.video.addition.options.AdditionElementOptions;
import ru.yandex.canvas.model.video.files.AudioSource;
import ru.yandex.canvas.model.video.files.FileType;
import ru.yandex.canvas.model.video.files.Movie;
import ru.yandex.canvas.model.video.files.PackShot;
import ru.yandex.canvas.model.video.files.StreamFormat;
import ru.yandex.canvas.repository.ItemsWithTotal;
import ru.yandex.canvas.repository.video.VideoAdditionsRepository;
import ru.yandex.canvas.service.DateTimeService;
import ru.yandex.canvas.service.DirectService;
import ru.yandex.canvas.service.OnCreativeService;
import ru.yandex.canvas.service.SequenceService;
import ru.yandex.canvas.service.TankerKeySet;
import ru.yandex.canvas.service.screenshooters.VideoAdditionScreenshooterHelperService;
import ru.yandex.canvas.service.video.overlay.OverlayService;
import ru.yandex.canvas.service.video.presets.PresetTag;
import ru.yandex.canvas.service.video.presets.VideoPreset;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.screenshooter.client.model.ScreenShooterScreenshot;

import static java.lang.Integer.min;
import static java.lang.Math.floor;
import static ru.yandex.canvas.VideoConstants.AUDIO_CREATIVE_PREVIEW_URL;
import static ru.yandex.canvas.VideoConstants.DEFAULT_VPAID_PCODE_URL;
import static ru.yandex.canvas.VideoConstants.INTERACTIVE_VPAID_PCODE_URL;
import static ru.yandex.canvas.VideoConstants.INTERACTIVE_VPAID_THEME;
import static ru.yandex.canvas.VideoConstants.VIDEO_AD_SYSTEM_CPC;
import static ru.yandex.canvas.VideoConstants.VIDEO_EMPTY_THEME;
import static ru.yandex.canvas.VideoConstants.VIDEO_SURVEY_PCODE_URL;
import static ru.yandex.canvas.VideoConstants.VIDEO_SURVEY_THEME;
import static ru.yandex.canvas.service.video.VAST.SURVEY_CONFIG_MACROS;
import static ru.yandex.direct.utils.CommonUtils.nvl;

public class VideoAdditionsService implements OnCreativeService<Addition> {
    private static final Logger logger = LoggerFactory.getLogger(VideoAdditionsService.class);
    public static final String FLV_MIMETYPE = "video/x-flv";

    private final SequenceService sequenceService;
    private final VideoAdditionsRepository additionsRepository;
    private final String canvasPreviewHost;
    private final VideoPresetsService presetsService;
    private final MovieServiceInterface movieService;
    private final PackshotServiceInterface packshotService;
    private final AudioService audioService;
    private final DirectService directService;
    private final DateTimeService dateTimeService;
    private final VideoAdditionScreenshooterHelperService videoAdditionScreenshooterHelperService;
    private final VideoPreviewUrlBuilder videoPreviewUrlBuilder;
    private final TankerKeySet videoValidationMessages = TankerKeySet.VIDEO_VALIDATION_MESSAGES;

    public VideoAdditionsService(SequenceService sequenceService,
                                 VideoAdditionsRepository additionsRepository, VideoPresetsService videoPresetsService,
                                 String canvasPreviewHost, MovieServiceInterface movieService,
                                 PackshotServiceInterface packshotService, DirectService directService,
                                 DateTimeService dateTimeService,
                                 AudioService audioService,
                                 VideoAdditionScreenshooterHelperService videoAdditionScreenshooterHelperService,
                                 VideoPreviewUrlBuilder videoPreviewUrlBuilder) {
        this.sequenceService = sequenceService;
        this.additionsRepository = additionsRepository;
        this.canvasPreviewHost = canvasPreviewHost;
        this.presetsService = videoPresetsService;
        this.movieService = movieService;
        this.packshotService = packshotService;
        this.directService = directService;
        this.dateTimeService = dateTimeService;
        this.audioService = audioService;
        this.videoAdditionScreenshooterHelperService = videoAdditionScreenshooterHelperService;
        this.videoPreviewUrlBuilder = videoPreviewUrlBuilder;
    }

    public Long getNextAdditionId() {
        List<Long> nextCreativeIdsList = sequenceService.getNextCreativeIdsList(1);
        return nextCreativeIdsList.get(0);
    }

    public void createAdditionWithScreenshot(Long clientId, Addition addition,
                                             CustomVastParams customVastParams) throws IOException, URISyntaxException {
        VideoPreset preset = presetsService.getPreset(addition.getPresetId());
        if (preset.getDescription().getInteractiveVpaid()) {
            addition.setVpaidPcodeUrl(INTERACTIVE_VPAID_PCODE_URL);
        }
        createAddition(clientId, addition, getNextAdditionId(), customVastParams);

        if (preset.getDescription().getCpmAudio()) {
            String audioPreviewUrl = getAudioPreviewUrl(addition);
            addition.setPreviewUrl(audioPreviewUrl);
            addition.setScreenshotUrl(audioPreviewUrl);
            addition.setScreenshotIsDone(true);
            return;
        }

        ScreenShooterScreenshot screenshot = videoAdditionScreenshooterHelperService.getScreenshot(
                addition, null, clientId);

        if (screenshot.getIsDone()) {
            addition.setScreenshotUrl(screenshot.getUrl());
        } else {
            addition.setScreenshotUrl(videoPreviewUrlBuilder.buildScreenshotUrl(clientId, addition.getCreativeId()));
        }

        addition.setScreenshotIsDone(screenshot.getIsDone());
    }

    private String getAudioPreviewUrl(Addition addition) {
        AdditionElementOptions options = addition.findFilesOptions();
        if (options.getPackshotId() != null) {
            PackShot packshot = packshotService.lookupPackshot(options.getPackshotId(), addition.getClientId());
            if (packshot != null) {
                return packshot.getFormats().stream()
                        .filter(e -> e.getSize().equals("optimize"))
                        .map(e -> e.getUrl())
                        .findFirst().orElse(AUDIO_CREATIVE_PREVIEW_URL);
            } else {
                throw new NotFoundException("Packshot with id " + options.getPackshotId() + " not found");
            }
        }
        return AUDIO_CREATIVE_PREVIEW_URL;
    }

    public void createAddition(Long clientId, Addition addition, Long creativeId, CustomVastParams customVastParams) {
        addition.setClientId(clientId);

        addition.setArchive(false);

        Date currentDate = dateTimeService.getCurrentDate();
        if (addition.getCreationTime() == null) {
            addition.setCreationTime(currentDate);
        }
        if (addition.getStatusRtb() == null) {
            addition.setStatusRtb(RtbStatus.NEW);
        }
        if (addition.getDate() == null) {
            addition.setDate(currentDate);
        }

        addition.setCreativeId(creativeId);

        // TODO: возможно это не всегда так, возможно надо искать это id оп паре video_id / audio_id
        addition.setStockCreativeId(addition.getCreativeId());

        try {
            addition.setVast(makeVastForAddition(addition, customVastParams).getParameterizedXml());
        } catch (IOException | URISyntaxException e) {
            logger.warn("makeVastForAddition", e);
            throw new InternalServerError();
        }
    }

    public VAST makeVastForAddition(Addition addition, CustomVastParams customVastParams) throws IOException,
            URISyntaxException {
        Long clientId = addition.getClientId();
        Set<String> features = directService.getFeatures(clientId, null);
        AdditionElementOptions options = addition.findFilesOptions();

        VideoPreset preset = presetsService.getPreset(addition.getPresetId());
        var presetDescription = preset.getDescription();
        var vpaidPcodeUrl = addition.getVpaidPcodeUrl() != null
                ? addition.getVpaidPcodeUrl()
                : customVastParams.getVpaidPcodeUrl();
        var theme = addition.getData().getBundle().getName();
        //DIRECT-144335 для interactive_viewer.js
        // проставить поле theme со значением video-banner_interactive-viewer
        if (INTERACTIVE_VPAID_PCODE_URL.equals(vpaidPcodeUrl)) {
            theme = INTERACTIVE_VPAID_THEME;
        }
        VAST.Builder vastBuilder = VAST.builder()
                .setCreativeId(addition.getCreativeId())
                .setTheme(theme)
                .setCountdown(preset.isControlsAllowed() ? 1L : 0L)
                .setHasAbuseButton(preset.isControlsAllowed())
                .setHasSkipButton(preset.isControlsAllowed())
                .setAdLabel(preset.isControlsAllowed() ? 1L : 0L)
                .setElements(addition.getData().getElements())
                .setVpaidPcodeUrl(vpaidPcodeUrl)
                .setPythia(getPythia(addition))
                .setSkipUrl(addition.getSkipUrl())
                .setAddPixelImpression(preset.getAddImpressionPixel())
                .setSoundBtn(preset.isControlsAllowed() ? 1L : 0L)
                .setShowVideoClicks(true)
                .setShowVpaid(!presetDescription.getSkipVpaid())
                .setUseVpaidImpressions(preset.getUseVpaidImpressions())
                .setSocialAdvertisement(features.contains(FeatureName.SOCIAL_ADVERTISING.getName()))
                .setAuctionRenderUrlMacros(features.contains(FeatureName.CANVAS_AUCTION_RENDER_URL_MACROS.getName()))
                .setUseTrackingEvents(preset.getUseTrackingEvents());

        if (// DIRECT-138450 для креативов с темой empty и ссылкой vpaid-creative.js заменяем ссылку
                VIDEO_EMPTY_THEME.equals(theme) && DEFAULT_VPAID_PCODE_URL.equals(vpaidPcodeUrl)
            //DIRECT-148292: Поменять ссылки на лоадер креатива video-banner_survey в вастах на VpaidPlayer
                || VIDEO_SURVEY_THEME.equals(theme) && VIDEO_SURVEY_PCODE_URL.equals(vpaidPcodeUrl)) {
            vastBuilder.setVpaidPcodeUrl(INTERACTIVE_VPAID_PCODE_URL);
        }

        if (customVastParams.getCustomTemplateName() != null) {
            vastBuilder.setCustomTemplateName(customVastParams.getCustomTemplateName());
        }

        Set<PresetTag> presetTags = preset.getTags();
        if (presetTags.contains(PresetTag.COMMON) ||
            presetTags.contains(PresetTag.MOBILE_CONTENT) ||
            presetTags.contains(PresetTag.MOBILE_CONTENT_VIDEO)) {
            vastBuilder.setCustomAdSystem(VIDEO_AD_SYSTEM_CPC);
        }

        if (options.getPackshotId() != null) {
            PackShot packshot = packshotService.lookupPackshot(options.getPackshotId(), clientId);
            if (packshot != null) {
                vastBuilder.setPackshotUrl(PackshotUtils.getPackshotUrl(packshot));
            } else {
                throw new NotFoundException("Packshot with id " + options.getPackshotId() + " not found");
            }
        }
        long duration = 0;
        if (presetDescription.getCpmAudio()) {
            String audioId = options.getAudioId();
            AudioSource audioSource = audioService.lookupAudio(audioId, clientId);

            if (audioSource != null) {
                duration = audioSource.getDuration();
                vastBuilder.setDuration(new TimeDelta(duration))
                        .setMediaFiles(MediaFilesUtils.getMediaFiles(audioSource))
                        .setIcon(vastBuilder.getPackshotUrl());
                // DIRECT-100073 Когда нет картинки, на площадке заглушка. В превью тоже нужно показывать заглушку.
                if (vastBuilder.getPackshotUrl() == null) {
                    vastBuilder.setIcon(AUDIO_CREATIVE_PREVIEW_URL);
                }
            } else { // для превью могли ещё настоящую дорожку не загрузить
                vastBuilder.setDuration(new TimeDelta(0));
            }
        } else {
            Movie movie = movieService.lookupMovie(options.getVideoId(), options.getAudioId(), clientId,
                    addition.getPresetId());
            if (movie == null) {
                // для уже существующего в БД видое креатива может случаться если требования к видефайлу
                // менялись после заливки этого креатива
                throw new NotFoundException(videoValidationMessages.key("video_requirements_changed"));
            }
            duration = (long) floor(movie.getDuration() + 0.5);
            vastBuilder.setStrmPrefix(movie.getStrmPrefix())
                    .setMediaFiles(getMediaFiles(movie, clientId))
                    .setInjectMediaFilesMacros(!features.contains(FeatureName.CANVAS_VIDEO_MEDIAFILES_MACROS.getName()))
                    .setVideoMetaId(movie.getVideoSource().getVideoMetaId())
                    .setPlayerId(movie.getVideoSource().getPlayerId())
                    .setEstKaltura(movie.getEstKaltura())
                    .setEstMobileVertical(movie.getEstMobileVertical())
                    .setFirstFrame(createFirstFrameParameters(movie))
                    .setDuration(new TimeDelta(duration))
                    .setIsStock(movie.isStock());
        }

        TimeDelta skipOffset = SkipOffsetUtils.getSkipOffset(preset, features, duration);
        vastBuilder.setSkipOffset(skipOffset);
        return vastBuilder.build();
    }

    private PythiaParams getPythia(Addition addition) {
        PythiaParams pythiaParams = addition.getPythia();
        if (pythiaParams != null) {
            pythiaParams.setSurveyConfig(SURVEY_CONFIG_MACROS);
        }
        return pythiaParams;
    }

    private AdParams.FirstFrameParameters createFirstFrameParameters(Movie movie) {
        var src = movie.getVideoSource();
        if (src.getFirstFrameUrl() == null) {
            return null;
        }
        var param = new AdParams.FirstFrameParameters();
        AdParams.Images img = new AdParams.Images();
        img.url = src.getFirstFrameUrl();
        img.height = src.getHeight();
        img.width = src.getWidth();
        param.images = List.of(img);
        return param;
    }

    private List<StreamFormat> getMediaFiles(@NotNull Movie movie, Long clientId) {
        if (movie.getFormats() != null && !movie.getFormats().isEmpty()) {
            return StreamEx.of(movie.getFormats())
                    .filter(it -> !nvl(it.getMimeType(), "").equals(FLV_MIMETYPE))
                    .toList();
        }
        //если попали сюда, то конвертация файла не завершена и нет formats
        String videoId = movie.getVideoSource().getId();
        VideoFiles videoFile = movieService.getFileById(videoId, FileType.VIDEO, clientId);
        StillageFileInfo info = videoFile.getStillageFileInfo();
        VideoMetaData videoMetaData = new ObjectMapper().convertValue(info.getMetadataInfo(), VideoMetaData.class);
        if (videoMetaData.getVideoStreams() != null && !videoMetaData.getVideoStreams().isEmpty()) {
            VideoMetaData.VideoStreamInfo streamInfo = videoMetaData.getVideoStreams().get(0);
            StreamFormat media = new StreamFormat("progressive", streamInfo.getWidth(),
                    HttpUtils.makeURI(videoFile.getUrl()).toASCIIString(), info.getMimeType(),
                    streamInfo.getHeight(), videoId, (long) info.getFileSize(),
                    videoMetaData.getAudioStreams() != null && !videoMetaData.getAudioStreams().isEmpty());
            return ImmutableList.of(media);
        }
        throw new NotFoundException();
    }

    public void saveAddition(Addition addition) {
        AdditionElementOptions options = addition.findFilesOptions();
        VideoPreset preset = presetsService.getPreset(addition.getPresetId());

        if (preset.getDescription().getCpmAudio()) {
            AudioSource audioSource = audioService.lookupAudio(options.getAudioId(), addition.getClientId());
            String markId = audioService.markFileUsed(audioSource, addition.getClientId());
            options.setAudioId(markId);
        } else {
            Movie movie = movieService.lookupMovie(options.getVideoId(), options.getAudioId(), addition.getClientId(),
                    addition.getPresetId());

            MovieService.MarkedMovieIds newIds = movieService.markFileUsed(movie, addition.getClientId());

            options.setVideoId(newIds.getVideoMarkId());
            options.setAudioId(newIds.getAudioMarkId());
        }

        PackShot packshot = packshotService.lookupPackshot(options.getPackshotId(), addition.getClientId());

        if (packshot != null) {
            options.setPackshotId(packshotService.markFileUsed(packshot, addition.getClientId()));
        }

        additionsRepository.createAddition(addition);
    }

    public void afterAdditionCreate(Addition addition) {
        VideoPreset preset = presetsService.getPreset(addition.getPresetId());

        if (preset.getDescription().getCpmAudio()) {
            AdditionElementOptions options = addition.findFilesOptions();
            AudioSource audioSource = audioService.lookupAudio(options.getAudioId(), addition.getClientId());
            //если нужно конвертировать, то запустить задачу в sandbox
            if (audioSource.isNotReady()) {
                audioService.startConverting(audioSource);
            }
        }
    }

    public String getPreviewUrl(String additionId, Boolean isCompact, boolean isOverlay) {
        if (isOverlay) {
            return OverlayService.OVERLAY_PREVIEW_URL;
        }

        StringBuilder previewUrlBuilder =
                new StringBuilder("https://").append(canvasPreviewHost).append("/video-additions/")
                        .append(additionId).append("/preview");

        if (isCompact) {
            previewUrlBuilder.append("?compact=1");
        }

        return previewUrlBuilder.toString();
    }

    public boolean renameAddition(Long clientId, String additionId, String newName) {

        VideoAdditionsRepository.QueryBuilder queryBuilder = new VideoAdditionsRepository.QueryBuilder()
                .withArchive(false)
                .withId(additionId)
                .withClientId(clientId);

        VideoAdditionsRepository.UpdateBuilder updateBuilder = new VideoAdditionsRepository.UpdateBuilder()
                .withName(newName);

        return additionsRepository.update(queryBuilder, updateBuilder).getModifiedCount() > 0;
    }

    public Addition getAddition(Long clientId, String additionId) {
        return additionsRepository.findByIdAndQuery(additionId,
                new VideoAdditionsRepository.QueryBuilder().withClientId(clientId).withArchive(false));
    }

    public Addition getAdditionByIdArchivedAlso(String additionId) {
        return additionsRepository.getAdditionByIdArchivedAlso(additionId);
    }

    public Addition getAddition(String additionId) {
        return additionsRepository
                .getAdditionById(additionId);
    }

    public Addition getAdditionByCreativeId(Long creativeId) {
        return additionsRepository.findByQuery(new VideoAdditionsRepository.QueryBuilder()
                .withArchive(false)
                .withCreativeId(creativeId)).get(0);
    }

    public Addition getAdditionByCreativeIdArchivedAlso(Long creativeId) {
        return additionsRepository.findByQuery(new VideoAdditionsRepository.QueryBuilder()
                .withCreativeId(creativeId)).get(0);
    }

    public Addition getAdditionByCreativeId(Long creativeId, Long clientId) {
        return additionsRepository.findByQuery(new VideoAdditionsRepository.QueryBuilder()
                .withArchive(false)
                .withClientId(clientId)
                .withCreativeId(creativeId)).get(0);
    }

    public List<Addition> getAdditionsByCreativeIds(List<Long> creativeIds) {
        if (CollectionUtils.isEmpty(creativeIds)) {
            return Collections.emptyList();
        }

        return additionsRepository.findByQuery(new VideoAdditionsRepository.QueryBuilder()
                .withArchive(false)
                .withCreativeIds(creativeIds));
    }

    public List<Addition> getAdditionsByCreativeIdsArchivedAlso(List<Long> creativeIds) {
        if (CollectionUtils.isEmpty(creativeIds)) {
            return Collections.emptyList();
        }

        return additionsRepository.findByQuery(new VideoAdditionsRepository.QueryBuilder()
                .withCreativeIds(creativeIds));
    }

    public ItemsWithTotal<Addition> getAdditions(
            Long userId,
            Long clientId,
            Integer limit,
            Integer offset,
            Sort.Direction sortOrder,
            Boolean archive,
            String name,
            Boolean isCompact,
            List<Long> properPresetsList,
            Integer scrollTo,
            boolean tieCampaigns) {
        VideoAdditionsRepository.QueryBuilder queryBuilder = new VideoAdditionsRepository.QueryBuilder()
                .withClientId(clientId)
                .withArchive(archive)
                .withPresetIds(properPresetsList);

        if (!name.isEmpty()) {
            queryBuilder.withNameRegexp(".*" + Pattern.quote(name) + ".*", "i");
        }

        Function<List<Addition>, Set<String>> prepare = null;
        BiFunction<Addition, Set<String>, Boolean> filter = null;
        Predicate<List<Addition>> isDone = null;

        if (tieCampaigns) {
            prepare = list -> getTiedAdditions(userId, clientId, list);
            filter = (addition, prepared) -> prepared.contains(addition.getId());
        }

        final boolean[] scrolled = {false};

        if (scrollTo != null) {
            BiFunction<Addition, Set<String>, Boolean> finalFilter = filter;

            filter = (addition, prepared) -> {

                if (finalFilter != null && !finalFilter.apply(addition, prepared)) {
                    return false;
                }

                return scrollToFilter(addition, scrolled, scrollTo);
            };

            isDone = e -> scrolled[0] && e.size() >= limit;
        }

        ItemsWithTotal<Addition> additionsWithTotal = additionsRepository
                .getVideoAdditionsWithTotalByQuery(queryBuilder, limit, offset, sortOrder, prepare, filter, isDone);

        if (scrollTo != null && !scrolled[0]) {
            additionsWithTotal = new ItemsWithTotal<>(
                    additionsWithTotal.getItems().subList(0, min(limit, additionsWithTotal.getItems().size())),
                    additionsWithTotal.getTotal(), additionsWithTotal.getRealOffset());
        }

        for (Addition addition : additionsWithTotal.getItems()) {
            addition.setPreviewUrl(videoPreviewUrlBuilder.getPreviewUrl(clientId, addition, isCompact));
        }

        return additionsWithTotal;
    }

    Set<String> getTiedAdditions(Long userId, Long clientId, List<Addition> additions) {
        List<Long> creativeIds = additions.stream().map(Addition::getCreativeId).collect(Collectors.toList());

        Map<Long, List<CreativeCampaignResult>> creativesCampaigns =
                directService.getCreativesCampaigns(creativeIds, userId, clientId);

        return additions.stream().filter(a -> creativesCampaigns.containsKey(a.getCreativeId()))
                .map(Addition::getId)
                .collect(Collectors.toSet());
    }

    private static boolean scrollToFilter(Addition addition, boolean[] scrolled, int scrollTo) {

        if (addition.getCreativeId().equals((long) scrollTo)) {
            scrolled[0] = true;
        }

        return true;
    }

    public String getPreview(String id, PreviewData previewData) throws IOException {
        Addition addition = additionsRepository.getAdditionByIdArchivedAlso(id);

        if (addition == null) {
            return null;
        }

        return getPreview(addition, previewData);
    }

    public String getPreview(Addition addition, PreviewData previewData) throws IOException {
        previewData.setButtonUrl(previewData.getUrl());

        DCParams dcParams = new DCParams(previewData, addition.getData(), false);

        String vast = addition.getVast();
        String customPreviewTemplateName =
                presetsService.getPreset(addition.getPresetId()).getDescription().getCustomPreviewTemplateName();
        if (customPreviewTemplateName != null) {
            try {
                vast = makeVastForAddition(addition,
                        new CustomVastParams().setCustomTemplateName(customPreviewTemplateName))
                        .getParameterizedXml();
            } catch (URISyntaxException e) {
                throw new NotFoundException("VAST template " + customPreviewTemplateName + " not found");
            }
        }

        vast = injectMediaFiles(addition, vast);
        return new VAST(vast).injectDcParams(dcParams);
    }

    private String injectMediaFiles(Addition addition, String vast) throws JsonProcessingException {
        if (!vast.contains(AdParams.RTB_HOST_MEDIA_FILES_MACRO)) {
            return vast;//нет макроса для раскрытия
        }
        Movie movie = getMovie(addition);
        if (movie == null) {
            return vast;
        }
        return VAST.injectMediaFiles(vast, getMediaFiles(movie, addition.getClientId()));
    }

    public Movie getMovie(Addition addition) {
        AdditionElementOptions options = addition.findFilesOptions();
        return movieService.lookupMovie(options.getVideoId(), options.getAudioId(), addition.getClientId(),
                addition.getPresetId());
    }

    public PreviewResponseEntity.Ratio getRatio(Addition addition) {
        VideoPreset preset = presetsService.getPreset(addition.getPresetId());
        if (preset.getDescription().getCpmAudio()) {
            return new PreviewResponseEntity.Ratio(1, 1);
        }
        try {
            AdditionElementOptions options = addition.findFilesOptions();
            Movie movie = movieService.lookupMovie(options.getVideoId(), options.getAudioId(),
                    addition.getClientId(), addition.getPresetId());
            Ratio ratio = new Ratio(movie.getVideoSource().getRatio());
            return new PreviewResponseEntity.Ratio(ratio.getWidth(), ratio.getHeight());
        } catch (Exception e) {
            logger.warn("preview ratio read error " + e.getMessage(), e);
        }
        return null;
    }

    @Override
    public List<Long> filterPresent(List<Long> additionIds) {
        if (CollectionUtils.isEmpty(additionIds)) {
            return Collections.emptyList();
        }

        HashSet<Long> idsHashSet = new HashSet<>(additionIds);
        return getAdditionsByCreativeIds(additionIds).stream()
                .map(Addition::getCreativeId)
                .filter(id -> idsHashSet.contains(id))
                .collect(Collectors.toList());
    }

    @Override
    public List<Addition> fetchByIds(List<Long> creativeIds) {
        if (CollectionUtils.isEmpty(creativeIds)) {
            return Collections.emptyList();
        }

        return getAdditionsByCreativeIdsArchivedAlso(creativeIds);
    }

    @Override
    public Map<Long, List<Addition>> fetchForTypeByIdsGroupedByClient(List<Long> creativeIds) {
        if (CollectionUtils.isEmpty(creativeIds)) {
            return Collections.emptyMap();
        }
        return getAdditionsByCreativeIdsArchivedAlso(creativeIds).stream()
                .collect(Collectors.groupingBy(Addition::getClientId));
    }

    @Override
    public Class<Addition> worksOn() {
        return Addition.class;
    }
}
