package ru.yandex.canvas.service.video;

import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.canvas.exceptions.ValidationErrorsException;
import ru.yandex.canvas.model.stillage.StillageFileInfo;
import ru.yandex.canvas.model.video.vc.feed.VideoConstructorFeedFieldType;
import ru.yandex.canvas.service.FileValidator;
import ru.yandex.canvas.service.ImageValidator;
import ru.yandex.canvas.service.TankerKeySet;
import ru.yandex.canvas.service.VideoLimitsInterface;

@ParametersAreNonnullByDefault
public class VideoConstructorFeedItemValidator extends FileValidator {
    private final int rowNumber;
    private final String fieldName;
    private final StillageFileInfo itemInfo;
    private final VideoConstructorFeedFieldType actualType;
    private final VideoConstructorFeedFieldType expectedType;
    private final VideoMetaData videoMetaData;
    private final VideoLimitsInterface limits;
    private final Set<String> features;
    private final VideoGeometryService videoGeometryService;

    public VideoConstructorFeedItemValidator(int rowNumber,
                                             String fieldName,
                                             @Nullable StillageFileInfo itemInfo,
                                             @Nullable VideoConstructorFeedFieldType actualType,
                                             @Nullable VideoConstructorFeedFieldType expectedType,
                                             @Nullable VideoMetaData videoMetaData,
                                             @Nullable VideoLimitsInterface limits,
                                             @Nullable Set<String> features,
                                             VideoGeometryService videoGeometryService) {
        super(TankerKeySet.VIDEO_VALIDATION_MESSAGES);
        this.rowNumber = rowNumber;
        this.fieldName = fieldName;
        this.itemInfo = itemInfo;
        this.actualType = actualType;
        this.expectedType = expectedType;
        this.videoMetaData = videoMetaData;
        this.limits = limits;
        this.features = features;
        this.videoGeometryService = videoGeometryService;
    }

    @Override
    public void validate() {
        // в зависимости от типа файла создаем разные валидаторы
        final FileValidator validator;
        if (actualType == VideoConstructorFeedFieldType.IMAGE) {
            validator = new ImageValidator(itemInfo);
        } else if (actualType == VideoConstructorFeedFieldType.VIDEO) {
            validator = new MovieValidator(videoMetaData, itemInfo, limits, VideoCreativeType.VIDEO_CONSTRUCTOR_FEED,
                    features, null, videoGeometryService,
                    0L//Валидацию для конвертации пропускаем, для фидов отдельное ограничение на день
            );
        } else if (actualType == VideoConstructorFeedFieldType.AUDIO) {
            validator = new AudioValidator(itemInfo, AudioUploadService.parseMetaData(itemInfo), limits);
        } else {
            addErrorMessage("video-constructor-feed-item-invalid-format", rowNumber, fieldName);
            throwValidationErrors();
            return;
        }

        if (expectedType != null && expectedType != actualType) {
            addErrorMessage("video-constructor-feed-item-different-formats", fieldName);
            throwValidationErrors();
        }

        // Перехватываем исключения из дочерней валидации и добавляем свое
        try {
            validator.validate();
        } catch (ValidationErrorsException e) {
            addErrorMessage("video-constructor-feed-item-validation-failed", rowNumber, fieldName);
            addPlainErrorMessages(e.getMessages());
            throwValidationErrors();
        }

        throwValidationErrors();
    }
}
