package ru.yandex.canvas.service.video;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;

import javax.annotation.Nullable;

import com.google.common.collect.ImmutableSet;

import ru.yandex.canvas.model.video.vc.feed.VideoConstructorFeedParsed;
import ru.yandex.canvas.service.FileValidator;
import ru.yandex.canvas.service.TankerKeySet;

public class VideoConstructorFeedValidator extends FileValidator {
    public static final Set<String> ALLOWED_FORMATS = ImmutableSet.of("CSV in UTF-8 encoding");
    public static final int MAX_ROWS_NUMBER = 500;
    public static final int MAX_FIELDS_NUMBER = 50;

    private final VideoConstructorFeedParsed feedParsed;

    public VideoConstructorFeedValidator(@Nullable VideoConstructorFeedParsed feedParsed) {
        super(TankerKeySet.VIDEO_VALIDATION_MESSAGES);
        this.feedParsed = feedParsed;
    }

    @Override
    public void validate() {
        if (feedParsed == null) {
            addErrorMessage("video-constructor-feed-invalid-format", String.join(", ", ALLOWED_FORMATS));
            throwValidationErrors();
            return;
        }

        if (feedParsed.getFieldsCount() == 0) {
            addErrorMessage("video-constructor-feed-no-header");
        }

        if (feedParsed.getRowsCount() == 0) {
            addErrorMessage("video-constructor-feed-empty");
        }

        if (feedParsed.getFieldsCount() > MAX_FIELDS_NUMBER) {
            addErrorMessage("video-constructor-feed-too-much-fields", MAX_FIELDS_NUMBER);
        }

        if (feedParsed.getRowsCount() > MAX_ROWS_NUMBER) {
            addErrorMessage("video-constructor-feed-too-much-rows", MAX_ROWS_NUMBER);
        }

        throwValidationErrors();

        checkFieldUniqueness();

        checkUrlsCorrectness();

        throwValidationErrors();
    }

    /* Определяем, является ли колонка ссылкой или нет, только по первой строке.
       Если да, то проверяем всю колонку на корректность ссылок.
    */
    private void checkUrlsCorrectness() {
        assert feedParsed != null;

        String[] firstRow = feedParsed.getRowValues(0);
        for (int i = 0; i < feedParsed.getFieldsCount(); ++i) {
            if (firstRow[i] != null && (firstRow[i].startsWith("http://") || firstRow[i].startsWith("https://"))) {
                for (int j = 0; j < feedParsed.getRowsCount(); ++j) {
                    String[] curRow = feedParsed.getRowValues(j);
                    try {
                        new URL(curRow[i]);
                    } catch (MalformedURLException e) {
                        addErrorMessage("video-constructor-feed-invalid-url", j + 1, feedParsed.getFieldNames()[i]);
                        throwValidationErrors();
                    }
                }
            }
        }
    }

    private void checkFieldUniqueness() {
        assert feedParsed != null;

        Set<String> fieldsSet = new HashSet<>();
        Set<String> duplicateFieldsSet = new HashSet<>();

        Arrays.stream(feedParsed.getFieldNames()).forEach(fieldName -> {
            if (fieldsSet.contains(fieldName)) {
                duplicateFieldsSet.add(fieldName);
            } else {
                fieldsSet.add(fieldName);
            }
        });

        if (!duplicateFieldsSet.isEmpty()) {
            addErrorMessage("video-constructor-feed-duplicate-fields", String.join(", ", duplicateFieldsSet));
            throwValidationErrors();
        }
    }
}
