package ru.yandex.canvas.service.video;

import java.net.URI;
import java.net.URISyntaxException;
import java.sql.Date;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import com.google.common.collect.ImmutableSet;
import org.springframework.data.domain.Sort;

import ru.yandex.canvas.model.video.vc.feed.VideoConstructorFeed;
import ru.yandex.canvas.model.video.vc.files.VideoConstructorFile;
import ru.yandex.canvas.repository.video.VideoConstructorFeedsRepository;
import ru.yandex.canvas.repository.video.VideoConstructorFilesRepository;
import ru.yandex.canvas.service.FileValidator;
import ru.yandex.canvas.service.TankerKeySet;

public class VideoConstructorFileValidator extends FileValidator {
    private static final Set<String> ALLOWED_SCHEMES = ImmutableSet.of("http", "https");

    private static final Set<String> ALLOWED_DOMAINS_DIRECT = ImmutableSet.of("direct", "test-direct");
    private static final Set<String> ALLOWED_DOMAINS_DIRECT_TLD = ImmutableSet.of("ru", "net", "kz", "by", "ua", "com",
            "com.tr");
    private static final Set<String> ALLOWED_DOMAINS_PREVIEW = ALLOWED_DOMAINS_DIRECT.stream()
            .flatMap(directDomain -> ALLOWED_DOMAINS_DIRECT_TLD.stream()
                    .map(directTldDomain -> String.format("%s.yandex.%s", directDomain, directTldDomain)))
            .collect(Collectors.toUnmodifiableSet());

    private static final Set<String> ALLOWED_DOMAINS_RESOURCE = ImmutableSet.of("storage.mds.yandex.net");
    private static final int MAX_CREATED_VIDEO = 2000;

    private final VideoConstructorFeedsRepository videoConstructorFeedsRepository;
    private final VideoConstructorFilesRepository videoConstructorFilesRepository;
    private final VideoConstructorFile videoConstructorFile;

    public VideoConstructorFileValidator(VideoConstructorFeedsRepository videoConstructorFeedsRepository,
                                         VideoConstructorFilesRepository videoConstructorFilesRepository,
                                         VideoConstructorFile videoConstructorFile) {
        super(TankerKeySet.VIDEO_VALIDATION_MESSAGES);
        this.videoConstructorFeedsRepository = videoConstructorFeedsRepository;
        this.videoConstructorFilesRepository = videoConstructorFilesRepository;
        this.videoConstructorFile = videoConstructorFile;
    }

    @Override
    public void validate() {
        if (videoConstructorFile.getUrl() == null ||
                urlIsInvalid(videoConstructorFile.getUrl(), ALLOWED_DOMAINS_PREVIEW, true)) {
            addErrorMessage("video-constructor-invalid-url");
        }

        if (videoConstructorFile.getAudioUrl() != null &&
                urlIsInvalid(videoConstructorFile.getAudioUrl(), ALLOWED_DOMAINS_RESOURCE, false)) {
            addErrorMessage("video-constructor-invalid-audio-url");
        }

        if (videoConstructorFile.getFeedId() != null) {
            VideoConstructorFeed videoConstructorFeed = videoConstructorFeedsRepository.findById(
                    videoConstructorFile.getFeedId(), videoConstructorFile.getClientId());
            if (videoConstructorFeed == null) {
                addErrorMessage("video-constructor-feed-unknown");
            } else {
                int createdVideoCount = getCreatedVideoCount();
                if (createdVideoCount + videoConstructorFeed.getRowsCount() >= MAX_CREATED_VIDEO) {
                    addErrorMessage("video-constructor-feed-limit-exceeded", MAX_CREATED_VIDEO, createdVideoCount);
                }
            }
        }

        throwValidationErrors();
    }

    private boolean urlIsInvalid(String url, Set<String> allowedDomains, boolean allowSubDomains) {
        URI uri;
        try {
            uri = new URI(url);
        } catch (URISyntaxException e) {
            return true;
        }

        if (uri.getHost() == null || uri.getScheme() == null || !ALLOWED_SCHEMES.contains(uri.getScheme())) {
            return true;
        }

        String domain = uri.getHost();
        if (allowedDomains.contains(domain)) {
            return false;
        }

        if (allowSubDomains) {
            String[] parts = domain.split("\\.");
            return IntStream.range(2, 5)
                    .noneMatch(i -> parts.length >= i && allowedDomains.contains(
                            String.join(".", Arrays.copyOfRange(parts, parts.length - i, parts.length))));
        }

        return true;
    }

    private int getCreatedVideoCount() {
        // берем файлы с фидом клиента за последний день
        VideoConstructorFilesRepository.QueryBuilder queryBuilder = new VideoConstructorFilesRepository.QueryBuilder()
                .withClientId(videoConstructorFile.getClientId())
                .withFeedExists(true)
                .withDateFrom(Date.from(Instant.now().minus(1, ChronoUnit.DAYS)));

        List<VideoConstructorFile> videoConstructorFiles = videoConstructorFilesRepository.findByQuery(queryBuilder,
                Sort.Direction.ASC, null, null);

        List<String> feedIds = videoConstructorFiles.stream()
                .filter(vcFile -> vcFile.getFeedId() != null)
                .map(VideoConstructorFile::getFeedId).collect(Collectors.toList());

        List<VideoConstructorFeed> videoConstructorFeeds = videoConstructorFeedsRepository.findByIds(feedIds,
                videoConstructorFile.getClientId());
        Map<String, VideoConstructorFeed> feedId2Feed = videoConstructorFeeds.stream()
                .collect(Collectors.toMap(VideoConstructorFeed::getId, feed -> feed));

        return videoConstructorFiles.stream()
                .map(file -> feedId2Feed.get(file.getFeedId()))
                .mapToInt(VideoConstructorFeed::getRowsCount)
                .sum();
    }
}
