package ru.yandex.canvas.service.video;

import java.io.IOException;
import java.net.URISyntaxException;
import java.sql.Date;
import java.time.temporal.ChronoUnit;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.zip.CRC32;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonSerializer;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializerProvider;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import com.google.common.collect.ImmutableMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.HttpMediaTypeNotAcceptableException;

import ru.yandex.adv.direct.CreativeParameters;
import ru.yandex.adv.direct.video.VideoParameters;
import ru.yandex.canvas.exceptions.InternalServerError;
import ru.yandex.canvas.exceptions.NotFoundException;
import ru.yandex.canvas.model.direct.CreativeCampaignResult;
import ru.yandex.canvas.model.direct.CreativeUploadData;
import ru.yandex.canvas.model.direct.DirectUploadResult;
import ru.yandex.canvas.model.direct.UcCreativeCanvasData;
import ru.yandex.canvas.model.screenshooter.CreativeWithClientId;
import ru.yandex.canvas.model.video.Addition;
import ru.yandex.canvas.model.video.AudioFiles;
import ru.yandex.canvas.model.video.CustomVastParams;
import ru.yandex.canvas.model.video.VideoFiles;
import ru.yandex.canvas.model.video.addition.AdditionElement;
import ru.yandex.canvas.model.video.addition.Options;
import ru.yandex.canvas.model.video.addition.RtbStatus;
import ru.yandex.canvas.model.video.addition.options.AdditionElementOptions;
import ru.yandex.canvas.model.video.files.AudioSource;
import ru.yandex.canvas.model.video.files.FileStatus;
import ru.yandex.canvas.model.video.files.Movie;
import ru.yandex.canvas.model.video.files.PackShot;
import ru.yandex.canvas.model.video.files.StreamFormat;
import ru.yandex.canvas.model.video.files.VideoSource;
import ru.yandex.canvas.repository.video.StockVideoAdditionsRepository;
import ru.yandex.canvas.repository.video.VideoAdditionsRepository;
import ru.yandex.canvas.service.DateTimeService;
import ru.yandex.canvas.service.DirectService;
import ru.yandex.canvas.service.RTBHostExportService;
import ru.yandex.canvas.service.TankerKeySet;
import ru.yandex.canvas.service.direct.VideoAdditionDirectUploadHelper;
import ru.yandex.canvas.service.rtbhost.CreativeParametersSerializer;
import ru.yandex.canvas.service.rtbhost.RtbHostUploadHelper;
import ru.yandex.direct.bs.dspcreative.model.DspCreativeExportEntry;
import ru.yandex.direct.feature.FeatureName;

import static java.lang.Math.floor;
import static ru.yandex.canvas.VideoConstants.DEFAULT_VPAID_PCODE_URL;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class VideoCreativesService implements RtbHostUploadHelper<Addition> {
    private static final Logger logger = LoggerFactory.getLogger(VideoCreativesService.class);

    private final MovieServiceInterface movieService;
    private final AudioService audioService;
    private final PackshotServiceInterface packshotService;
    private final VideoAdditionsService videoAdditionsService;
    private final VideoAdditionsRepository videoAdditionsRepository;
    private final StockVideoAdditionsRepository stockVideoAdditionsRepository;
    private final DirectService directService;
    private final RTBHostExportService rtbHostExportService;
    private final CmsConversionStatusUpdateService cmsConversionStatusUpdateService;
    private final VideoPresetsService videoPresetsService;
    private final VideoAdditionDirectUploadHelper videoAdditionDirectUploadHelper;
    private final DateTimeService dateTimeService;
    private final TankerKeySet videoValidationMessages = TankerKeySet.VIDEO_VALIDATION_MESSAGES;

    public VideoCreativesService(MovieServiceInterface movieService, AudioService audioService,
                                 PackshotServiceInterface packshotService, VideoAdditionsService videoAdditionsService,
                                 VideoAdditionsRepository videoAdditionsRepository, DirectService directService,
                                 RTBHostExportService rtbHostExportService,
                                 StockVideoAdditionsRepository stockVideoAdditionsRepository,
                                 CmsConversionStatusUpdateService cmsConversionStatusUpdateService,
                                 VideoPresetsService videoPresetsService,
                                 VideoAdditionDirectUploadHelper videoAdditionDirectUploadHelper,
                                 DateTimeService dateTimeService) {
        this.movieService = movieService;
        this.audioService = audioService;
        this.packshotService = packshotService;
        this.videoAdditionsService = videoAdditionsService;
        this.videoAdditionsRepository = videoAdditionsRepository;
        this.stockVideoAdditionsRepository = stockVideoAdditionsRepository;
        this.directService = directService;
        this.rtbHostExportService = rtbHostExportService;
        this.cmsConversionStatusUpdateService = cmsConversionStatusUpdateService;
        this.videoPresetsService = videoPresetsService;
        this.videoAdditionDirectUploadHelper = videoAdditionDirectUploadHelper;
        this.dateTimeService = dateTimeService;
    }

    public <T extends Addition> void uploadToRtbHost(List<T> additions) {
        rtbHostExportService.exportToRtbHost(additions, this);
        videoAdditionsRepository.updateStatusRtb(mapList(additions, Addition::getCreativeId), RtbStatus.SENT);
    }

    public DirectUploadResult uploadToDirect(List<String> ids, Long userId, Long clientId) {
        List<Addition> additions = videoAdditionsRepository.findByQuery(
                new VideoAdditionsRepository.QueryBuilder()
                        .withIds(ids)
                        .withArchive(false));

        return uploadAdditionsToDirect(additions, userId, clientId);
    }

    public DirectUploadResult uploadAdditionsToDirect(List<Addition> additions, Long userId, Long clientId) {
        return directService.sendCreatives(userId, clientId, additions, videoAdditionDirectUploadHelper);
    }

    public Map<Long, CreativeUploadData> getCreatives(List<Long> parsedIds, Long clientId) {
        List<Addition> additions = videoAdditionsRepository.
                findByQuery(
                        new VideoAdditionsRepository.QueryBuilder()
                                .withCreativeIds(parsedIds)
                                .withClientId(clientId)
                                .withArchive(false));

        return additions.stream().collect(Collectors.toMap(Addition::getCreativeId,
                creative -> videoAdditionDirectUploadHelper.toCreativeUploadData(creative, clientId)));
    }

    public Map<Long, UcCreativeCanvasData> getUcCreatives(List<Long> parsedIds, Long clientId) {
        List<Addition> additions = videoAdditionsRepository.
                findByQuery(
                        new VideoAdditionsRepository.QueryBuilder()
                                .withCreativeIds(parsedIds)
                                .withClientId(clientId)
                                .withArchive(false));

        return additions.stream()
                .map(videoAdditionDirectUploadHelper::toUcCreativeCanvasData)
                .filter(Objects::nonNull)
                .collect(Collectors.toMap(UcCreativeCanvasData::getId, Function.identity()));
    }


    public CreativeWithClientId<Addition> getCreativeWithClientIdOrThrow(Long creativeId) {
        Addition addition = videoAdditionsService.getAdditionByCreativeIdArchivedAlso(creativeId);

        if (addition == null) {
            throw new NotFoundException();
        }

        return new CreativeWithClientId<>(addition, addition.getClientId());
    }

    public static class RtbElementsSerializer extends JsonSerializer<List<AdditionElement>> {

        private static final Map<String, Function<Options, String>> fmap = ImmutableMap.of(
                "text", Options::getText,
                "backgroundColor", Options::getBackgroundColor,
                "color", Options::getColor,
                "textColor", Options::getTextColor,
                "borderColor", Options::getBorderColor
        );

        @Override
        public void serialize(List<AdditionElement> value, JsonGenerator gen, SerializerProvider serializers)
                throws IOException {
            gen.writeStartArray();

            for (AdditionElement element : value) {
                gen.writeStartObject();
                Options options = element.getOptions();

                gen.writeStringField("type", element.getType().name().toLowerCase());

                for (Map.Entry<String, Function<Options, String>> entry : fmap.entrySet()) {
                    String fieldValue = entry.getValue().apply(options);

                    if (fieldValue != null && !fieldValue.equals("")) {
                        gen.writeStringField(entry.getKey(), fieldValue);
                    }
                }

                gen.writeEndObject();
            }

            gen.writeEndArray();
        }
    }


    public static class VideoCreativeData {
        @JsonProperty("template")
        String template;

        @JsonProperty("underlayerID")
        long underlayerID;

        @JsonProperty("duration")
        long duration;

        @JsonProperty("isStock")
        boolean isStock;

        @JsonProperty("hasPackshot")
        boolean hasPackshot;

        @JsonProperty("playbackParameters")
        AdParams.PlaybackParameters playbackParameters;

        @JsonProperty("elements")
        @JsonSerialize(using = RtbElementsSerializer.class)
        List<AdditionElement> elements;

        @JsonProperty("formats")
        List<VideoCreativeFormat> formats;

        @JsonInclude(JsonInclude.Include.NON_NULL)
        @JsonProperty
        private Boolean isSocialAdvertising;

        @JsonInclude(JsonInclude.Include.NON_NULL)
        @JsonProperty("creative_parameters")
        @JsonSerialize(using = CreativeParametersSerializer.class)
        private CreativeParameters creativeParameters;

        public static class VideoCreativeFormat {
            @JsonProperty("height")
            private String height;

            @JsonProperty("width")
            private String width;

            @JsonProperty("type")
            private String mimeType;

            @JsonProperty("url")
            private String url;

            public VideoCreativeFormat(StreamFormat streamFormat) {
                this.height = streamFormat.getHeight() == null ? null : streamFormat.getHeight().toString();
                this.width = streamFormat.getWidth() == null ? null : streamFormat.getWidth().toString();
                this.mimeType = streamFormat.getMimeType();
                this.url = streamFormat.getUrl();
            }
        }

        public void setFormats(
                List<VideoCreativeFormat> formats) {
            this.formats = formats;
        }

        public VideoCreativeData setUnderlayerID(long underlayerID) {
            this.underlayerID = underlayerID;
            return this;
        }

        public VideoCreativeData setDuration(long duration) {
            this.duration = duration;
            return this;
        }

        public VideoCreativeData setElements(List<AdditionElement> elements) {
            this.elements = elements;
            return this;
        }

        public VideoCreativeData setTemplate(String template) {
            this.template = template;
            return this;
        }

        public void setIsStock(boolean isStock) {
            this.isStock = isStock;
        }

        public void setHasPackshot(boolean hasPackshot) {
            this.hasPackshot = hasPackshot;
        }

        public void setSocialAdvertising(Boolean socialAdvertising) {
            isSocialAdvertising = socialAdvertising;
        }

        public void setPlaybackParameters(AdParams.PlaybackParameters playbackParameters) {
            this.playbackParameters = playbackParameters;
        }

        public void setVideoParameters(VideoParameters videoParameters) {
            this.creativeParameters = CreativeParameters.newBuilder()
                    .setVideo(videoParameters)
                    .build();
        }
    }

    //This code converts java's CRC32 value to python's abs(crc32(val))
    static long calculateUnderlayerId(String id) {
        CRC32 crc32 = new CRC32();
        crc32.update(id.getBytes());

        long val = crc32.getValue();

        if ((val & 0x80000000) != 0) { //It was negative number
            val |= 0xffffffff00000000L;
            return -val;
        }

        return val;
    }

    /*
     Creates rtbhost's import-dsp-creative entry
608     See https://wiki.yandex-team.ru/ekaterinagobareva/dspmonitoringqueue/#poluchaemobvjazkikreativov
     */
    @Override
    public DspCreativeExportEntry toImportDspCreativeEntry(Addition object, ObjectMapper objectMapper) {
        AdditionElementOptions options = object.findFilesOptions();
        boolean isOverlay = object.isOverlayAddition();
        boolean cpmAudio = options.getVideoId() == null && !isOverlay;

        VideoParametersBuilder videoParametersBuilder = new VideoParametersBuilder();
        videoParametersBuilder.setVpaidPcodeUrl(object.getVpaidPcodeUrl());
        videoParametersBuilder.setTheme(object.getData().getBundle().getName());
        videoParametersBuilder.setPythiaParams(object.getPythia());
        videoParametersBuilder.setCreativeId(object.getCreativeId());
        videoParametersBuilder.setShowVideoClicks(true);
        videoParametersBuilder.setSkipUrl(object.getSkipUrl());

        VideoCreativeData videoCreativeData = new VideoCreativeData();
        videoCreativeData.setTemplate(object.getData().getBundle().getName());
        var elements = object.getData().getElements().stream()
                .filter(AdditionElement::getAvailable)
                .collect(Collectors.toList());
        videoCreativeData.setElements(elements);
        videoParametersBuilder.setElements(elements);

        if (options.getPackshotId() != null) {
            PackShot packshot = packshotService.lookupPackshot(options.getPackshotId(), object.getClientId());
            if (packshot != null) {
                videoParametersBuilder.setPackshotUrl(PackshotUtils.getPackshotUrl(packshot));
            } else {
                throw new NotFoundException("Packshot with id " + options.getPackshotId() + " not found");
            }
        }

        var features = directService.getFeatures(object.getClientId(), null);

        long duration = 0;

        var size = new Size(0, 0);
        if (isOverlay) {
            // nothing special yet
        } else if (cpmAudio) {
            AudioSource audioSource = audioService.lookupAudio(options.getAudioId(), object.getClientId());
            duration = audioSource.getDuration();
            videoCreativeData.setDuration(duration);
            videoCreativeData.setUnderlayerID(calculateUnderlayerId(audioSource.getId()));

            videoParametersBuilder.setDuration((double) duration);
            videoParametersBuilder.setMediaFiles(MediaFilesUtils.getMediaFiles(audioSource));
            videoParametersBuilder.setNeedIcon(true);
        } else {
            Movie movie = movieService.lookupMovie(options.getVideoId(), options.getAudioId(), object.getClientId(),
                    object.getPresetId());
            if (movie == null) {
                // для уже существующего в БД видое креатива может случаться если требования к видеофайлу
                // менялись после заливки этого креатива
                throw new NotFoundException(videoValidationMessages.key("video_requirements_changed"));
            }

            duration = (long) floor(movie.getDuration() + 0.5);
            videoCreativeData.setDuration(duration);
            videoCreativeData
                    .setFormats(movie.getFormats().stream().map(VideoCreativeData.VideoCreativeFormat::new).collect(
                            Collectors.toList()));

            videoParametersBuilder.setDuration(movie.getDuration());
            videoParametersBuilder.setMediaFiles(movie.getFormats());
            videoParametersBuilder.setStrmPrefix(movie.getStrmPrefix());
            var src = movie.getVideoSource();
            if (src.getVideoMetaId() != null) {
                videoParametersBuilder.setVideoMetaId(String.valueOf(src.getVideoMetaId()));
            }
            if (src.getPlayerId() != null) {
                videoParametersBuilder.setPlayerId(src.getPlayerId());
            }
            videoParametersBuilder.setEstKaltura(movie.getEstKaltura());
            videoParametersBuilder.setEstMobileVertical(movie.getEstMobileVertical());
            videoParametersBuilder.setFirstFrameUrl(src.getFirstFrameUrl());
            videoParametersBuilder.setHeight(src.getHeight());
            videoParametersBuilder.setWidth(src.getWidth());
            videoParametersBuilder.setSignaturesParameters(src.getSignaturesUrl());

            boolean rtbVideoSize = features.contains("rtb_video_size");

            if (rtbVideoSize) {
                size = getMaxVideoSize(videoCreativeData.formats);
            }

            var isMovieStock = movie.isStock();
            videoCreativeData.setIsStock(isMovieStock);
            videoParametersBuilder.setIsStock(isMovieStock);
            String videoSourceId;
            if (isMovieStock) {
                videoSourceId = movie.getVideoSource().getStockId();
            } else {
                videoSourceId = movie.getVideoSource().getId();
            }
            videoCreativeData.setUnderlayerID(calculateUnderlayerId(videoSourceId));
            videoCreativeData.setHasPackshot(options.getPackshotId() != null);
            videoCreativeData.setSocialAdvertising(features.contains(FeatureName.SOCIAL_ADVERTISING.getName()) ? true
                    : null);
            videoParametersBuilder.setSocialAdvertisement(features.contains(FeatureName.SOCIAL_ADVERTISING.getName()));
        }

        if (object.getPresetId() != null && videoPresetsService.contains(object.getPresetId())) {
            var preset = videoPresetsService.getPreset(object.getPresetId());
            var showSkipButton = preset.isControlsAllowed();
            var timeDelta = SkipOffsetUtils.getSkipOffset(preset, features, duration);
            Long skipDelay = timeDelta == null ? null : (long) timeDelta.asSeconds();
            var playbackParameters = new AdParams.PlaybackParameters();
            playbackParameters.showSkipButton = skipDelay != null && showSkipButton;
            playbackParameters.skipDelay = skipDelay == null || skipDelay == 0 || !showSkipButton ? null : skipDelay;
            videoCreativeData.setPlaybackParameters(playbackParameters);

            videoParametersBuilder.setPlaybackParameters(showSkipButton, SkipOffsetUtils.getSkipOffset(preset,
                    features, duration));
            videoParametersBuilder.fillPresetParameters(preset);
        }

        videoCreativeData.setVideoParameters(videoParametersBuilder.build());

        try {
            return DspCreativeExportEntry.builder()
                    .setDspId(1L)
                    .setCreativeId(object.getCreativeId())
                    .setCreativeVersionId(object.getCreativeId())
                    .setData(object.getVast())
                    .setConstructorData(objectMapper.writeValueAsString(videoCreativeData))
                    .setEnabled(true)
                    .setVideo(!cpmAudio)
                    .setAudio(cpmAudio)
                    .setStaticData(String.format("{\"creative_id\":\"%d\"}", object.getCreativeId()))
                    .setPostmoderated(true)
                    .setStatic(false)
                    .setTag("yabs")
                    .setWidth(size.getWidth())
                    .setHeight(size.getHeight())
                    .build();
        } catch (JsonProcessingException e) {
            throw new RuntimeException(e);
        }
    }

    private static Size getMaxVideoSize(List<VideoCreativeData.VideoCreativeFormat> formats) {
        return formats.stream()
                .filter(it -> it.width != null && it.height != null)
                .map(it -> new Size(Integer.parseInt(it.width), Integer.parseInt(it.height)))
                .max(comparatorLargestWidthVideoFormat())
                .orElse(new Size(0, 0));
    }

    protected static Comparator<Size> comparatorLargestWidthVideoFormat() {
        return Comparator.comparingInt(it -> it.width);
    }

    public static class Size {
        private final Integer width;
        private final Integer height;

        public Size(Integer width, Integer height) {
            this.width = width;
            this.height = height;
        }

        public Integer getWidth() {
            return width;
        }

        public Integer getHeight() {
            return height;
        }
    }

    public Addition patchStockAdditionVast(Addition addition) throws HttpMediaTypeNotAcceptableException, IOException,
            URISyntaxException {
        if (addition == null) {
            throw new NotFoundException(""); //"Not found %s (id=%s)" % (collection_name, addition_id)
        }

        if (!addition.getCreativeId().equals(addition.getStockCreativeId())) {
            throw new HttpMediaTypeNotAcceptableException("");
        }

        String vast = makeVastForAddition(addition);
        stockVideoAdditionsRepository.updateStockVastById(addition.getId(), vast);
        videoAdditionsRepository.updateVastByCreativeId(addition.getStockCreativeId(), vast);
        addition.setVast(vast);
        return addition;
    }

    public Addition patchAdditionVast(Addition addition) throws HttpMediaTypeNotAcceptableException, IOException,
            URISyntaxException {
        if (addition == null) {
            throw new NotFoundException(""); //"Not found %s (id=%s)" % (collection_name, addition_id)
        }

        if (!addition.getCreativeId().equals(addition.getStockCreativeId())) {
            throw new HttpMediaTypeNotAcceptableException("");
        }

        String vast = makeVastForAddition(addition);
        videoAdditionsRepository.updateVastById(addition.getId(), vast);
        addition.setVast(vast);
        return addition;
    }

    public Addition patchStockAdditionVastById(String id) throws HttpMediaTypeNotAcceptableException, IOException,
            URISyntaxException {
        Addition addition = stockVideoAdditionsRepository.getStockAdditionById(id);
        if (addition == null) {
            throw new NotFoundException(""); //"Not found %s (id=%s)" % (collection_name, addition_id)
        }
        return patchStockAdditionVast(addition);
    }

    public Addition patchAdditionVastById(String id) throws HttpMediaTypeNotAcceptableException, IOException,
            URISyntaxException {
        Addition addition = videoAdditionsRepository.getAdditionById(id);
        if (addition == null) {
            throw new NotFoundException(""); //"Not found %s (id=%s)" % (collection_name, addition_id)
        }
        return patchAdditionVast(addition);
    }

    public String makeVastForAddition(Addition addition) throws IOException, URISyntaxException {
        return videoAdditionsService.makeVastForAddition(addition, new CustomVastParams()
                        .setVpaidPcodeUrl(nvl(addition.getVpaidPcodeUrl(), DEFAULT_VPAID_PCODE_URL)))
                .getParameterizedXml();
    }

    public List<CreativeCampaignResult> getCampaigns(String additionId, Long uid, Long clientId) {
        Addition addition = videoAdditionsRepository.getAdditionById(additionId);

        if (addition == null) {
            logger.warn("No addition with id " + additionId);
            throw new NotFoundException();
        }

        Map<Long, List<CreativeCampaignResult>> result =
                directService.getCreativesCampaigns(Collections.singletonList(addition.getCreativeId()), uid, clientId);

        if (!result.containsKey(addition.getCreativeId())) {
            logger.warn("No creative with id " + addition.getCreativeId());
            throw new InternalServerError();
        }

        return result.get(addition.getCreativeId());
    }

    /**
     * для аудиокреативов с этим файлом нужно пересоздать VAST и переотправить всё в RTBhost
     */
    public void patchCpmAudioAddition(AudioFiles file) {
        List<Addition> additions = videoAdditionsRepository.findByQuery(
                new VideoAdditionsRepository.QueryBuilder()
                        .withClientId(file.getClientId())
                        .withAudioId(file.getId()));
        for (Addition addition : additions) {
            AdditionElementOptions options = addition.findFilesOptions();
            AudioSource audioSource = audioService.lookupAudio(options.getAudioId(), addition.getClientId());
            if (audioSource != null && audioSource.getId().equals(file.getId())) {
                patchAddition(addition, false);
            }
        }
    }

    public void patchVideoAddition(String videoFileId) {
        VideoFiles videoFile = movieService.getFileByIdInternal(videoFileId);
        // данные из сэндобкса приходят за несколько раз с разными статусами,
        // обновляем креативы только тогда, когда все готово
        if (videoFile.getStatus() == FileStatus.READY) {
            patchVideoAddition(videoFile);
        }
    }

    /**
     * для видеокреативов с этим файлом нужно пересоздать VAST и переотправить всё в RTBhost
     */
    public void patchVideoAddition(VideoFiles file) {
        List<Addition> additions = videoAdditionsRepository.findByQuery(
                new VideoAdditionsRepository.QueryBuilder()
                        .withClientId(file.getClientId())
                        .withVideoId(file.getId()));

        for (Addition addition : additions) {
            AdditionElementOptions options = addition.findFilesOptions();
            Movie movie = movieService.lookupMovie(options.getVideoId(), options.getAudioId(), addition.getClientId(),
                    addition.getPresetId());
            VideoSource videoSource = movie.getVideoSource();

            if (videoSource != null && videoSource.getId().equals(file.getId())) {
                patchAddition(addition, true);
            }
        }
    }

    private void patchAddition(Addition addition, boolean resendToDirect) {
        try {
            Addition patchedAddition = patchAdditionVastById(addition.getId());
            // переотправка в директ
            if (resendToDirect) {
                uploadAdditionsToDirect(Collections.singletonList(addition), null, addition.getClientId());
            }
            // переотправка в RTBhost
            uploadToRtbHost(Collections.singletonList(patchedAddition));
        } catch (Exception e) {
            logger.error("patchAddition failed, addition id {}", addition.getId());
            throw new InternalServerError(e);
        }
    }

    /**
     * Проверяем и чиним подвисшие конвертации CMS.
     *
     * @param ageSeconds
     */
    public void cleanupCmsConversions(long ageSeconds) {
        //Находим файлы, где конвертация ещё идёт и файл старше ageSeconds
        for (VideoFiles file : movieService.findOldConvertingCmsFiles(ageSeconds)) {
            if (cmsConversionStatusUpdateService.updateStatus(file).getStatus() == FileStatus.READY) {
                patchVideoAddition(file);//Если новый статус READY, то переотправить в RTB
            }
        }

        //Находим креативы, которые ждут отправки в RTB
        List<Addition> additions = videoAdditionsRepository.findByQuery(
                new VideoAdditionsRepository.QueryBuilder()
                        .withStatusRtb(RtbStatus.NEW)
                        .withDateBetween(
                                Date.from(dateTimeService.getCurrentInstant().minus(3, ChronoUnit.DAYS)),
                                Date.from(dateTimeService.getCurrentInstant().minusSeconds(ageSeconds)))
                        .withArchive(false)
        );
        for (Addition addition : additions) {
            AdditionElementOptions options = addition.findFilesOptions();
            String videoId = options.getVideoId();
            if (videoId != null) {
                VideoFiles file = movieService.getFileByIdInternal(videoId);
                if (file != null && file.getStatus() == FileStatus.READY) {
                    patchAddition(addition, true);
                }
            }
        }
    }
}
