package ru.yandex.canvas.service.video;


import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;

import static ru.yandex.canvas.VideoConstants.CANVAS_RANGE_RATIO_CPC_FEATURE;
import static ru.yandex.canvas.VideoConstants.CANVAS_RANGE_RATIO_CPC_FEATURE_ALLOWED_CREATIVE_TYPES;
import static ru.yandex.canvas.VideoConstants.CANVAS_RANGE_RATIO_FEATURE;

public class VideoGeometryService {
    private static final Ratios OLD_GEOMETRY = new Ratios(new Ratio("16:9"), new Ratio("16:9"));
    private static final Ratios UNIVERSAL_GEOMETRY = new Ratios(new Ratio("3:1"), new Ratio("1:3"));
    private static final Ratios TALL_GEOMETRY = new Ratios(new Ratio("254:255"), new Ratio("1:3"));
    private static final Ratios WIDE_GEOMETRY = new Ratios(new Ratio("3:1"), new Ratio("255:254"));
    private static final Ratios SQUARE_GEOMETRY = new Ratios(new Ratio("1:1"), new Ratio("1:1"));

    private final VideoLimitsService limits;

    public VideoGeometryService(VideoLimitsService limits) {
        this.limits = limits;
    }


    public static class Ratios {
        private final Ratio from;
        private final Ratio to;

        public Ratios(Ratio from, Ratio to) {
            this.from = from;
            this.to = to;
        }

        public Ratio getFrom() {
            return from;
        }

        public Ratio getTo() {
            return to;
        }
    }

    private boolean checkFeatures(Set<String> features, VideoCreativeType videoCreativeType) {
        return (features.contains(CANVAS_RANGE_RATIO_FEATURE) || (features.contains(CANVAS_RANGE_RATIO_CPC_FEATURE) &&
                CANVAS_RANGE_RATIO_CPC_FEATURE_ALLOWED_CREATIVE_TYPES.contains(videoCreativeType)));
    }

    public boolean hasAllowedRatiosInterval(Geometry geometry, Set<String> features,
                                            VideoCreativeType videoCreativeType) {
        if (geometry != Geometry.NONE && checkFeatures(features, videoCreativeType)) {
            return true;
        }

        return false;
    }

    public List<Ratio> getAllowedRatios(VideoCreativeType creativeType, Long presetId) {
        return limits.getRatiosByVideoCreativeType(creativeType, presetId)
                .stream()
                .map(e -> e == null ? null : new Ratio(e))
                .collect(Collectors.toList());
    }

    public Ratios getRatiosByPreset(@Nonnull Geometry geometry, Set<String> features,
                                    VideoCreativeType videoCreativeType) {
        switch (geometry) {
            case UNIVERSAL:
                if (checkFeatures(features, videoCreativeType)) {
                    return UNIVERSAL_GEOMETRY;
                } else {
                    return OLD_GEOMETRY;
                }
            case WIDE:
                if (checkFeatures(features, videoCreativeType)) {
                    return WIDE_GEOMETRY;
                } else {
                    return OLD_GEOMETRY;
                }
            case TALL:
                return TALL_GEOMETRY;
            case SQUARE:
                return SQUARE_GEOMETRY;
            case NONE:
                return null;
            default:
                throw new IllegalArgumentException();
        }

    }


}
