package ru.yandex.canvas.service.video;

import java.util.Collections;
import java.util.List;
import java.util.Set;

import javax.validation.constraints.NotNull;

import ru.yandex.adv.direct.video.AdditionOptions;
import ru.yandex.adv.direct.video.MediaFile;
import ru.yandex.adv.direct.video.Packshot;
import ru.yandex.adv.direct.video.PlaybackParameters;
import ru.yandex.adv.direct.video.VideoParameters;
import ru.yandex.canvas.TimeDelta;
import ru.yandex.canvas.model.video.PythiaParams;
import ru.yandex.canvas.model.video.addition.AdditionElement;
import ru.yandex.canvas.model.video.addition.Options;
import ru.yandex.canvas.model.video.files.StreamFormat;
import ru.yandex.canvas.service.video.presets.PresetTag;
import ru.yandex.canvas.service.video.presets.VideoPreset;

import static java.lang.Math.floor;
import static java.util.stream.Collectors.toList;
import static ru.yandex.canvas.VideoConstants.AUDIO_CREATIVE_PREVIEW_URL;
import static ru.yandex.canvas.VideoConstants.VIDEO_AD_SYSTEM_CPC;
import static ru.yandex.canvas.VideoConstants.VIDEO_AD_SYSTEM_DEFAULT;

public class VideoParametersBuilder {
    private String vpaidPcodeUrl;
    private String theme;
    private Double duration;
    private List<StreamFormat> mediaFiles;
    private String packshotUrl;
    private boolean hasAbuseButton = false;
    private boolean socialAdvertisement = false;
    private AdParams.PlaybackParameters playbackParameters;
    private PythiaParams pythiaParams;
    private boolean useTrackingEvents = false;
    private boolean isStock = false;
    private List<AdditionElement> elements;
    private boolean interactiveVpaid = false;
    private boolean addPixelImpression = false;
    private Long creativeId;
    private String strmPrefix;
    private boolean showVpaid = true;
    private boolean showVideoClicks = false;
    private Long soundbtnLayout;
    private Long adlabelLayout;
    private Long countdownLayout;
    private boolean needIcon = false;
    private String skipUrl;
    private boolean useVpaidImpressions = false;
    private String adSystem;
    private String videoMetaId;
    private String estKaltura;
    private String estMobileVertical;
    private Integer width;
    private Integer height;
    private String firstFrameUrl;
    private String playerId;
    private String signaturesUrl;


    public VideoParameters build() {
        var builder = VideoParameters.newBuilder();

        if (vpaidPcodeUrl != null) {
            builder.setVpaidPcodeUrl(vpaidPcodeUrl);
        }
        if (theme != null) {
            builder.setTheme(theme);
        }
        if (duration != null) {
            builder.setDuration(duration);
        }
        builder.addAllMediaFiles(buildMediaFiles());
        if (packshotUrl != null) {
            builder.setPackshot(buildPackshot());
        }
        builder.setHasAbuseButton(hasAbuseButton);
        builder.setSocialAdvertisement(socialAdvertisement);
        if (playbackParameters != null) {
            builder.setPlaybackParameters(buildPlaybackParameters());
        }
        if (pythiaParams != null) {
            builder.setPythiaParams(buildPythiaParams());
        }
        builder.setUseTrackingEvents(useTrackingEvents);
        builder.setIsStock(isStock);
        builder.addAllAdditionElements(buildAdditionElements());

        builder.setInteractiveVpaid(interactiveVpaid);
        builder.setAddPixelImpression(addPixelImpression);
        if (creativeId != null) {
            builder.setCreativeId(creativeId);
        }
        if (strmPrefix != null) {
            builder.setStrmPrefix(strmPrefix);
        }
        builder.setShowVpaid(showVpaid);
        builder.setShowVideoClicks(showVideoClicks);
        if (soundbtnLayout != null) {
            builder.setSoundbtnLayout(soundbtnLayout);
        }
        if (adlabelLayout != null) {
            builder.setAdlabelLayout(adlabelLayout);
        }
        if (countdownLayout != null) {
            builder.setCountdownLayout(countdownLayout);
        }
        if (needIcon) {
            if (packshotUrl != null) {
                builder.setIcon(packshotUrl);
            } else {
                builder.setIcon(AUDIO_CREATIVE_PREVIEW_URL);
            }
        }
        if (skipUrl != null) {
            builder.setSkipUrl(skipUrl);
        }
        builder.setUseVpaidImpressions(useVpaidImpressions);
        if (adSystem != null) {
            builder.setAdSystem(adSystem);
        }
        if (videoMetaId != null) {
            builder.setVideoMetaId(videoMetaId);
        }
        if (estKaltura != null || estMobileVertical != null) {
            builder.setKalturaStream(buildKalturaStream());
        }
        if (firstFrameUrl != null) {
            builder.addFirstFrameParameters(buildFirstFrameParameter());
        }
        if (playerId != null) {
            builder.setPlayerId(playerId);
        }
        if (signaturesUrl != null) {
            builder.setSignatures(buildSignaturesParameter());
        }
        return builder.build();
    }

    private List<MediaFile> buildMediaFiles() {
        if (mediaFiles == null) {
            return Collections.emptyList();
        }
        return mediaFiles.stream().map(VideoParametersBuilder::buildMediaFile).collect(toList());
    }

    private static MediaFile buildMediaFile(StreamFormat mediaFile) {
        var builder = MediaFile.newBuilder();
        if (mediaFile.getId() != null) {
            builder.setId(mediaFile.getId());
        }
        if (mediaFile.getDelivery() != null) {
            builder.setDelivery(mediaFile.getDelivery());
        }
        if (mediaFile.getWidth() != null) {
            builder.setWidth(mediaFile.getWidth());
        }
        if (mediaFile.getHeight() != null) {
            builder.setHeight(mediaFile.getHeight());
        }
        if (mediaFile.getUrl() != null) {
            builder.setUrl(mediaFile.getUrl());
        }
        if (mediaFile.getMimeType() != null) {
            builder.setMimeType(mediaFile.getMimeType());
        }
        if (mediaFile.getBitrate() != null) {
            builder.setBitrate(mediaFile.getBitrate());
        }
        if (mediaFile.getCodec() != null) {
            builder.setCodec(mediaFile.getCodec());
        }
        if (mediaFile.getFramerate() != null) {
            builder.setFramerate(mediaFile.getFramerate());
        }
        if (mediaFile.getFileSize() != null) {
            builder.setFileSize(mediaFile.getFileSize());
        }
        if (mediaFile.getHasAudio() != null) {
            builder.setHasAudio(mediaFile.getHasAudio());
        }
        if (mediaFile.getVmafAvg() != null) {
            builder.setVmafAvg(mediaFile.getVmafAvg());
        }
        return builder.build();
    }

    private Packshot buildPackshot() {
        var builder = Packshot.newBuilder()
                .setImageUrl(packshotUrl);
        if (duration != null) {
            builder.setDuration(3.0 + (floor(duration + 0.5) - duration));
        }
        return builder.build();
    }

    private PlaybackParameters buildPlaybackParameters() {
        var builder = PlaybackParameters.newBuilder();
        builder.setShowSkipButton(playbackParameters.showSkipButton != null && playbackParameters.showSkipButton);
        if (playbackParameters.skipDelay != null) {
            builder.setSkipDelay(playbackParameters.skipDelay);
        }
        return builder.build();
    }

    private ru.yandex.adv.direct.video.SignaturesParameter buildSignaturesParameter() {
        var builder = ru.yandex.adv.direct.video.SignaturesParameter.newBuilder();
        if (signaturesUrl != null) {
            builder.setSignaturesUrl(signaturesUrl);
        }
        return builder.build();
    }

    private ru.yandex.adv.direct.video.FirstFrameParameter buildFirstFrameParameter() {
        var builder = ru.yandex.adv.direct.video.FirstFrameParameter.newBuilder();
        builder.setUrl(firstFrameUrl);
        builder.setHeight(height);
        builder.setWidth(width);
        builder.setType("image/jpeg");//У аватарницы всегда такой mimeType
        return builder.build();
    }

    private ru.yandex.adv.direct.video.KalturaStream buildKalturaStream() {
        var builder = ru.yandex.adv.direct.video.KalturaStream.newBuilder();
        if (estKaltura != null) {
            builder.setEstKaltura(estKaltura);
        }
        if (estMobileVertical != null) {
            builder.setEstMobileVertical(estMobileVertical);
        }
        return builder.build();
    }

    private ru.yandex.adv.direct.video.PythiaParams buildPythiaParams() {
        var builder = ru.yandex.adv.direct.video.PythiaParams.newBuilder();
        if (pythiaParams.getSlug() != null) {
            builder.setSlug(pythiaParams.getSlug());
        }
        if (pythiaParams.getBasePath() != null) {
            builder.setBasePath(pythiaParams.getBasePath());
        }
        if (pythiaParams.getExtra() != null) {
            builder.setExtra(pythiaParams.getExtra());
        }
        return builder.build();
    }

    private List<ru.yandex.adv.direct.video.AdditionElement> buildAdditionElements() {
        if (elements == null) {
            return Collections.emptyList();
        }
        return elements.stream()
                .filter(element -> element.getType().getPcodeName() != null)
                .map(VideoParametersBuilder::buildAdditionElement)
                .collect(toList());
    }

    private static ru.yandex.adv.direct.video.AdditionElement buildAdditionElement(AdditionElement element) {
        var builder = ru.yandex.adv.direct.video.AdditionElement.newBuilder();
        builder.setType(element.getType().getPcodeName());
        if (element.getOptions() != null) {
            builder.setOptions(buildOptions(element.getOptions()));
        }
        return builder.build();
    }

    private static AdditionOptions buildOptions(Options options) {
        var builder = AdditionOptions.newBuilder();
        if (options.getBackgroundColor() != null) {
            builder.setBackgroundColor(options.getBackgroundColor());
        }
        if (options.getTextColor() != null) {
            builder.setTextColor(options.getTextColor());
        }
        if (options.getText() != null) {
            builder.setText(options.getText());
        }
        if (options.getPlaceholder() != null) {
            builder.setPlaceholder(options.getPlaceholder());
        }
        if (options.getVideoId() != null) {
            builder.setVideoId(options.getVideoId());
        }
        if (options.getAudioId() != null) {
            builder.setAudioId(options.getAudioId());
        }
        if (options.getOverlayId() != null) {
            builder.setOverlayId(options.getOverlayId());
        }
        if (options.getPackshotId() != null) {
            builder.setPackshotId(options.getPackshotId());
        }
        if (options.getBorderColor() != null) {
            builder.setBorderColor(options.getBorderColor());
        }
        if (options.getPosition() != null) {
            builder.setPosition(options.getPosition());
        }
        if (options.getCustomLabel() != null && !"—".equals(options.getCustomLabel())) {
            builder.setCustomLabel(options.getCustomLabel());
        }
        if (options.getColor() != null) {
            builder.setColor(options.getColor());
        }
        return builder.build();
    }

    public void setVpaidPcodeUrl(String vpaidPcodeUrl) {
        this.vpaidPcodeUrl = vpaidPcodeUrl;
    }

    public void setTheme(String theme) {
        this.theme = theme;
    }

    public void setDuration(Double duration) {
        this.duration = duration;
    }

    public void setMediaFiles(List<StreamFormat> mediaFiles) {
        this.mediaFiles = mediaFiles;
    }

    public void setPackshotUrl(String packshotUrl) {
        this.packshotUrl = packshotUrl;
    }

    public void setHasAbuseButton(boolean hasAbuseButton) {
        this.hasAbuseButton = hasAbuseButton;
    }

    public void setSocialAdvertisement(boolean socialAdvertisement) {
        this.socialAdvertisement = socialAdvertisement;
    }

    public void setPlaybackParameters(boolean showSkipButton, TimeDelta skipDelay) {
        playbackParameters = new AdParams.PlaybackParameters();
        playbackParameters.showSkipButton = skipDelay == null ? false : showSkipButton;
        playbackParameters.skipDelay = skipDelay == null || skipDelay.asSeconds() == 0 || !showSkipButton ?
                null : (long) skipDelay.asSeconds();
    }

    public void setSignaturesParameters(String signaturesUrl) {
        this.signaturesUrl = signaturesUrl;
    }

    public void setPythiaParams(PythiaParams pythiaParams) {
        this.pythiaParams = pythiaParams;
    }

    public void setUseTrackingEvents(boolean useTrackingEvents) {
        this.useTrackingEvents = useTrackingEvents;
    }

    public void setIsStock(boolean isStock) {
        this.isStock = isStock;
    }

    public void setElements(List<AdditionElement> elements) {
        this.elements = elements;
    }

    public void setInteractiveVpaid(boolean interactiveVpaid) {
        this.interactiveVpaid = interactiveVpaid;
    }

    public void setAddPixelImpression(boolean addPixelImpression) {
        this.addPixelImpression = addPixelImpression;
    }

    public void setCreativeId(Long creativeId) {
        this.creativeId = creativeId;
    }

    public VideoParametersBuilder setVideoMetaId(String videoMetaId) {
        this.videoMetaId = videoMetaId;
        return this;
    }

    public VideoParametersBuilder setEstKaltura(String estKaltura) {
        this.estKaltura = estKaltura;
        return this;
    }

    public VideoParametersBuilder setEstMobileVertical(String estMobileVertical) {
        this.estMobileVertical = estMobileVertical;
        return this;
    }

    public VideoParametersBuilder setWidth(Integer width) {
        this.width = width;
        return this;
    }

    public VideoParametersBuilder setHeight(Integer height) {
        this.height = height;
        return this;
    }

    public VideoParametersBuilder setFirstFrameUrl(String firstFrameUrl) {
        this.firstFrameUrl = firstFrameUrl;
        return this;
    }

    public void setStrmPrefix(String strmPrefix) {
        this.strmPrefix = strmPrefix;
    }

    public void setShowVpaid(boolean showVpaid) {
        this.showVpaid = showVpaid;
    }

    public void setShowVideoClicks(boolean showVideoClicks) {
        this.showVideoClicks = showVideoClicks;
    }

    public void setSoundbtnLayout(Long soundbtnLayout) {
        this.soundbtnLayout = soundbtnLayout;
    }

    public void setAdlabelLayout(Long adlabelLayout) {
        this.adlabelLayout = adlabelLayout;
    }

    public void setCountdownLayout(Long countdownLayout) {
        this.countdownLayout = countdownLayout;
    }

    public void setNeedIcon(boolean needIcon) {
        this.needIcon = needIcon;
    }

    public void setSkipUrl(String skipUrl) {
        this.skipUrl = skipUrl;
    }

    public void setUseVpaidImpressions(boolean useVpaidImpressions) {
        this.useVpaidImpressions = useVpaidImpressions;
    }

    public void setAdSystem(String adSystem) {
        this.adSystem = adSystem;
    }

    public void setPlayerId(String playerId) {
        this.playerId = playerId;
    }

    public void fillPresetParameters(@NotNull VideoPreset preset) {
        setHasAbuseButton(preset.isControlsAllowed());
        setSoundbtnLayout(preset.isControlsAllowed() ? 1L : 0L);
        setAdlabelLayout(preset.isControlsAllowed() ? 1L : 0L);
        setCountdownLayout(preset.isControlsAllowed() ? 1L : 0L);
        setAddPixelImpression(preset.getAddImpressionPixel() != null && preset.getAddImpressionPixel());
        setUseVpaidImpressions(preset.getUseVpaidImpressions() != null && preset.getUseVpaidImpressions());
        setUseTrackingEvents(preset.getUseTrackingEvents() != null && preset.getUseTrackingEvents()
                || preset.getDescription().getCpmAudio());//DIRECT-171366 Нужно для единого ответа, иначе они не соберут VAST

        Set<PresetTag> presetTags = preset.getTags();
        if (presetTags.contains(PresetTag.COMMON) ||
            presetTags.contains(PresetTag.MOBILE_CONTENT) ||
            presetTags.contains(PresetTag.MOBILE_CONTENT_VIDEO)) {
            setAdSystem(VIDEO_AD_SYSTEM_CPC);
        } else {
            setAdSystem(VIDEO_AD_SYSTEM_DEFAULT);
        }

        var presetDescription = preset.getDescription();
        setInteractiveVpaid(presetDescription.getInteractiveVpaid());
        setShowVpaid(!presetDescription.getSkipVpaid());
    }
}
