package ru.yandex.canvas.service.video;

import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.io.Writer;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.file.FileSystem;
import java.nio.file.FileSystemNotFoundException;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.datatype.jdk8.Jdk8Module;
import com.github.mustachejava.DefaultMustacheFactory;
import com.github.mustachejava.Mustache;
import com.google.common.io.Resources;
import one.util.streamex.StreamEx;

import ru.yandex.canvas.service.AuthRequestParams;
import ru.yandex.canvas.service.DirectService;
import ru.yandex.canvas.service.TankerKeySet;
import ru.yandex.canvas.service.VideoLimitsInterface;
import ru.yandex.canvas.service.video.presets.PresetDescription;
import ru.yandex.canvas.service.video.presets.PresetTag;
import ru.yandex.canvas.service.video.presets.VideoPreset;
import ru.yandex.canvas.service.video.presets.configs.ConfigType;
import ru.yandex.canvas.service.video.presets.configs.SourceTypes;
import ru.yandex.canvas.service.video.presets.configs.options.FileOptionConfig;
import ru.yandex.canvas.service.video.presets.elements.PresetElementType;
import ru.yandex.direct.feature.FeatureName;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static org.springframework.context.i18n.LocaleContextHolder.getLocale;
import static ru.yandex.canvas.VideoConstants.CANVAS_RANGE_RATIO_CPC_FEATURE;
import static ru.yandex.canvas.VideoConstants.CANVAS_RANGE_RATIO_CPC_FEATURE_TAGS;
import static ru.yandex.canvas.VideoConstants.CANVAS_RANGE_RATIO_FEATURE;
import static ru.yandex.canvas.VideoConstants.SUBTITLES_FOR_VIDEO_CREATIVES_ENABLED_FEATURE;

public class VideoPresetsService {

    private static final String PRESETS_DIRECTORY = "video_presets";
    private Map<Long, VideoPreset> presets;
    private Map<PresetTag, List<VideoPreset>> presetsByTag;
    private VideoLimitsService videoLimitsService;
    private DirectService directService;
    private AuthRequestParams authRequestParams;

    protected Path getPath(URI presetsUri) throws IOException {
        if (presetsUri.getScheme().equalsIgnoreCase("jar")) {
            FileSystem fileSystem;

            try {
                fileSystem = FileSystems.getFileSystem(presetsUri);
            } catch (FileSystemNotFoundException e) {
                fileSystem = FileSystems.newFileSystem(presetsUri, emptyMap());
            }

            return fileSystem.getPath(PRESETS_DIRECTORY);
        } else {
            return Paths.get(presetsUri);
        }
    }

    public VideoPresetsService(VideoLimitsService videoLimitsService, DirectService directService,
                               AuthRequestParams authRequestParams) {
        this.videoLimitsService = videoLimitsService;
        this.directService = directService;
        this.authRequestParams = authRequestParams;

        ObjectMapper objectMapper = makeObjectMapper();
        presets = new HashMap<>();
        presetsByTag = new HashMap<>();

        try {
            URI presetsUri = Resources.getResource(PRESETS_DIRECTORY).toURI();
            Path presetsPath = getPath(presetsUri);

            Files.walk(presetsPath)
                    .filter(Files::isRegularFile)
                    .forEach(presetPath -> {
                        try {
                            VideoPreset videoPreset = fromJson(Files.readString(presetPath), objectMapper);
                            hackVideoPresetWithVideoConstructorSource(videoPreset);
                            hackVideoPresetWithSubtitlesConfigAndElementList(videoPreset);
                            presets.put(videoPreset.getId(), videoPreset);

                            for (PresetTag tag : videoPreset.getTags()) {
                                presetsByTag
                                        .computeIfAbsent(tag, e -> new ArrayList<>())
                                        .add(videoPreset);
                            }

                        } catch (IOException e) {
//                            logger.error("can not load preset file: " + presetPath, e);
                            throw new RuntimeException("can not load preset file: " + presetPath, e);
                        }
                    });

        } catch (IOException | URISyntaxException e) {
            //        logger.error("can not load presets directory", e);
            throw new RuntimeException("can not load presets directory", e);
        }
    }

    public VideoCreativeType fromPresetId(Long presetId) {
        VideoPreset videoPreset = getPreset(presetId);
        return videoPreset.videoCreativeType();
    }

    protected VideoPreset fromJson(String json, ObjectMapper objectMapper) throws IOException {
        PresetDescription presetDescription = objectMapper.readValue(json, PresetDescription.class);
        return new VideoPreset(presetDescription);
    }

    protected ObjectMapper makeObjectMapper() {
        ObjectMapper objectMapper = new ObjectMapper();
        objectMapper.enable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);
        objectMapper.disable(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS);
        objectMapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);
        objectMapper.registerModule(new Jdk8Module());

        return objectMapper;
    }

    public boolean contains(Long id) {
        return presets.containsKey(id);
    }

    public List<Long> getPresetIds() {
        return new ArrayList<>(presets.keySet());
    }

    public VideoPreset getPresetSafe(Long id) {
        if (id == null || !contains(id)) {
            return null;
        }

        return getPreset(id);
    }

    public VideoPreset getPreset(Long id) {
        return presets.get(id);
    }

    public List<VideoPreset> getPresets() {
        return new ArrayList<>(presets.values());
    }

    public List<VideoPreset> getPresetsByTag(PresetTag tag) {
        Set<String> features = directService
                .getFeatures(authRequestParams.getClientId().orElse(null),
                        authRequestParams.getUserId().orElse(null));

        return presetsByTag.getOrDefault(tag, emptyList())
                .stream()
                .sorted(Comparator.comparing(VideoPreset::getOrder))
                .filter(it -> {
                    Geometry geometry = it.getDescription().getGeometry();
                    return geometry == Geometry.WIDE || geometry == Geometry.UNIVERSAL
                            || features.contains(CANVAS_RANGE_RATIO_FEATURE)
                            || (features.contains(CANVAS_RANGE_RATIO_CPC_FEATURE)
                            && CANVAS_RANGE_RATIO_CPC_FEATURE_TAGS.contains(tag));
                })
                .collect(Collectors.toUnmodifiableList());
    }

    private void hackVideoPresetWithVideoConstructorSource(VideoPreset videoPreset) {
        // добавляем источник 'видеоконструктор' только для клиентов с фичей
        videoPreset.getConfig().getConfigs().get(ConfigType.ADDITION).getOptionConfigs()
                .stream()
                .filter(oc -> FileOptionConfig.class.isAssignableFrom(oc.getClass()))
                .map(FileOptionConfig.class::cast)
                .filter(Predicate.not(FileOptionConfig::getCpmAudio))
                .filter(foc -> "video".equalsIgnoreCase(foc.getResourceType()))
                .forEach(foc -> foc.setSourcesSupplier(() -> {
                    Set<String> features = directService
                            .getFeatures(authRequestParams.getClientId().orElse(null),
                                    authRequestParams.getUserId().orElse(null));
                    if (features.contains(FeatureName.VIDEO_CONSTRUCTOR_ENABLED.getName())) {
                        return StreamEx.of(foc.getSources())
                                .append(SourceTypes.VIDEO_CONSTRUCTOR).toList();
                    } else {
                        return foc.getSources();
                    }
                }));
    }

    private void hackVideoPresetWithSubtitlesConfigAndElementList(VideoPreset videoPreset) {
        // DIRECT-136236 поле Субтитры должно быть доступно только по фиче
        if (videoPreset.hasSubtitlesElement()) {
            videoPreset.getConfig().setConfigsSupplier(videoPreset.getId(), () -> {
                Set<String> features = directService
                        .getFeatures(authRequestParams.getClientId().orElse(null),
                                authRequestParams.getUserId().orElse(null));
                var configs = videoPreset.getConfig().getConfigsBeforeFeatures();
                if (!features.contains(SUBTITLES_FOR_VIDEO_CREATIVES_ENABLED_FEATURE)) {
                    var configsCopy = new HashMap<>(configs);
                    configsCopy.remove(ConfigType.SUBTITLES);
                    return configsCopy;
                } else {
                    return configs;
                }
            });

            videoPreset.setPresetElementListSupplier(() -> {
                Set<String> features = directService
                        .getFeatures(authRequestParams.getClientId().orElse(null),
                                authRequestParams.getUserId().orElse(null));
                var presetElementList = videoPreset.getPresetElementListBeforeFeatures();
                if (!features.contains(SUBTITLES_FOR_VIDEO_CREATIVES_ENABLED_FEATURE)) {
                    var presetElementListCopy = new ArrayList<>(presetElementList);
                    presetElementListCopy.removeIf(presetElement ->
                            presetElement.getType() == PresetElementType.SUBTITLES);
                    return presetElementListCopy;
                } else {
                    return presetElementList;
                }
            });
        }
    }

    public List<VideoPreset> getPresetsByCreativeType(VideoCreativeType creativeType) {
        return getPresetsByTag(creativeType.toPresetTag());
    }

    public String tooltipInterpolator(String tooltipValue, VideoCreativeType creativeType, Long presetId) {
        if (tooltipValue == null) {
            return null;
        }

        if (tooltipValue.isEmpty()) {
            return tooltipValue;
        }

        String sourceTemplate = TankerKeySet.VIDEO.key(tooltipValue);

        if (sourceTemplate.contains("${")) {
            Map<String, String> replacements = new HashMap<>();

            VideoLimitsInterface videoLimits = videoLimitsService.getLimits(creativeType, presetId);
            Mustache mustache = new DefaultMustacheFactory()
                    .compile(new StringReader(sourceTemplate), tooltipValue + getLocale(), "${", "}");

            replacements.put("minDuration", videoLimits.getDurationMin().longValue() + "");
            replacements.put("maxDuration", videoLimits.getDurationMax().longValue() + "");
            replacements.put("isCaptureLessThanMaxDuration",
                    videoLimits.getDurationMax() - videoLimits.getDurationCaptureStop() > 0.01 ? "1" : null);
            replacements.put("captureLen", videoLimits.getDurationCaptureStop().longValue() + "");
            replacements.put("videoFileHelpUrl", TankerKeySet.VIDEO.key(videoLimits.getVideoFileHelpUrl()));

            Writer writer = new StringWriter();

            mustache.execute(writer, replacements);

            return writer.toString();
        } else {
            return sourceTemplate;
        }
    }

}
