package ru.yandex.canvas.service.video.files;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;

import ru.yandex.canvas.model.video.files.AudioSource;
import ru.yandex.canvas.model.video.files.Movie;
import ru.yandex.canvas.model.video.files.MovieAndVideoSourceFactory;
import ru.yandex.canvas.model.video.files.VideoSource;
import ru.yandex.canvas.service.video.VideoFuzzySearchService;
import ru.yandex.canvas.service.video.presets.VideoPreset;

public class StockMoviesService {
    private static final String FILE_PATH = "/video_stock/merged.json";
    private static final String CONSTRUCTOR_AUDIO_FILE_PATH = "/video_stock/constructor_audio.json";
    private static final String GROUPS_FILE_PATH = "/video_stock/groups.json";

    private Map<String, VideoSource> videos;
    private Map<String, AudioSource> audios;
    private Map<String, AudioSource> constructorAudios;
    private Map<String, Map<String, Movie>> indexedFiles;
    private Map<String, Set<String>> groupsDb;
    //TODO rewrite as List<GroupClass> groups; GroupClass { String name, Set<> videoIds; }
    private List<String> orderedGroupNames;
    private Map<String, Long> videoOrder;
    private Map<String, Long> audioOrder;
    private Map<String, Long> constructorAudioOrder;
    private Set<String> knownSubCategories;

    private AudioSearch audioSearch;
    private AudioSearch constructorAudioSearch;
    private VideoSearch videoSearch;

    private final MovieAndVideoSourceFactory movieAndVideoSourceFactory;

    public StockMoviesService(VideoFuzzySearchService videoFuzzySearchService,
                              MovieAndVideoSourceFactory movieAndVideoSourceFactory) {
        videos = new HashMap<>();
        audios = new HashMap<>();
        constructorAudios = new HashMap<>();
        groupsDb = new HashMap<>();
        indexedFiles = new HashMap<>();
        orderedGroupNames = new ArrayList<>();
        audioOrder = new HashMap<>();
        constructorAudioOrder = new HashMap<>();
        videoOrder = new HashMap<>();
        knownSubCategories = new HashSet<>();
        this.movieAndVideoSourceFactory = movieAndVideoSourceFactory;

        initMergedDb();
        initConstructorAudioDb();
        initGroupsDb();

        audioSearch = new AudioSearch(audios, audioOrder);
        constructorAudioSearch = new AudioSearch(constructorAudios, constructorAudioOrder);
        videoSearch = new VideoSearch(videos, groupsDb, orderedGroupNames, videoOrder, videoFuzzySearchService);

    }

    void initMergedDb() {
        try {
            readMergedJsonDatabase(FILE_PATH);
        } catch (IOException e) {
//                            logger.error("can not load preset file: " + presetPath, e);
            throw new RuntimeException("can not load preset file: " + FILE_PATH, e);
        }
    }

    void initConstructorAudioDb() {
        try {
            readConstructorAudioJsonDatabase(CONSTRUCTOR_AUDIO_FILE_PATH);
        } catch (IOException e) {
            throw new RuntimeException("can not load preset file: " + CONSTRUCTOR_AUDIO_FILE_PATH, e);
        }
    }

    void initGroupsDb() {
        try {
            readGroupsJsonDatabase(GROUPS_FILE_PATH);
        } catch (IOException e) {
//                            logger.error("can not load preset file: " + presetPath, e);
            throw new RuntimeException("can not load preset file: " + GROUPS_FILE_PATH, e);
        }
    }

    void readGroupsJsonDatabase(String filePath) throws IOException {
        ObjectMapper objectMapper = new ObjectMapper();
        InputStream inputStream = getClass().getResourceAsStream(filePath);
        TypeReference<GroupFileRecord> typeReference = new TypeReference<GroupFileRecord>() {
        };

        Iterator<GroupFileRecord> iterator = objectMapper.readerFor(typeReference).readValues(inputStream);

        while (iterator.hasNext()) {
            GroupFileRecord groupFileRecord = iterator.next();

            for (String videoId : groupFileRecord.getVideoIds()) {
                if (!videos.containsKey(videoId)) {
                    throw new IllegalArgumentException(GROUPS_FILE_PATH + " contains unknown video id " + videoId);
                }
            }

            groupsDb.put(groupFileRecord.getGroupName(), new HashSet<>(groupFileRecord.getVideoIds()));
            orderedGroupNames.add(groupFileRecord.getGroupName());
        }

    }

    void readMergedJsonDatabase(String presetPath) throws IOException {
        ObjectMapper objectMapper = new ObjectMapper();

        InputStream inputStream = getClass().getResourceAsStream(presetPath);

        TypeReference<MergedFileRecord> typeReference = new TypeReference<MergedFileRecord>() {
        };

        Iterator<MergedFileRecord> iterator = objectMapper.readerFor(typeReference).readValues(inputStream);

        final int[] stockNumber = {1};

        while (iterator.hasNext()) {
            MergedFileRecord record = iterator.next();

            VideoSource videoSource = videos.computeIfAbsent(VideoSource.getIdFromDescription(record),
                    e -> movieAndVideoSourceFactory.videoSourceFromMergedRecord(record, "Video " + stockNumber[0]++));

            AudioSource audioSource = audios.computeIfAbsent(AudioSource.getIdFromDescription(record),
                    e -> new AudioSource(record));

            Movie movie = movieAndVideoSourceFactory.movieFromVideoAudioSources(record, videoSource, audioSource);

            Map<String, Movie> row = indexedFiles.computeIfAbsent(videoSource.getId(), (k) -> new HashMap<>());
            row.put(audioSource.getId(), movie);

            if (record.getVideo().getInterfaceOrder() == null || record.getAudio().getInterfaceOrder() == null) {
                throw new IllegalArgumentException("Order not defined in stock file " + record.getId());
            }

            knownSubCategories.add(videoSource.getCategoryId());

            videoOrder.putIfAbsent(videoSource.getId(), record.getVideo().getInterfaceOrder());
            audioOrder.putIfAbsent(audioSource.getId(), record.getAudio().getInterfaceOrder());
        }

        knownSubCategories = Collections.unmodifiableSet(knownSubCategories);
    }

    void readConstructorAudioJsonDatabase(String presetPath) throws IOException {
        ObjectMapper objectMapper = new ObjectMapper();

        InputStream inputStream = getClass().getResourceAsStream(presetPath);

        TypeReference<MergedFileRecord.AudioDescription> typeReference = new TypeReference<>() {
        };

        Iterator<MergedFileRecord.AudioDescription> iterator =
                objectMapper.readerFor(typeReference).readValues(inputStream);

        long order = 0;

        while (iterator.hasNext()) {
            MergedFileRecord.AudioDescription record = iterator.next();

            if (record.getStillageId() != null) {
                AudioSource audioSource = constructorAudios.computeIfAbsent(record.getStillageId(),
                        e -> new AudioSource(record));
                constructorAudioOrder.put(audioSource.getId(), order++);
            }
        }
    }

    public Movie getFileByIds(String videoId, String audioId) {
        Map<String, Movie> row = indexedFiles.get(videoId);

        if (row != null) {
            return row.get(audioId);
        }

        return null;
    }

    public Iterator<VideoSource> iterator() {
        return videos.values().iterator();
    }

    public SearchResult<AudioSource> searchAudio(String nameSubString) {
        return audioSearch.search(nameSubString, audios.values().size(), 0);
    }

    public SearchResult<AudioSource> searchAudio(String nameSubString, int limit, int offset) {
        return audioSearch.search(nameSubString, limit, offset);
    }

    public SearchResult<AudioSource> searchConstructorAudio(String nameSubString, int limit, int offset) {
        return constructorAudioSearch.search(nameSubString, limit, offset);
    }

    public static class SearchResult<T> {
        private long total;
        private List<T> result;

        public SearchResult(long total, List<T> result) {
            this.total = total;
            this.result = result;
        }

        public long getTotal() {
            return total;
        }

        public List<T> getResult() {
            return result;
        }
    }

    public SearchResult<VideoSource> searchVideo(List<String> categories, List<Integer> groupIds,
                                                 String nameSubString) {
        return videoSearch.search(categories, groupIds, nameSubString);
    }

    public SearchResult<VideoSource> searchVideo(List<String> categories, List<Integer> groupIds,
                                                 String nameSubString, VideoPreset preset,
                                                 int limit, int offset, String semanticSearch) {
        return videoSearch.search(categories, groupIds, nameSubString, preset, limit, offset, semanticSearch);
    }

    public VideoSource getVideo(String id) {
        return videos.get(id);
    }

    public AudioSource getAudio(String id) {
        AudioSource source = audios.get(id);
        if (source == null) {
            source = constructorAudios.get(id);
        }
        return source;
    }

    public List<String> getGroups() {
        return Collections.unmodifiableList(orderedGroupNames);
    }

    public Set<String> getKnownSubCategories() {
        return knownSubCategories;
    }
}
