package ru.yandex.canvas.service.video.files;

import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import one.util.streamex.StreamEx;

import ru.yandex.canvas.model.video.files.VideoSource;
import ru.yandex.canvas.service.video.Ratio;
import ru.yandex.canvas.service.video.VideoFuzzySearchService;
import ru.yandex.canvas.service.video.presets.VideoPreset;

import static java.lang.Integer.min;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.groupingBy;

public class VideoSearch {
    private Map<String, VideoSource> videos;
    private Map<String, List<VideoSource>> indexByCategory;
    private Map<String, Set<String>> groupsDb;
    private List<String> orderedGroupNames;
    private List<VideoSource> allVideos;
    private Map<String, Long> videoOrder;
    private VideoFuzzySearchService videoFuzzySearchService;
    private static final Ratio DEFAULT_RATIO_FROM = new Ratio("3:1");
    private static final Ratio DEFAULT_RATIO_TO = new Ratio("10:16");

    public VideoSearch(Map<String, VideoSource> videos, Map<String, Set<String>> groupsDb,
                       List<String> orderedGroupNames, Map<String, Long> videoOrder,
                       VideoFuzzySearchService videoFuzzySearchService) {
        this.videos = videos;
        this.groupsDb = groupsDb;
        this.orderedGroupNames = orderedGroupNames;
        this.videoOrder = videoOrder;
        this.videoFuzzySearchService = videoFuzzySearchService;

        this.allVideos = videos.values()
                .stream()
                .sorted(Comparator.comparing(e -> videoOrder.get(e.getId())))
                .collect(Collectors.toList());

        indexByCategory = videos.values().stream().collect(groupingBy(e -> e.getCategoryId()));
    }

    public StockMoviesService.SearchResult<VideoSource> search(List<String> categories, List<Integer> groupIds,
                                                               String nameSubString) {
        return search(categories, groupIds, nameSubString, null, allVideos.size(), 0, null);
    }


    public StockMoviesService.SearchResult<VideoSource> search(List<String> categories, List<Integer> groupIds,
                                                               String nameSubString, VideoPreset preset,
                                                               int limit, int offset, String semanticSearch) {
        Stream<VideoSource> found;

        if (categories != null && groupIds != null) {
            found = fromBothIndexes(categories, groupIds);
        } else if (categories != null) {
            found = fromCategoriesIndex(categories);
        } else if (groupIds != null) {
            found = fromGroupsIndex(groupIds);
        } else {
            if (nameSubString == null) {
                found = null;
            } else {
                found = videos.values().stream();
            }
        }

        if (nameSubString != null) {
            found = found.filter(e -> e.getName().contains(nameSubString));
        }

        List<VideoSource> videoSources;

        if (found != null) {
            videoSources = found.distinct()
                    .sorted(Comparator.comparing(e -> videoOrder.get(e.getId())))
                    .collect(Collectors.toList());
        } else {
            videoSources = allVideos;
        }

        //фильтрация по соотношению сторон
        Ratio ratioFrom = ratioFrom(preset);
        Ratio ratioTo = ratioTo(preset);
        videoSources = StreamEx.of(videoSources)
                .filter(it -> {
                    var ratioPercent = it.getRatioPercent();
                    return ratioPercent <= ratioFrom.ratioPercent() && ratioPercent >= ratioTo.ratioPercent();
                })
                .toList();

        if (videoSources.size() == 0) {
            return new StockMoviesService.SearchResult<>(0, emptyList());
        }
        if (videoSources.size() < offset) {
            return new StockMoviesService.SearchResult<>(videoSources.size(), emptyList());
        }

        if (semanticSearch != null && semanticSearch.trim().length() > 0) {
            videoSources = fuzzySearch(semanticSearch, videoSources);
        }

        int fromIndex = min(offset, videoSources.size() - 1);
        int toIndex = min(offset + limit, videoSources.size());

        return new StockMoviesService.SearchResult<>(videoSources.size(), videoSources.subList(fromIndex, toIndex));
    }

    private static Ratio ratioFrom(VideoPreset preset) {
        if (preset != null && preset.getDescription() != null
                && preset.getDescription().getRatioFrom() != null
                && preset.getDescription().getRatioTo() != null) {
            return new Ratio(preset.getDescription().getRatioFrom());
        }
        return DEFAULT_RATIO_FROM;
    }

    private static Ratio ratioTo(VideoPreset preset) {
        if (preset != null && preset.getDescription() != null
                && preset.getDescription().getRatioFrom() != null
                && preset.getDescription().getRatioTo() != null) {
            return new Ratio(preset.getDescription().getRatioTo());
        }
        return DEFAULT_RATIO_TO;
    }

    Stream<VideoSource> fromBothIndexes(List<String> categories, List<Integer> groupIds) {

        Set<String> group = groupIds.stream()
                .filter(e -> e >= 0 && e < orderedGroupNames.size())
                .map(e -> orderedGroupNames.get(e))
                .filter(e -> groupsDb.containsKey(e))
                .flatMap(e -> groupsDb.get(e).stream())
                .collect(Collectors.toSet());

        return fromCategoriesIndex(categories).filter(e -> group.contains(e.getName()));
    }

    Stream<VideoSource> fromCategoriesIndex(List<String> categories) {
        return categories.stream().map(e -> indexByCategory.get(e)).filter(Objects::nonNull).flatMap(e -> e.stream());
    }

    Stream<VideoSource> fromGroupsIndex(List<Integer> groupIds) {
        return groupIds.stream()
                .filter(e -> e >= 0 && e < orderedGroupNames.size())
                .map(e -> groupsDb.get(orderedGroupNames.get(e)))
                .filter(Objects::nonNull)
                .flatMap(e -> e.stream())
                .map(e -> videos.get(e));
    }

    List<VideoSource> fuzzySearch(String semanticSearch, List<VideoSource> movies) {

        List<String> stillageIds = movies == allVideos ? null : movies.stream().map(e -> e.getStillageId()).collect(
                Collectors.toList());

        List<String> sortedStillageIds = videoFuzzySearchService
                .rangeFiles(semanticSearch, stillageIds, null, null);

        Map<String, Integer> orderMap = new HashMap<>();

        for (int i = 0; i < sortedStillageIds.size(); i++) {
            orderMap.put(sortedStillageIds.get(i), i);
        }

        VideoSource[] copy = new VideoSource[movies.size()];

        for (int i = 0; i < movies.size(); i++) {
            VideoSource entry = movies.get(i);
            copy[orderMap.get(entry.getStillageId())] = entry;
        }

        return Arrays.asList(copy);
    }

}
