package ru.yandex.canvas.service.video.overlay;

import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.io.UncheckedIOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.github.mustachejava.DefaultMustacheFactory;
import com.github.mustachejava.Mustache;
import one.util.streamex.EntryStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.multipart.MultipartFile;

import ru.yandex.canvas.model.stillage.StillageFileInfo;
import ru.yandex.canvas.model.video.overlay.OverlayAsset;
import ru.yandex.canvas.model.video.overlay.OverlayBundle;
import ru.yandex.canvas.model.video.overlay.OverlayCreative;
import ru.yandex.canvas.repository.video.OverlayCreativesRepository;
import ru.yandex.canvas.service.DateTimeService;
import ru.yandex.canvas.service.SequenceService;
import ru.yandex.canvas.service.StillageService;
import ru.yandex.canvas.service.html5.Html5Zip;

/**
 * Обработка оверлейных креативов
 */
public class OverlayService {
    private Logger logger = LoggerFactory.getLogger(OverlayService.class);

    private static final int MAX_FILE_SIZE = 1024 * 1024;  // 1 MB

    /**
     * Статичная картинка-заглушка для отображения превью оверлейных креативов.
     */
    public static final String OVERLAY_PREVIEW_URL =
            "https://storage.mds.yandex.net/get-bstor/2267992/20d63640-e1cb-4f1f-9fab-2eccca376c54.png";

    private final OverlayCreativesRepository overlayCreativesRepository;
    private final StillageService stillageService;
    private final SequenceService sequenceService;
    private final DateTimeService dateTimeService;

    public OverlayService(OverlayCreativesRepository overlayCreativesRepository,
                          StillageService stillageService,
                          SequenceService sequenceService,
                          DateTimeService dateTimeService) {
        this.overlayCreativesRepository = overlayCreativesRepository;
        this.stillageService = stillageService;
        this.sequenceService = sequenceService;
        this.dateTimeService = dateTimeService;
    }

    /**
     * Валидирует zip-архив, извлекает из него js- и xml- файлы, загружает их в stillage
     * После этого в xml- вставляет ссылки на загруженные js-ассеты (ссылки ведут в MDS)
     * И возвращает готовый к сохранению объект OverlayBundle, на базе которого далее можно будет создать креатив
     */
    public OverlayBundle processZipFile(MultipartFile file, Long clientId) {
        try {
            byte[] bytes = file.getBytes();
            if (bytes.length >= MAX_FILE_SIZE) {
                throw new OverlayValidationException("{file_too_big}", MAX_FILE_SIZE);
            }
            Html5Zip html5Zip = new Html5Zip(bytes);

            String vastAssetName = null;
            Set<String> jsAssetNames = new HashSet<>();
            for (String assetName : html5Zip.files()) {
                if (assetName.contains("/")) {
                    logger.info("Skip nested asset {}", assetName);
                    continue;
                }
                if (assetName.endsWith(".js")) {
                    logger.info("Found js asset {}", assetName);
                    jsAssetNames.add(assetName);
                } else if (assetName.endsWith(".xml")) {
                    if (vastAssetName != null) {
                        logger.error("Found duplicate VAST asset {}", assetName);
                        throw new OverlayValidationException("{found_duplicate_vast_asset}", assetName);
                    }
                    logger.info("Found VAST asset {}", assetName);
                    vastAssetName = assetName;
                } else {
                    logger.info("Skip asset {} (not .js or .xml)", assetName);
                }
            }

            if (vastAssetName == null) {
                throw new OverlayValidationException("{vast_asset_not_found}");
            }
            if (jsAssetNames.isEmpty()) {
                throw new OverlayValidationException("{no_js_assets_found}");
            }

            // Теперь загружаем в Stillage
            Map<String, StillageFileInfo> jsAssetsUploaded = new HashMap<>();
            StillageFileInfo vastUploaded = stillageService.uploadFile(vastAssetName,
                    html5Zip.getFileContent(vastAssetName));
            for (String jsAssetName : jsAssetNames) {
                byte[] content = html5Zip.getFileContent(jsAssetName);
                StillageFileInfo uploadedInfo = stillageService.uploadFile(jsAssetName, content);
                jsAssetsUploaded.put(jsAssetName, uploadedInfo);
            }

            // Готовим шаблонизированный VAST
            String vastTemplate = html5Zip.getFileAsUtf8String(vastAssetName);
            Map<String, String> vars = EntryStream.of(jsAssetsUploaded).mapValues(StillageFileInfo::getUrl).toMap();
            String vast = applyVastTemplate(vastTemplate, vars);

            List<OverlayAsset> overlayAssets = new ArrayList<>();
            overlayAssets.add(toOverlayAsset(vastAssetName, vastUploaded));
            overlayAssets.addAll(EntryStream.of(jsAssetsUploaded).mapKeyValue(this::toOverlayAsset).toList());

            return new OverlayBundle()
                    .setFiles(overlayAssets)
                    .setArchive(false)
                    .setClientId(clientId)
                    .setDate(dateTimeService.getCurrentDate())
                    .setVast(vast)
                    .setName(file.getOriginalFilename());
        } catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private OverlayAsset toOverlayAsset(String fileName, StillageFileInfo stillageFileInfo) {
        return new OverlayAsset()
                .setName(fileName)
                .setStillageId(stillageFileInfo.getId())
                .setStillageUrl(stillageFileInfo.getUrl());
    }

    /**
     * Выполняет шаблонизацию шаблона template с использованием набора значений placeholderValues.
     */
    static String applyVastTemplate(String template, Map<String, String> placeholderValues) {
        Mustache templater = new DefaultMustacheFactory().compile(new StringReader(template), "vars.json");
        try {
            try (StringWriter writer = new StringWriter()) {
                templater.execute(writer, prepareMustacheVars(placeholderValues));
                writer.flush();
                return writer.toString();
            }
        } catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    /**
     * Конвертирует переменные, в ключах которых могут содержаться точки, в переменные, пригодные для
     * шаблонизации через mustache (там уже в ключах точек быть не должно).
     * <p>
     * Для этого готовим вложенную структуру, чтобы mustache смог вставить полечко вида {{{file.name.js}}}
     * Дело в том, что mustache по точке всегда ожидает вложенный объект, и мы должны его предоставить
     */
    private static Map<String, Object> prepareMustacheVars(Map<String, String> dottedVars) {
        Map<String, Object> vars = new HashMap<>();
        for (String key : dottedVars.keySet()) {
            String[] split = key.split("\\.", -1);
            Map<String, Object> map = vars;
            for (int i = 0; i < split.length; i++) {
                if (i == split.length - 1) {
                    map.put(split[i], dottedVars.get(key));
                } else {
                    Map<String, Object> nestedMap = new HashMap<>();
                    map.put(split[i], nestedMap);
                    map = nestedMap;
                }
            }
        }
        return vars;
    }

    public void insert(OverlayBundle overlayBundle) {
        overlayCreativesRepository.insert(overlayBundle);
    }

    public OverlayBundle findById(String overlayId, Long clientId) {
        return overlayCreativesRepository.findById(overlayId, clientId);
    }

    public String getVast(OverlayBundle overlayBundle) {
        return overlayBundle.getVast();
    }

    public void save(OverlayCreative overlayCreative) {
        List<Long> nextCreativeIds = sequenceService.getNextCreativeIdsList(1);
        overlayCreative.setCreativeId(nextCreativeIds.get(0));
        overlayCreativesRepository.insertCreative(overlayCreative);
    }
}
