package ru.yandex.canvas.service.video.presets;

import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import ru.yandex.canvas.TimeDelta;
import ru.yandex.canvas.service.video.VideoCreativeType;
import ru.yandex.canvas.service.video.presets.elements.PresetElement;
import ru.yandex.canvas.service.video.presets.elements.PresetElementType;

public class VideoPreset {
    private String name;
    private String thumbnail;
    private String bundleName;
    private Set<PresetTag> tags;
    private List<PresetElement> presetElementList;
    private Supplier<List<PresetElement>> presetElementListSupplier;
    private Config config;
    private Long id;
    private TimeDelta skipOffset;
    private Boolean allowStockVideo;
    private Boolean packshotAllowed;
    private Boolean addTrackingUrl;
    private Boolean addImpressionPixel;
    private PresetDescription description;
    private Boolean controlsAllowed;
    private Boolean recentVideoAllowed;
    private Long order;
    private Boolean useVpaidImpressions;
    private Boolean useDefaultBodyAndTitle;
    private Boolean useTrackingEvents;
    private Boolean hasSubtitlesElement;

    public VideoPreset(PresetDescription description) {
        init(description);
    }

    protected void init(PresetDescription description) {
        id = description.getPresetId();

        bundleName = "video-banner_theme_" + description.getPresetTheme().toString().toLowerCase();
        tags = new HashSet<>(description.getTags());
        skipOffset = new TimeDelta(description.getSkipOffset());
        name = description.getPresetName();
        thumbnail = description.getThumbnail();
        allowStockVideo = description.getAllowStockVideo();
        packshotAllowed = description.getPackshotPresent();
        addTrackingUrl = description.getAddTrackingUrl();
        addImpressionPixel = description.getAddImpressionPixel();
        order = description.getOrder();
        controlsAllowed = description.getControlsAllowed();
        recentVideoAllowed = description.getAllowRecentVideoSource();
        useVpaidImpressions = description.getUseVpaidImpressions();
        useDefaultBodyAndTitle = description.getUseDefaultBodyAndTitle();
        useTrackingEvents = description.getUseTrackingEvents();
        hasSubtitlesElement = description.getSubtitlesPresent();

        presetElementList = Arrays.stream(PresetElementType.values())
                .map(v -> v.getFactory().create(description))
                .filter(Objects::nonNull)
                .collect(Collectors.toList());

        config = new Config(description);
        this.description = description;
    }

    public VideoCreativeType videoCreativeType() {
        var tag = getTags().iterator().next();

        if (tag == null) {
            throw new IllegalArgumentException("Preset " + getId() + " without tag!");
        }

        return VideoCreativeType.fromPresetTag(tag);
    }

    public PresetDescription getDescription() {
        return description;
    }

    public Boolean getAllowStockVideo() {
        return allowStockVideo;
    }

    public Boolean isPackshotAllowed() {
        return packshotAllowed;
    }

    public Long getId() {
        return id;
    }

    public String getName() {
        return name;
    }

    public String getThumbnail() {
        return thumbnail;
    }

    public String getBundleName() {
        return bundleName;
    }

    public Set<PresetTag> getTags() {
        return tags;
    }

    public TimeDelta getSkipOffset() {
        return skipOffset;
    }

    public Config getConfig() {
        return config;
    }

    public List<PresetElement> getPresetElementListBeforeFeatures() {
        return presetElementList;
    }

    public List<PresetElement> getPresetElementList() {
        return presetElementListSupplier != null ? presetElementListSupplier.get() : presetElementList;
    }

    public void setPresetElementListSupplier(Supplier<List<PresetElement>> presetElementListSupplier) {
        if (this.presetElementListSupplier != null) {
            throw new IllegalArgumentException("Preset " + getId() + " has multiple element list suppliers!");
        }
        this.presetElementListSupplier = presetElementListSupplier;
    }

    public Boolean getAddTrackingUrl() {
        return addTrackingUrl;
    }

    public Boolean getAddImpressionPixel() {
        return addImpressionPixel;
    }

    public Boolean getPackshotAllowed() {
        return packshotAllowed;
    }

    public Long getOrder() {
        return order;
    }

    public boolean isControlsAllowed() {
        return controlsAllowed;
    }

    public Boolean isRecentVideoAllowed() {
        return recentVideoAllowed;
    }

    public Boolean getUseVpaidImpressions() {
        return useVpaidImpressions;
    }

    public void setUseVpaidImpressions(Boolean useVpaidImpressions) {
        this.useVpaidImpressions = useVpaidImpressions;
    }

    public Boolean getUseDefaultBodyAndTitle() {
        return useDefaultBodyAndTitle;
    }

    public Boolean getUseTrackingEvents() {
        return useTrackingEvents;
    }

    public Boolean hasSubtitlesElement() {
        return hasSubtitlesElement;
    }
}
