package ru.yandex.direct.common;

import java.io.IOException;
import java.net.URL;
import java.util.Arrays;
import java.util.Collection;
import java.util.Locale;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContext;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.core.io.Resource;
import org.springframework.core.io.support.ResourcePatternResolver;
import org.springframework.stereotype.Service;

import ru.yandex.direct.i18n.I18NBundle;
import ru.yandex.direct.i18n.Translatable;
import ru.yandex.direct.i18n.Translator;
import ru.yandex.direct.i18n.TranslatorFactory;
import ru.yandex.direct.i18n.dict.BundleDictionaries;
import ru.yandex.direct.i18n.dict.DictionaryLoader;

@Service
@ParametersAreNonnullByDefault
public class TranslationService {
    private static final Logger logger = LoggerFactory.getLogger(TranslationService.class);

    private final TranslatorFactory translatorFactory;

    @Autowired
    @SuppressWarnings("unchecked")
    public TranslationService(ResourcePatternResolver resourceResolver) throws IOException {
        Collection<URL> urls = Arrays.stream(resourceResolver.getResources("classpath*:locale/ru/yandex/**/*.json"))
                .map(el -> resourceUrl(el))
                .collect(Collectors.toList());
        logger.info("discovered locale bundles: {}", urls);
        BundleDictionaries dicts = new DictionaryLoader("/locale/").loadFromURLs(urls);
        this.translatorFactory = I18NBundle.makeTranslatorFactory(dicts);
        logger.info("locale bundles loaded");
    }

    //нужно для читабельности лямбд (кинуть IOException прямо в лямбде нельзя)
    private URL resourceUrl(Resource rsrc) {
        try {
            return rsrc.getURL();
        } catch (IOException ex) {
            throw new RuntimeException(ex);
        }
    }

    public String translate(Translatable translatable) {
        return translate(translatable, getLocale());
    }

    public String translate(Translatable translatable, Locale locale) {
        Translator translator = translatorFactory.getTranslator(locale);
        return translatable.translate(translator);
    }

    @Nullable
    public String translate(String bundleName, @Nullable String key) {
        return translate(bundleName, key, getLocale());
    }

    @Nullable
    public String translate(String bundleName, @Nullable String key, Locale locale) {
        if (key == null) {
            return null;
        }
        Translator translator = translatorFactory.getTranslator(locale);
        return translator.translateKey(bundleName, key);
    }

    // CHECKSTYLE:OFF
    /**
     * Долгий путь локали от HTTP-запроса клиента до транслятора:
     *
     * <ul>
     * <li>ApiWsDispatcherServlet наследуется от {@link org.springframework.web.servlet.FrameworkServlet}, в т.ч.
     * чтобы переопределить
     * {@link org.springframework.web.servlet.FrameworkServlet#buildLocaleContext(javax.servlet.http.HttpServletRequest)}
     * </li>
     * <li>buildLocaleContext в свою очередь обращается к ApiLocaleResolver</li>
     * <li>после чего {@link org.springframework.web.servlet.FrameworkServlet#processRequest}
     * вызывает {@link LocaleContextHolder#setLocaleContext(LocaleContext, boolean)}</li>
     * <li>Локаль попадает в TranslationService</li>
     * </ul>
     *
     * @return
     */
    // CHECKSTYLE:ON
    public Locale getLocale() {
        return LocaleContextHolder.getLocale();
    }
}
