package ru.yandex.direct.common.admin.engine;

import java.io.IOException;
import java.io.PrintWriter;
import java.net.InetAddress;
import java.util.List;
import java.util.Optional;
import java.util.function.Function;

import javax.servlet.ServletConfig;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.google.common.collect.ImmutableMap;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.context.support.SpringBeanAutowiringSupport;

import ru.yandex.direct.common.net.NetAcl;
import ru.yandex.direct.common.util.HttpUtil;
import ru.yandex.direct.env.EnvironmentType;

import static java.util.stream.Collectors.joining;

/**
 * Сервлет для получения информации о системе
 */
public class AdminServlet extends HttpServlet {
    private static final Logger logger = LoggerFactory.getLogger(AdminServlet.class);
    private ImmutableMap<String, ActionExecutor> executorMap;

    @Autowired
    private EnvironmentType environmentType;

    @Autowired
    private NetAcl netAcl;

    @Autowired
    private List<ActionExecutor> executorList;

    private String getAction(ActionExecutor executor) {
        Class<?> clazz = executor.getClass();
        Action action = clazz.getAnnotation(Action.class);
        if (action == null) {
            throw new RuntimeException("Action annotation must be specified for " + clazz.getCanonicalName());
        }
        return action.value();
    }

    @Override
    protected void doGet(HttpServletRequest request, HttpServletResponse response) throws IOException {
        PrintWriter writer = response.getWriter();
        try {
            checkRights(request);

            String action = request.getParameter("action");
            if (!executorMap.containsKey(action)) {
                writer.write("Action is unknown. Available only:\n");
                writer.write(executorMap.keySet().stream().collect(joining("\n")));
                return;
            }
            executorMap.get(action).execute(request, response);
        } catch (RuntimeException e) {
            String message = Optional.ofNullable(e.getMessage()).orElse("Internal server error");
            logger.error(message, e);
            writer.write(message);
        }
    }

    @Override
    public void init(ServletConfig conf) throws ServletException {
        super.init(conf);
        if (environmentType == null) {
            // сервлет может инициализироваться как spring-bean, тогда всё уже заавтовайрено
            SpringBeanAutowiringSupport.processInjectionBasedOnServletContext(this,
                    conf.getServletContext());
        }

        executorMap = ImmutableMap.copyOf(StreamEx.of(executorList).toMap(this::getAction, Function.identity()));
    }

    private void checkRights(HttpServletRequest request) {
        InetAddress remoteAddress = HttpUtil.getRemoteAddress(request);

        if (environmentType.isProductionOrPrestable()) {
            if (!remoteAddress.isLoopbackAddress()) {
                throw new AdminServletException("Access denied");
            }
        } else if (!netAcl.isInternalIp(remoteAddress)) {
            throw new AdminServletException("Access denied");
        }
    }
}
