package ru.yandex.direct.common.alive;

import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FsFreeSpaceInspector implements HealthInspector {
    private static final Logger logger = LoggerFactory.getLogger(FsFreeSpaceInspector.class);

    private final Set<File> partitions;
    private final double freeSpaceBorder;

    public FsFreeSpaceInspector(Collection<File> partitionsBag, double freeSpaceBorder) {
        this.partitions = partitionsBag == null ? Collections.emptySet() : new HashSet<>(partitionsBag);
        this.freeSpaceBorder = freeSpaceBorder;
    }

    @Override
    public HealthStatus inspectHealth() {
        List<String> issuesList = new ArrayList<>();
        for (File partitionFile : partitions) {
            if (partitionFile.exists()) {
                double freeSpaceRatio = partitionFile.getFreeSpace() / (double) partitionFile.getTotalSpace();
                if (!(Double.isFinite(freeSpaceRatio) && freeSpaceRatio > freeSpaceBorder)) {
                    logger.error("df:space:{} - not enough space: {} < {}", partitionFile.getPath(), freeSpaceRatio,
                            freeSpaceBorder);
                    issuesList.add("df:space:" + partitionFile.getPath());
                }
            } else {
                logger.error("df:{} - path does not exist", partitionFile.getPath());
                issuesList.add("df:" + partitionFile.getPath());
            }
        }
        return new HealthStatus(issuesList);
    }
}
