package ru.yandex.direct.common.alive;

import java.io.File;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import javax.annotation.PreDestroy;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.dbutil.wrapper.DatabaseWrapper;
import ru.yandex.direct.dbutil.wrapper.DatabaseWrapperProvider;

@Component
public class HealthInspectorBean implements HealthInspector {
    private static final Logger logger = LoggerFactory.getLogger(HealthInspectorBean.class);

    private final Collection<String> dbNames;
    private final FsFreeSpaceInspector fsFreeSpaceInspector;
    private final FsWritableStatusInspector fsWritableStatusInspector;
    private final AliveForceDownChecker aliveForceDownChecker;

    private final DatabaseWrapperProvider dbWrapperProvider;

    private final boolean cacheEnabled;
    private ScheduledExecutorService cacheRefreshExecutor;
    private HealthStatus cachedHealthStatus;
    private Instant cachedLastUpdated;
    private Duration expireTimeout;

    @Autowired
    public HealthInspectorBean(DirectConfig directConfig, DatabaseWrapperProvider dbWrapperProvider) {
        DirectConfig healthInspectorConfig = directConfig.getBranch("health_checker");
        dbNames = healthInspectorConfig.getStringList("db_names");
        this.dbWrapperProvider = dbWrapperProvider;
        aliveForceDownChecker = createAliveForceDownChecker(healthInspectorConfig);
        fsFreeSpaceInspector = createFsFreeSpaceInspector(healthInspectorConfig);
        fsWritableStatusInspector = createFsWritableStatusInspector(healthInspectorConfig);

        cacheEnabled = healthInspectorConfig.getBoolean("enable_cache");
        if (cacheEnabled) {
            cacheRefreshExecutor = Executors.newSingleThreadScheduledExecutor(new ThreadFactoryBuilder()
                    .setNameFormat("healthCheckRefresh")
                    .setDaemon(true)
                    .build());

            expireTimeout = healthInspectorConfig.getDuration("expire_timeout");
            cachedLastUpdated = null;

            Duration refreshInterval = healthInspectorConfig.getDuration("refresh_interval");
            cacheRefreshExecutor.scheduleAtFixedRate(this::refreshCached, 0,
                    refreshInterval.toMillis(), TimeUnit.MILLISECONDS);
        }
    }

    private FsWritableStatusInspector createFsWritableStatusInspector(DirectConfig healthInspectorConfig) {
        Collection<File> writableCheckPaths = healthInspectorConfig
                .getStringList("writable_checks")
                .stream()
                .map(File::new)
                .collect(Collectors.toList());
        return new FsWritableStatusInspector(writableCheckPaths);
    }

    private FsFreeSpaceInspector createFsFreeSpaceInspector(DirectConfig healthInspectorConfig) {
        return new FsFreeSpaceInspector(
                healthInspectorConfig.getStringList("partitions").stream().map(File::new).collect(Collectors.toList()),
                healthInspectorConfig.getDouble("free_space_limit"));
    }

    private AliveForceDownChecker createAliveForceDownChecker(DirectConfig healthInspectorConfig) {
        String aliveFileName = healthInspectorConfig.getString("alive_file_name");
        return new AliveForceDownChecker(aliveFileName);
    }

    private void refreshCached() {
        try {
            cachedHealthStatus = getHealthStatus();
        } catch (RuntimeException e) {
            cachedHealthStatus = new HealthStatus(List.of("exception:" + e.getClass().getSimpleName()));
        }
        cachedLastUpdated = Instant.now();
    }

    @Override
    public HealthStatus inspectHealth() {
        if (cacheEnabled) {
            // Если с последнего обновления прошло меньше времени, чем expireTimeout
            if (cachedLastUpdated != null &&
                    Duration.between(cachedLastUpdated, Instant.now()).compareTo(expireTimeout) < 0) {
                return cachedHealthStatus;
            } else {
                throw new RuntimeException("Health status expired, last update was at " + cachedLastUpdated);
            }
        }
        return getHealthStatus();
    }

    private HealthStatus getHealthStatus() {
        return fsFreeSpaceInspector.inspectHealth()
                .merge(fsWritableStatusInspector.inspectHealth())
                .merge(inspectDbAvailability())
                .merge(aliveForceDownChecker.inspectHealth());
    }

    private HealthStatus inspectDbAvailability() {
        List<String> issuesList = new ArrayList<>();
        for (String dbName : dbNames) {
            DatabaseWrapper dbWrapper = dbWrapperProvider.get(dbName);
            if (dbWrapper == null) {
                logger.error("unknown db {}", dbName);
                throw new MonitoringConfigException("unknown database: " + dbName);
            }
            if (!dbWrapper.isAlive()) {
                issuesList.add("db:" + dbWrapper.getDbname());
                logger.error("db {} not available", dbWrapper.getDbname());
            }
        }
        return new HealthStatus(issuesList);
    }

    @PreDestroy
    public void close() {
        if (cacheRefreshExecutor != null) {
            cacheRefreshExecutor.shutdownNow();
        }
    }
}
