package ru.yandex.direct.common.configuration;

import java.lang.management.ManagementFactory;
import java.time.Clock;
import java.time.Duration;
import java.time.ZoneId;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;

import javax.net.ssl.SSLException;

import com.google.common.primitives.Ints;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import io.netty.handler.ssl.SslContextBuilder;
import io.netty.handler.ssl.util.InsecureTrustManagerFactory;
import io.netty.util.HashedWheelTimer;
import one.util.streamex.StreamEx;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.DefaultAsyncHttpClient;
import org.asynchttpclient.DefaultAsyncHttpClientConfig;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.FilterType;
import org.springframework.context.annotation.Import;
import org.springframework.context.annotation.Lazy;
import org.springframework.context.annotation.Primary;

import ru.yandex.direct.asynchttp.FetcherSettings;
import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.common.jetty.JettyLauncherStartListener;
import ru.yandex.direct.common.liveresource.zookeeper.ZookeeperLiveResourceProvider;
import ru.yandex.direct.common.log.service.LogMetricsService;
import ru.yandex.direct.common.mobilecontent.MobileContentStoreType;
import ru.yandex.direct.common.mobilecontent.MobileContentYtTable;
import ru.yandex.direct.common.mobilecontent.MobileContentYtTablesConfig;
import ru.yandex.direct.common.net.NetAcl;
import ru.yandex.direct.common.util.DirectThreadPoolExecutor;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.config.EssentialConfiguration;
import ru.yandex.direct.dbutil.configuration.DbUtilConfiguration;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.libs.curator.CuratorFrameworkProvider;
import ru.yandex.direct.libs.mirrortools.MirrorToolsConfig;
import ru.yandex.direct.libs.mirrortools.utils.HostingsHandler;
import ru.yandex.direct.liveresource.LiveResource;
import ru.yandex.direct.liveresource.LiveResourceFactory;
import ru.yandex.direct.liveresource.LiveResourceWatcherFactory;
import ru.yandex.direct.liveresource.provider.LiveResourceFactoryBean;
import ru.yandex.direct.rbac.configuration.RbacConfiguration;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.regions.GeoTreeFactory;
import ru.yandex.direct.regions.GeoTreeLoader;
import ru.yandex.direct.regions.GeoTreeType;
import ru.yandex.direct.regions.SimpleGeoTreeFactory;
import ru.yandex.direct.regions.SwitchingGeoTreeFactory;
import ru.yandex.direct.solomon.SolomonPushClient;
import ru.yandex.direct.yav.client.YavClient;
import ru.yandex.direct.yav.client.YavClientImpl;
import ru.yandex.direct.yav.client.YavClientImpl.YavClientConfig;

/**
 * Common configuration for Direct applications
 */
@Configuration
@ComponentScan(
        basePackages = "ru.yandex.direct.common",
        excludeFilters = {@ComponentScan.Filter(value = Configuration.class, type = FilterType.ANNOTATION),
                @ComponentScan.Filter(type = FilterType.REGEX, pattern = "ru.yandex.direct.common.logging\\..*")}
)
@Import({
        EssentialConfiguration.class,
        TracingConfiguration.class,
        MetricsConfiguration.class,
        DbUtilConfiguration.class,
        TvmIntegrationConfiguration.class,
        RbacConfiguration.class,
        RedisConfiguration.class
})
public class CommonConfiguration {

    /*
        Явные имена даются бинам для возможности их переопределения в
        другом @Configuration-классе, который импортирует данный
     */
    public static final String DIRECT_EXECUTOR_SERVICE = "DirectExecutorService";
    public static final String SYSTEM_UTC_CLOCK_NAME = "systemUtcClock";
    public static final String GO_ZORA_ASYNC_HTTP_CLIENT = "goZoraAsyncHttpClient";

    public static final long LIVE_CONFIG_CHECK_RATE = TimeUnit.SECONDS.toMillis(10); // 10 seconds

    private static final Logger logger = LoggerFactory.getLogger(CommonConfiguration.class);

    @Autowired
    DirectConfig directConfig;

    @Bean
    ZoneId defaultZoneId() {
        return ZoneId.of(directConfig.getString("timezone"));
    }

    @Bean(SYSTEM_UTC_CLOCK_NAME)
    Clock systemUtcClock() {
        return Clock.systemUTC();
    }

    @Bean
    @Lazy
    @Primary
    AsyncHttpClient asyncHttpClient() {
        DefaultAsyncHttpClientConfig.Builder builder = new DefaultAsyncHttpClientConfig.Builder();

        setCommonOptions(builder);

        return new DefaultAsyncHttpClient(builder.build());
    }

    @Bean(GO_ZORA_ASYNC_HTTP_CLIENT)
    @Lazy
    AsyncHttpClient goZoraAsyncHttpClient() throws SSLException {
        DefaultAsyncHttpClientConfig.Builder builder = new DefaultAsyncHttpClientConfig.Builder();

        setCommonOptions(builder);

        builder.setCookieStore(null);
        builder.setKeepAlive(false);
        builder.setSslContext(SslContextBuilder.forClient()
                .trustManager(InsecureTrustManagerFactory.INSTANCE)
                .build());

        return new DefaultAsyncHttpClient(builder.build());
    }

    private void setCommonOptions(DefaultAsyncHttpClientConfig.Builder builder) {
        DirectConfig config = directConfig.getBranch("default_http_client");

        builder.setRequestTimeout(Ints.saturatedCast(config.getDuration("request_timeout").toMillis()));
        builder.setReadTimeout(Ints.saturatedCast(config.getDuration("read_timeout").toMillis()));
        builder.setConnectTimeout(Ints.saturatedCast(config.getDuration("connect_timeout").toMillis()));
        builder.setConnectionTtl(Ints.saturatedCast(config.getDuration("connection_ttl").toMillis()));
        builder.setPooledConnectionIdleTimeout(
                Ints.saturatedCast(config.getDuration("pooled_connection_idle_timeout").toMillis()));
        builder.setIoThreadsCount(config.getInt("threads_count"));
        builder.setNettyTimer(new HashedWheelTimer(
                new ThreadFactoryBuilder().setNameFormat("ahc-timer-%02d").setDaemon(true).build()));
    }

    @Bean
    @Lazy
    NetAcl networkConfigFactory(LiveResourceWatcherFactory liveResourceWatcherFactory,
                                LiveResourceFactoryBean liveResourceFactoryBean) {
        String networkConfig = directConfig.getString("network_config");
        logger.debug("read {}", networkConfig);
        LiveResource liveResource = liveResourceFactoryBean.get(networkConfig);
        return NetAcl.createAndWatch(liveResource, liveResourceWatcherFactory);
    }

    @Bean
    @Lazy
    public GeoTreeFactory geoTreeFactory(DirectConfig config, PpcPropertiesSupport propertiesSupport) {
        List<GeoTree> geoTrees = StreamEx.of(GeoTreeType.values())
                .map(geoTreeType -> {
                    String file = config.getString("regions_data." + geoTreeType.name().toLowerCase());
                    String jsonBody = LiveResourceFactory.get(file).getContent();
                    return GeoTreeLoader.build(jsonBody, geoTreeType);
                })
                .toList();

        SimpleGeoTreeFactory prev = new SimpleGeoTreeFactory(geoTrees);

        List<GeoTree> nextGeoTrees = StreamEx.of(GeoTreeType.values())
                .map(geoTreeType -> {
                    String file = config.getString("regions_data_next." + geoTreeType.name().toLowerCase());
                    String jsonBody = LiveResourceFactory.get(file).getContent();
                    return GeoTreeLoader.build(jsonBody, geoTreeType);
                })
                .toList();

        SimpleGeoTreeFactory next = new SimpleGeoTreeFactory(nextGeoTrees);

        PpcProperty<Boolean> useNextGeoTreeProperty =
                propertiesSupport.get(PpcPropertyNames.USE_NEXT_GEO_TREE, Duration.ofSeconds(30));
        return new SwitchingGeoTreeFactory(() -> useNextGeoTreeProperty.getOrDefault(false), prev, next);
    }

    @Bean
    @Lazy
    HostingsHandler hostingsHandler() {
        MirrorToolsConfig config = new MirrorToolsConfig();
        return new HostingsHandler(config.getHostings(), config.getPublicSecondLevelDomains());
    }

    @Lazy
    @Bean
    MobileContentYtTablesConfig mobileContentYtTables() {
        Map<String, MobileContentYtTable> shops = new HashMap<>();
        DirectConfig mobileContentData = directConfig.getBranch("mobile_content_data");
        for (MobileContentStoreType type : MobileContentStoreType.values()) {
            mobileContentData.findBranch(type.getName()).ifPresent(c -> shops.put(type.getName(),
                    new MobileContentYtTable(c.getStringList("clusters"), c.getString("table"))));
        }
        return new MobileContentYtTablesConfig(shops);
    }

    @Lazy
    @Bean(DIRECT_EXECUTOR_SERVICE)
    ExecutorService directExecutorService(@Value("${db_shards}") int numOfPpcShards) {
        return new DirectThreadPoolExecutor(3 * numOfPpcShards, 200, "DirectThreadPoolExecutor");
    }

    @Bean(name = "solomonProject")
    public String solomonProject(EnvironmentType environmentType) {
        switch (environmentType) {
            case PRODUCTION:
            case PRESTABLE:
                return "direct";

            case TESTING:
            case TESTING2:
            case LOADTEST:
            case DB_TESTING:
            case SANDBOX:
            case SANDBOX_TESTING:
                return "direct-test";

            case DEVTEST:
            case DEV7:
            case DEVELOPMENT:
            case SANDBOX_DEVTEST:
            case SANDBOX_DEV7:
            case SANDBOX_DEVELOPMENT:
            case ROPROD:
                return "direct-junk";

            default:
                throw new IllegalStateException("Unknown environment type " + environmentType);
        }
    }

    @Lazy
    @Bean
    SolomonPushClient solomonPushClient(DirectConfig directConfig, AsyncHttpClient asyncHttpClient) {
        DirectConfig cfg = directConfig.getBranch("solomon.push");
        var parallelFetcherFactory = new ParallelFetcherFactory(
                asyncHttpClient,
                new FetcherSettings()
                        .withGlobalTimeout(cfg.findDuration("global_timeout").orElse(Duration.ofSeconds(3)))
                        .withRequestTimeout(cfg.findDuration("request_timeout").orElse(Duration.ofSeconds(3)))
                        .withRequestRetries(cfg.findInt("retries").orElse(2))
                        .withSoftTimeout(cfg.findDuration("soft_timeout").orElse(Duration.ofSeconds(1)))
        );
        return new SolomonPushClient(cfg.getString("url"), parallelFetcherFactory);
    }

    @Lazy
    @Bean
    public ZookeeperLiveResourceProvider zookeeperLiveResourceProvider(
            @Value("${zookeeper.config_servers}") String servers,
            @Value("${zookeeper.lock_path}") String lockPath
    ) {
        return new ZookeeperLiveResourceProvider(new CuratorFrameworkProvider(servers, lockPath));
    }

    @Bean
    YavClient yavClient(AsyncHttpClient asyncHttpClient,
                        @Value("${yav_client.url}") String baseUrl,
                        @Value("${yav_client.token}") String tokenUrl) {
        var config = directConfig.getBranch("yav_client");

        var fetcherSettings = new FetcherSettings()
                .withFailFast(true)
                .withRequestRetries(config.findInt("request_retries").orElse(2))
                .withRequestTimeout(config.findDuration("request_timeout").orElse(Duration.ofSeconds(1)));

        var yavClientConfig = new YavClientConfig(baseUrl);
        if (tokenUrl.contains("~/")) {
            tokenUrl = tokenUrl.replace("~/", System.getProperty("user.home") + "/");
        }
        var tokenLiveResource = LiveResourceFactory.get(tokenUrl);

        return new YavClientImpl(asyncHttpClient, fetcherSettings, yavClientConfig, tokenLiveResource);
    }

    @Bean
    public JettyLauncherStartListener jettyStartTimeReporter(LogMetricsService logMetricsService) {
        return () -> {
            logMetricsService.sendMetricAsync("java-jetty-start-time",
                    (double) ManagementFactory.getRuntimeMXBean().getUptime() / 1000,
                    Map.of(
                            "debug_mode", System.getProperty("intellij.debug.agent", "false"),
                            "main_class", System.getProperty("sun.java.command", "unknown")
                    )
            );
        };
    }
}
