package ru.yandex.direct.common.configuration;

import java.time.Duration;

import com.yandex.ydb.core.auth.AuthProvider;
import com.yandex.ydb.core.grpc.GrpcTransport;
import com.yandex.ydb.core.rpc.RpcTransport;
import com.yandex.ydb.table.SessionRetryContext;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.rpc.grpc.GrpcTableRpc;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.context.annotation.Lazy;

import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.ydb.YdbPath;
import ru.yandex.direct.ydb.client.YdbClient;
import ru.yandex.direct.ydb.client.YdbSessionProperties;
import ru.yandex.direct.ydb.tvm.YdbTvmAuthProvider;

@Lazy
@Configuration
@Import(TvmIntegrationConfiguration.class)
public class FrontDbYdbConfiguration {
    public static final String FRONTDB_YDB_SESSION_PROPERTIES_BEAN = "FrontDbYdbSessionProperties";
    public static final String FRONTDB_YDB_PATH_BEAN = "FrontDbYdbPath";
    public static final String FRONTDB_YDB_RPC_TRANSPORT_BEAN = "FrontDbRpcTransport";
    public static final String FRONTDB_YDB_TABLE_CLIENT_BEAN = "FrontDbTableClient";
    public static final String FRONTDB_YDB_CLIENT_BEAN = "FrontYdbClient";
    public static final String FRONTDB_YDB_INFO_BEAN = "FrontDbYdbInfo";

    @Bean(FRONTDB_YDB_PATH_BEAN)
    public YdbPath ydbPath(@Value("${ydb.cluster}") String cluster,
                           @Value("${ydb.frontdb.db_name}") String dbName) {
        return YdbPath.of(cluster, dbName);
    }

    @Bean
    public AuthProvider authProvider(TvmIntegration tvmIntegration) {
        return new YdbTvmAuthProvider(tvmIntegration);
    }


    @Bean(FRONTDB_YDB_RPC_TRANSPORT_BEAN)
    public RpcTransport grpcTransport(
            @Value("${ydb.endpoint}") String endpoint,
            @Qualifier(FRONTDB_YDB_PATH_BEAN) YdbPath dbPath,
            AuthProvider authProvider
    ) {
        GrpcTransport.Builder transportBuilder = GrpcTransport.forEndpoint(endpoint, dbPath.getPath())
                .withAuthProvider(authProvider);
        return transportBuilder.build();
    }


    @Bean(FRONTDB_YDB_SESSION_PROPERTIES_BEAN)
    public YdbSessionProperties ydbProperties(
            @Value("${ydb.frontdb.max_query_retries}") int maxQueryRetries,
            @Value("${ydb.frontdb.retry_not_found}") boolean retryNotFound
    ) {
        return YdbSessionProperties.builder()
                .withMaxQueryRetries(maxQueryRetries)
                .withRetryNotFound(retryNotFound)
                .build();
    }

    @Bean(FRONTDB_YDB_TABLE_CLIENT_BEAN)
    public TableClient tableClient(@Qualifier(FRONTDB_YDB_RPC_TRANSPORT_BEAN) RpcTransport rpcTransport,
                                   @Value("${ydb.frontdb.session_pool_min_size}") int sessionPoolMinSize,
                                   @Value("${ydb.frontdb.session_pool_max_size}") int sessionPoolMaxSize) {
        return TableClient.newClient(GrpcTableRpc.useTransport(rpcTransport))
                .sessionPoolSize(sessionPoolMinSize, sessionPoolMaxSize)
                .build();
    }

    @Bean(FRONTDB_YDB_CLIENT_BEAN)
    public YdbClient ydbClient(
            @Qualifier(FRONTDB_YDB_TABLE_CLIENT_BEAN) TableClient tableClient,
            @Qualifier(FRONTDB_YDB_SESSION_PROPERTIES_BEAN) YdbSessionProperties frontdbYdbSessionProperties,
            @Value("${ydb.frontdb.query_timeout}") long queryTimeoutSeconds) {
        var sessionRetryContext = SessionRetryContext.create(tableClient)
                .maxRetries(frontdbYdbSessionProperties.getMaxQueryRetries())
                .retryNotFound(frontdbYdbSessionProperties.isRetryNotFound())
                .build();

        return new YdbClient(sessionRetryContext, Duration.ofSeconds(queryTimeoutSeconds));
    }
}
