package ru.yandex.direct.common.configuration;

import java.util.Collection;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.context.event.ContextClosedEvent;
import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.context.event.EventListener;

import ru.yandex.direct.config.EssentialConfiguration;
import ru.yandex.direct.logging.StatsFilter;
import ru.yandex.direct.logging.UnifiedAgentLog4jUtils;
import ru.yandex.direct.metric.collector.MetricCollector;
import ru.yandex.direct.metric.collector.MetricProvider;
import ru.yandex.direct.solomon.SolomonUtils;
import ru.yandex.direct.version.DirectVersion;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.GaugeInt64;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

import static ru.yandex.direct.solomon.SolomonUtils.SOLOMON_REGISTRY;

/**
 * Отправка метрик о работе приложения в graphite
 */
@Configuration
@Import({EssentialConfiguration.class, GraphiteConfiguration.class})
public class MetricsConfiguration {
    public static final String METRIC_COLLECTOR_BEAN_NAME = "metricCollectorWrapper";

    @Autowired
    public MetricsConfiguration(@Value("${dont_add_default_metrics:false}") boolean dontAddDefaultMetrics,
                                MetricRegistry metricRegistry) {
        if (!dontAddDefaultMetrics) {
            SolomonUtils.addJvmMetrics(metricRegistry);

            metricRegistry.lazyCounter("sensors_total", metricRegistry::estimateCount);
            metricRegistry.lazyCounter("external_sensors_total", SolomonUtils::getExternalMetricsRegistryEstimateCount);

            GaugeInt64 major = metricRegistry.gaugeInt64("major_version");
            major.set(DirectVersion.getMajorVersion());

            GaugeInt64 minor = metricRegistry.gaugeInt64("minor_version");
            minor.set(DirectVersion.getMinorVersion());

            SolomonUtils.addUnifiedAgentLog4jMetrics(UnifiedAgentLog4jUtils.getUnifiedAgentLog4jMetrics());
        }
    }

    /**
     * Сделано бином для того, чтобы при необходимости можно было переопределить в приложениях
     */
    @Bean
    public static MetricRegistry metricRegistry() {
        return SOLOMON_REGISTRY;
    }

    /**
     * статистика о количестве залогированных сообщений
     */
    @Bean
    MetricProvider log4jMetricProvider() {
        return (buf) -> {
            // buf не используется в силу отказа от графита
            for (StatsFilter.StatItem statItem : StatsFilter.getStats()) {
                GaugeInt64 logsTotalSensor = SOLOMON_REGISTRY.gaugeInt64(
                        "log_lines_total",
                        Labels.of(
                                "stat_name", statItem.statName(),
                                "level", statItem.level().name().toLowerCase())
                );
                logsTotalSensor.set(statItem.count());
            }
        };
    }

    @Bean(name = METRIC_COLLECTOR_BEAN_NAME)
    CollectorWrapper metricCollectorWrapper(
            Collection<MetricProvider> metricProviders
    ) {
        return new CollectorWrapper(
                new MetricCollector(metricProviders, null)
        );
    }

    public static class CollectorWrapper {
        private final MetricCollector metricCollector;

        public CollectorWrapper(MetricCollector metricCollector) {
            this.metricCollector = metricCollector;
        }

        @EventListener
        public void init(ContextRefreshedEvent event) {
            if (metricCollector != null) {
                metricCollector.init();
            }
        }

        @EventListener
        public void close(ContextClosedEvent event) {
            if (metricCollector != null) {
                metricCollector.close();
            }
        }
    }
}
