package ru.yandex.direct.common.configuration;

import java.time.Duration;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import io.lettuce.core.ClientOptions;
import io.lettuce.core.RedisURI;
import io.lettuce.core.cluster.ClusterClientOptions;
import io.lettuce.core.cluster.ClusterTopologyRefreshOptions;
import io.lettuce.core.cluster.RedisClusterClient;
import io.lettuce.core.resource.ClientResources;
import io.lettuce.core.resource.DefaultClientResources;
import io.lettuce.core.resource.DnsResolvers;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.context.annotation.Lazy;

import ru.yandex.direct.common.lettuce.LettuceConnectionProvider;
import ru.yandex.direct.config.EssentialConfiguration;
import ru.yandex.direct.db.config.DbConfig;
import ru.yandex.direct.db.config.DbConfigFactory;
import ru.yandex.direct.dbutil.configuration.DbUtilConfiguration;
import ru.yandex.direct.utils.HostPort;

@Configuration
@Import({
        EssentialConfiguration.class,
        DbUtilConfiguration.class,
})
public class RedisConfiguration {
    public static final Duration LETTUCE_UPDATE_PERIOD = Duration.ofMinutes(1);
    public static final Duration LETTUCE_REFRESH_TIMEOUT = Duration.ofSeconds(30);
    public static final int LETTUCE_COMPUTATION_POOL_SIZE = 4;
    public static final int LETTUCE_IO_POOL_SIZE = 4;

    public static final String REDIS_DBNAME = "redis";
    public static final String LETTUCE_CLIENT = "lettuceClient";
    public static final String LETTUCE = "lettuce";

    public static final String REDIS_CACHE_DBNAME_PARAM = "${redis_cache.name}";
    public static final String LETTUCE_CACHE_CLIENT = "lettuceCacheClient";
    public static final String LETTUCE_CACHE = "lettuceCache";

    @Lazy
    @Bean
    ClientResources clientResources() {
        return DefaultClientResources.builder()
                .dnsResolver(DnsResolvers.JVM_DEFAULT)
                .computationThreadPoolSize(LETTUCE_COMPUTATION_POOL_SIZE)
                .ioThreadPoolSize(LETTUCE_IO_POOL_SIZE)
                .build();
    }

    public static RedisClusterClient createRedisClient(
            DbConfig redisConfig,
            ClientResources clientResources) {
        List<RedisURI> clusterMembers = redisConfig.getHosts().stream()
                .map(HostPort::fromString)
                .map(hostPort -> RedisURI.Builder.redis(hostPort.getHost(), hostPort.getPort())
                        .withPassword(redisConfig.getPass()).build())
                .collect(Collectors.toList());
        RedisClusterClient client = RedisClusterClient.create(clientResources, clusterMembers);
        Duration timeout = Duration.ofMillis(redisConfig.getConnectTimeout(TimeUnit.MILLISECONDS));
        client.setDefaultTimeout(timeout);

        ClusterTopologyRefreshOptions topologyRefreshOptions = ClusterTopologyRefreshOptions.builder()
                .enableAllAdaptiveRefreshTriggers()
                .adaptiveRefreshTriggersTimeout(LETTUCE_REFRESH_TIMEOUT)
                .enablePeriodicRefresh(LETTUCE_UPDATE_PERIOD)
                .build();
        ClusterClientOptions options = ClusterClientOptions.builder()
                .autoReconnect(true)
                .topologyRefreshOptions(topologyRefreshOptions)
                .cancelCommandsOnReconnectFailure(true)
                .validateClusterNodeMembership(false)
                .disconnectedBehavior(ClientOptions.DisconnectedBehavior.REJECT_COMMANDS)
                .build();
        client.setOptions(options);
        return client;
    }

    @Lazy
    @Bean(name = LETTUCE_CLIENT)
    RedisClusterClient lettuceClient(DbConfigFactory dbConfigFactory, ClientResources clientResources) {
        return createRedisClient(dbConfigFactory.get("redis"), clientResources);
    }

    @Lazy
    @Bean(name = LETTUCE)
    LettuceConnectionProvider lettuce(
            DbConfigFactory dbConfigFactory,
            @Qualifier(LETTUCE_CLIENT) RedisClusterClient client) {
        DbConfig redisConfig = dbConfigFactory.get(REDIS_DBNAME);
        int maxAttempts = redisConfig.getRequestRetries();
        return new LettuceConnectionProvider(REDIS_DBNAME, client, maxAttempts);
    }

    @Lazy
    @Bean(name = LETTUCE_CACHE_CLIENT)
    RedisClusterClient lettuceCacheCluster(
            DbConfigFactory dbConfigFactory,
            @Value(REDIS_CACHE_DBNAME_PARAM) String redisCacheName,
            ClientResources clientResources) {
        return createRedisClient(dbConfigFactory.get(redisCacheName), clientResources);
    }

    @Lazy
    @Bean(name = LETTUCE_CACHE)
    LettuceConnectionProvider lettuceCache(
            DbConfigFactory dbConfigFactory,
            @Qualifier(LETTUCE_CACHE_CLIENT) RedisClusterClient client,
            @Value(REDIS_CACHE_DBNAME_PARAM) String redisCacheName) {
        DbConfig redisConfig = dbConfigFactory.get(redisCacheName);
        int maxAttempts = redisConfig.getRequestRetries();
        return new LettuceConnectionProvider(redisCacheName, client, maxAttempts);
    }
}
