package ru.yandex.direct.common.configuration

import com.yandex.ydb.core.auth.AuthProvider
import com.yandex.ydb.core.grpc.GrpcTransport
import com.yandex.ydb.core.rpc.RpcTransport
import com.yandex.ydb.table.SessionRetryContext
import com.yandex.ydb.table.TableClient
import com.yandex.ydb.table.rpc.grpc.GrpcTableRpc
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.beans.factory.annotation.Value
import org.springframework.context.annotation.Bean
import org.springframework.context.annotation.Configuration
import org.springframework.context.annotation.Import
import ru.yandex.direct.tvm.TvmIntegration
import ru.yandex.direct.ydb.YdbPath
import ru.yandex.direct.ydb.client.YdbClient
import ru.yandex.direct.ydb.client.YdbSessionProperties
import ru.yandex.direct.ydb.tvm.YdbTvmAuthProvider
import java.time.Duration

@Configuration
@Import(TvmIntegrationConfiguration::class)
open class UacYdbConfiguration {

    companion object {
        const val UAC_YDB_PATH_BEAN = "UacYdbPath"
        const val UAC_YDB_RPC_TRANSPORT_BEAN = "UacYdbRpcTransport"
        const val UAC_YDB_TABLE_CLIENT_BEAN = "UacYdbTableClient"
        const val UAC_YDB_SESSION_PROPERTIES_BEAN = "UacYdbSessionProperties"
        const val UAC_YDB_CLIENT_BEAN = "UacYdbClient"

        const val UAC_YDB_INFO_BEAN = "UacYdbInfo"
    }

    @Bean
    open fun authProvider(tvmIntegration: TvmIntegration): AuthProvider = YdbTvmAuthProvider(tvmIntegration)

    @Bean(UAC_YDB_PATH_BEAN)
    open fun ydbPath(
        @Value("\${ydb.cluster}") cluster: String,
        @Value("\${ydb.uac.db_name}") dbName: String,
    ): YdbPath = YdbPath.of(cluster, dbName)

    @Bean(UAC_YDB_RPC_TRANSPORT_BEAN)
    open fun grpcTransport(
        @Value("\${ydb.endpoint}") endpoint: String,
        @Qualifier(UAC_YDB_PATH_BEAN) dbPath: YdbPath,
        authProvider: AuthProvider,
    ): RpcTransport = GrpcTransport.forEndpoint(endpoint, dbPath.path)
        .withAuthProvider(authProvider)
        .build()

    @Bean(UAC_YDB_TABLE_CLIENT_BEAN)
    open fun tableClient(
        @Qualifier(UAC_YDB_RPC_TRANSPORT_BEAN) rpcTransport: RpcTransport,
        @Value("\${ydb.uac.session_pool_min_size}") sessionPoolMinSize: Int,
        @Value("\${ydb.uac.session_pool_max_size}") sessionPoolMaxSize: Int,
    ): TableClient = TableClient.newClient(GrpcTableRpc.useTransport(rpcTransport))
        .sessionPoolSize(sessionPoolMinSize, sessionPoolMaxSize)
        .build()

    @Bean(UAC_YDB_SESSION_PROPERTIES_BEAN)
    open fun ydbProperties(
        @Value("\${ydb.uac.max_query_retries}") maxQueryRetries: Int,
        @Value("\${ydb.uac.retry_not_found}") retryNotFound: Boolean,
    ): YdbSessionProperties = YdbSessionProperties.builder()
        .withMaxQueryRetries(maxQueryRetries)
        .withRetryNotFound(retryNotFound)
        .build()

    @Bean(UAC_YDB_CLIENT_BEAN)
    open fun ydbClient(
        @Qualifier(UAC_YDB_TABLE_CLIENT_BEAN) tableClient: TableClient,
        @Qualifier(UAC_YDB_SESSION_PROPERTIES_BEAN) uacYdbSessionProperties: YdbSessionProperties,
        @Value("\${ydb.uac.query_timeout}") queryTimeoutSeconds: Long,
    ): YdbClient {
        val sessionRetryContext = SessionRetryContext.create(tableClient)
            .maxRetries(uacYdbSessionProperties.maxQueryRetries)
            .retryNotFound(uacYdbSessionProperties.isRetryNotFound)
            .build()
        return YdbClient(sessionRetryContext, Duration.ofSeconds(queryTimeoutSeconds))
    }
}
