package ru.yandex.direct.common.db;

import java.time.Duration;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.jooq.InsertOnDuplicateStep;
import org.jooq.impl.DSL;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.dbschema.ppcdict.tables.records.PpcPropertiesRecord;
import ru.yandex.direct.dbutil.SqlUtils;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static ru.yandex.direct.dbschema.ppcdict.tables.PpcProperties.PPC_PROPERTIES;

/**
 * Позволяет удобно создавать/обновлять/загружать строковые пары "ключ-значение"
 * из таблицы {@code ppc_properties}.
 */
@Component
@ParametersAreNonnullByDefault
public class PpcPropertiesSupport {
    public static final int NAME_MAX_LENGTH = PPC_PROPERTIES.NAME.getDataType().length();
    private final DslContextProvider dslContextProvider;

    @Autowired
    public PpcPropertiesSupport(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    private void checkName(String name) {
        if (name.length() > NAME_MAX_LENGTH) {
            throw new IllegalArgumentException("Property name '" + name + "'is too long");
        }
    }

    /**
     * Аналог get, но возвращает Optional
     */
    public Optional<String> find(String name) {
        return Optional.ofNullable(get(name));
    }

    /**
     * @param name название ключа
     * @return значение ключа, null - если такого ключа нет.
     */
    @Nullable
    public String get(String name) {
        return getByNames(Collections.singleton(name)).get(name);
    }

    /**
     * Получить объект PpcProperty, кеширующий полученное значение на expiredDuration
     * </p>
     * Кеширование полезно, только если вы получаете значение проперти чаще, чем получение объекта PpcProperty
     * То есть, например, получаете объект PpcProperty в конструкторе и сохраняете его как поле класса,
     * а property.get() делаете в методе.
     */
    public <T> PpcProperty<T> get(PpcPropertyName<T> name, Duration expiredDuration) {
        return new PpcProperty<>(this, name, expiredDuration);
    }

    @Nonnull
    public <T> PpcProperty<T> get(PpcPropertyName<T> name) {
        return new PpcProperty<>(this, name);
    }

    /**
     * Получить список всех пропертей, имена которых начинаются с переданного префикса
     */
    public Map<String, String> getByPrefix(String prefix) {
        if (prefix.isEmpty()) {
            throw new IllegalStateException("Prefix cannot be empty");
        }

        return dslContextProvider.ppcdict()
                .select(PPC_PROPERTIES.NAME, PPC_PROPERTIES.VALUE)
                .from(PPC_PROPERTIES)
                .where(PPC_PROPERTIES.NAME.startsWith(prefix))
                .fetchMap(PPC_PROPERTIES.NAME, PPC_PROPERTIES.VALUE);
    }

    /**
     * Получить список имен всех пропертей, имена которых начинаются с переданного префикса
     */
    public List<String> getAllNamesByPrefix(String prefix) {
        if (prefix.isEmpty()) {
            throw new IllegalStateException("Prefix cannot be empty");
        }

        return dslContextProvider.ppcdict()
                .select(PPC_PROPERTIES.NAME)
                .from(PPC_PROPERTIES)
                .where(PPC_PROPERTIES.NAME.startsWith(prefix))
                .fetch(PPC_PROPERTIES.NAME);
    }

    /**
     * Получить список всех пропертей. Если значения в базе нет, в результат добавлятся ключ с null-значением
     */
    public Map<String, String> getByNames(Collection<String> names) {
        if (names.isEmpty()) {
            return Collections.emptyMap();
        }
        names.forEach(this::checkName);

        Map<String, String> keysToValues = dslContextProvider.ppcdict()
                .select(PPC_PROPERTIES.NAME, PPC_PROPERTIES.VALUE)
                .from(PPC_PROPERTIES)
                .where(PPC_PROPERTIES.NAME.in(names))
                .fetchMap(PPC_PROPERTIES.NAME, PPC_PROPERTIES.VALUE);

        names.forEach(n -> keysToValues.putIfAbsent(n, null));

        return keysToValues;
    }

    /**
     * Хелпер метод парсить результат работы функции {@link #getByNames(Collection)}
     */
    public static <T> Optional<T> deserialize(PpcPropertyName<T> name, Map<String, String> nameToValueMap) {
        return Optional.ofNullable(nameToValueMap.get(name.getName()))
                .map(value -> name.getType().deserialize(value));
    }

    /**
     * Получить список всех пропертей с временем последнего изменения. Если значения в базе нет, в результат
     * добавлятся ключ с null-значением
     */
    public Map<String, PpcPropertyData<String>> getFullByNames(Collection<String> names) {
        if (names.isEmpty()) {
            return Collections.emptyMap();
        }
        names.forEach(this::checkName);

        Map<String, PpcPropertyData<String>> keysToValues = dslContextProvider.ppcdict()
                .select(PPC_PROPERTIES.NAME, PPC_PROPERTIES.VALUE, PPC_PROPERTIES.LAST_CHANGE)
                .from(PPC_PROPERTIES)
                .where(PPC_PROPERTIES.NAME.in(names))
                .fetchInto(PPC_PROPERTIES)
                .stream()
                .collect(Collectors.toMap(PpcPropertiesRecord::getName, p -> new PpcPropertyData<>(p.getValue(),
                        p.getLastChange())));

        names.forEach(n -> keysToValues.putIfAbsent(n, null));

        return keysToValues;
    }

    /**
     * Получить список всех пропертей
     */
    public Map<String, String> getAllProperties() {
        return dslContextProvider.ppcdict()
                .select(PPC_PROPERTIES.NAME, PPC_PROPERTIES.VALUE)
                .from(PPC_PROPERTIES)
                .fetchMap(PPC_PROPERTIES.NAME, PPC_PROPERTIES.VALUE);
    }

    public void set(PpcPropertyName<?> ppcPropertyName, String value) {
        set(dslContextProvider.ppcdict(), ppcPropertyName.getName(), value);
    }

    /**
     * Создает пару ключ-значение. Если такой ключ уже существует, значение в таблице обновляется
     *
     * @param name  название ключа
     * @param value значение ключа
     */
    public void set(String name, String value) {
        set(dslContextProvider.ppcdict(), name, value);
    }

    /**
     * То же самое, что {@link #set(String, String)}, только можно выполнить внутри открытой транзакции.
     *
     * @param dslContext открытая транзакция в PPCDICT
     * @param name       название ключа
     * @param value      значение ключа
     */
    public void set(DSLContext dslContext, String name, String value) {
        checkName(name);
        InsertOnDuplicateStep<PpcPropertiesRecord> insertStep = dslContext
                .insertInto(PPC_PROPERTIES, PPC_PROPERTIES.NAME, PPC_PROPERTIES.VALUE)
                .values(name, value);
        SqlUtils.onConflictUpdate(insertStep, Collections.singleton(PPC_PROPERTIES.VALUE))
                .execute();
    }

    /**
     * Реализация операции Compare-And-Swap - обновить значение для ключа name,
     * но только в том случае, если текущее значение совпадает с oldValue
     * <p>
     * Отсуствие строки считается эквивалентным oldValue == null
     *
     * @param name     название ключа
     * @param oldValue текущее значение ключа
     * @param newValue новое значение ключа
     * @return - удалось ли обновить значение
     */
    public boolean cas(String name, @Nullable String oldValue, @Nullable String newValue) {
        checkName(name);
        int affected;
        if (oldValue == null) {
            affected = dslContextProvider.ppcdict()
                    .insertInto(PPC_PROPERTIES, PPC_PROPERTIES.NAME, PPC_PROPERTIES.VALUE)
                    .values(name, newValue)
                    .onDuplicateKeyUpdate()
                    .set(PPC_PROPERTIES.VALUE,
                            DSL.decode()
                                    .when(MySQLDSL.values(PPC_PROPERTIES.VALUE).isNull(), newValue)
                                    .otherwise(PPC_PROPERTIES.VALUE))
                    // TODO: если значение не было изменено, здесь возвращается 1 вместо ожидаемого 0
                    .execute();
        } else {
            affected = dslContextProvider.ppcdict()
                    .update(PPC_PROPERTIES)
                    .set(PPC_PROPERTIES.VALUE, newValue)
                    .where(PPC_PROPERTIES.NAME.eq(name))
                    .and(PPC_PROPERTIES.VALUE.eq(oldValue))
                    .execute();

        }
        return affected > 0;
    }

    public boolean remove(PpcPropertyName<?> ppcPropertyName) {
        return remove(ppcPropertyName.getName());
    }

    /**
     * Удалить значение указанного ключа
     *
     * @param name название ключа
     * @return - удалось ли удалить (или такого ключа и не было)
     */
    public boolean remove(String name) {
        checkName(name);
        int affected = dslContextProvider.ppcdict()
                .deleteFrom(PPC_PROPERTIES)
                .where(PPC_PROPERTIES.NAME.eq(name))
                .execute();
        return affected > 0;
    }
}
