package ru.yandex.direct.common.db;

import java.time.Duration;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Suppliers;
import org.jooq.DSLContext;

import ru.yandex.direct.dbschema.ppcdict.Tables;

/**
 * Представляет собой типизированный маппинг значений строчки из таблицы {@link Tables#PPC_PROPERTIES}
 * Фабричные методы для создания объектов находятся в классе PpcPropertiesSupport
 */
@ParametersAreNonnullByDefault
public class PpcProperty<T> {
    private PpcPropertyName<T> name;
    private Supplier<T> value;
    private PpcPropertiesSupport ppcPropertiesSupport;

    PpcProperty(PpcPropertiesSupport ppcPropertiesSupport, PpcPropertyName<T> name, @Nullable Duration expire) {
        this.name = name;
        this.ppcPropertiesSupport = ppcPropertiesSupport;

        if (expire != null) {
            this.value = Suppliers.memoizeWithExpiration(
                    () -> deserialize(ppcPropertiesSupport.get(name.getName())),
                    expire.toMillis(),
                    TimeUnit.MILLISECONDS);
        } else {
            this.value = () -> deserialize(ppcPropertiesSupport.get(name.getName()));
        }
    }

    PpcProperty(PpcPropertiesSupport ppcPropertiesSupport, PpcPropertyName<T> name) {
        this(ppcPropertiesSupport, name, null);
    }

    /**
     * @return значение проперти из базы
     */
    @Nullable
    public T get() {
        return value.get();
    }

    /**
     * @return значение из базы или {@code defaultValue}, если занчения не было
     */
    public T getOrDefault(T defaultValue) {
        T result = get();
        return result != null ? result : defaultValue;
    }

    /**
     * Аналог get, но возвращает Optional
     */
    public Optional<T> find() {
        return Optional.ofNullable(get());
    }

    /**
     * Установить значение {@param value} проперти в базе
     */
    public void set(T value) {
        ppcPropertiesSupport.set(name.getName(), name.getType().serialize(value));
    }

    /**
     * Установить значение {@code value} проперти в базе в открытой транзакции {@code dslContext}
     */
    public void set(DSLContext dslContext, T value) {
        ppcPropertiesSupport.set(dslContext, name.getName(), name.getType().serialize(value));
    }

    /**
     * Удалить значение проперти из базы
     *
     * @return удалось ли удалить (или такого ключа и не было)
     */
    public boolean remove() {
        return ppcPropertiesSupport.remove(name.getName());
    }

    /**
     * Обновить значение для ключа name, но только в том случае, если текущее значение совпадает с oldValue
     *
     * @return удалось ли обновить значение
     */
    public boolean cas(@Nullable T oldValue, @Nullable T newValue) {
        return ppcPropertiesSupport
                .cas(name.getName(), serialize(oldValue), serialize(newValue));
    }

    public PpcPropertyName<T> getName() {
        return name;
    }

    @Nullable
    private T deserialize(@Nullable String value) {
        if (value == null) {
            return null;
        } else {
            return name.getType().deserialize(value);
        }
    }

    @Nullable
    private String serialize(@Nullable T value) {
        if (value == null) {
            return null;
        } else {
            return name.getType().serialize(value);
        }
    }
}
