package ru.yandex.direct.common.db;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import static java.time.format.DateTimeFormatter.ISO_LOCAL_DATE;
import static java.time.format.DateTimeFormatter.ISO_LOCAL_DATE_TIME;
import static ru.yandex.direct.utils.JsonUtils.fromJson;
import static ru.yandex.direct.utils.JsonUtils.toJson;

public interface PpcPropertyType<T> {
    PpcPropertyType<Boolean> BOOLEAN = new PpcPropertyType.BooleanType();
    PpcPropertyType<String> STRING = new PpcPropertyType.StringType();
    PpcPropertyType<Integer> INTEGER = new PpcPropertyType.IntegerType();
    PpcPropertyType<Long> LONG = new PpcPropertyType.LongType();
    PpcPropertyType<Double> DOUBLE = new PpcPropertyType.DoubleType();
    PpcPropertyType<LocalDateTime> LOCAL_DATE_TIME = new PpcPropertyType.LocalDateTimeType();
    PpcPropertyType<LocalDate> LOCAL_DATE = new PpcPropertyType.LocalDateType();
    PpcPropertyType<List<Integer>> INT_LIST = new PpcPropertyType.IntegerListType();
    PpcPropertyType<Set<Integer>> INT_SET = new PpcPropertyType.IntegerSet();
    PpcPropertyType<Set<Integer>> SHARDS_SET = new PpcPropertyType.ShardSet();
    PpcPropertyType<List<Long>> LONG_LIST = new PpcPropertyType.LongListType();
    PpcPropertyType<Set<Long>> LONG_SET = new PpcPropertyType.LongSet();
    PpcPropertyType<Set<String>> STRING_SET = new PpcPropertyType.StringSet();
    PpcPropertyType<List<String>> STRING_LIST = new PpcPropertyType.StringList();
    PpcPropertyType<Map<Long, Long>> LONG_TO_LONG_MAP = new PpcPropertyType.LongToLongMap();
    PpcPropertyType<Map<String, String>> STRING_TO_STRING_MAP = new PpcPropertyType.StringToStringMap();
    PpcPropertyType<Map<String, String>> STRING_TO_STRING_MAP_JSON = new PpcPropertyType.StringToStringMapJson();
    PpcPropertyType<Map<String, Integer>> STRING_TO_INTEGER_MAP = new PpcPropertyType.StringToIntegerMap();
    PpcPropertyType<Map<String, Double>> STRING_TO_DOUBLE_MAP = new PpcPropertyType.StringToDoubleMap();
    PpcPropertyType<Map<Integer, Map<String, Double>>> INTEGER_TO_STRING_TO_DOUBLE_MAP =
            new PpcPropertyType.IntegerToStringToDoubleMap();

    String UNSUPPORTED_TYPE = "Unsupported type";

    T deserialize(String value);

    String serialize(T value);

    default String reSerialize(String value) {
        return serialize(deserialize(value));
    }

    class BooleanType implements PpcPropertyType<Boolean> {
        BooleanType() {
        }

        @Override
        public Boolean deserialize(String value) {
            return value.trim().equals("1") || value.equals("true") ?
                    Boolean.TRUE : Boolean.FALSE;
        }

        @Override
        public String serialize(Boolean value) {
            return value != null && value ? "1" : "0";
        }
    }

    class IntegerType implements PpcPropertyType<Integer> {
        IntegerType() {
        }

        @Override
        public Integer deserialize(String value) {
            try {
                return Integer.valueOf(value);
            } catch (NumberFormatException e) {
                throw new PpcPropertyParseException(UNSUPPORTED_TYPE, e);
            }
        }

        @Override
        public String serialize(Integer value) {
            return value.toString();
        }
    }

    class StringType implements PpcPropertyType<String> {
        StringType() {
        }

        @Override
        public String deserialize(String value) {
            return value;
        }

        @Override
        public String serialize(String value) {
            return value;
        }
    }

    class LongType implements PpcPropertyType<Long> {
        LongType() {
        }

        @Override
        public Long deserialize(String value) {
            try {
                return Long.valueOf(value);
            } catch (NumberFormatException e) {
                throw new PpcPropertyParseException(UNSUPPORTED_TYPE, e);
            }
        }

        @Override
        public String serialize(Long value) {
            return value.toString();
        }
    }

    class DoubleType implements PpcPropertyType<Double> {
        DoubleType() {
        }

        @Override
        public Double deserialize(String value) {
            try {
                return Double.valueOf(value);
            } catch (NumberFormatException e) {
                throw new PpcPropertyParseException(UNSUPPORTED_TYPE, e);
            }
        }

        @Override
        public String serialize(Double value) {
            return value.toString();
        }
    }

    class LocalDateTimeType implements PpcPropertyType<LocalDateTime> {
        private static final DateTimeFormatter LOCALE_DATE_TIME_FORMATTER = DateTimeFormatter
                .ofPattern("yyyy-MM-dd HH:mm:ss.nnnnnnnnn", Locale.ENGLISH)
                .withZone(ZoneId.systemDefault());

        private static final DateTimeFormatter DATE_TIME_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");

        LocalDateTimeType() {
        }

        @Override
        public LocalDateTime deserialize(String value) {
            return parseISODateTime(value);
        }

        private LocalDateTime parseISODateTime(String value) {
            try {
                return ISO_LOCAL_DATE_TIME.parse(value, LocalDateTime::from);
            } catch (DateTimeParseException e) {
                return parseDateTime(value);
            }
        }

        private LocalDateTime parseDateTime(String value) {
            try {
                return DATE_TIME_FORMATTER.parse(value, LocalDateTime::from);
            } catch (DateTimeParseException e) {
                return parseLocaleDateTime(value);
            }
        }

        private LocalDateTime parseLocaleDateTime(String value) {
            try {
                return LOCALE_DATE_TIME_FORMATTER.parse(value, LocalDateTime::from);
            } catch (DateTimeParseException e) {
                throw new PpcPropertyParseException(UNSUPPORTED_TYPE, e);
            }
        }

        @Override
        public String serialize(LocalDateTime value) {
            return value.format(ISO_LOCAL_DATE_TIME);
        }
    }

    class LocalDateType implements PpcPropertyType<LocalDate> {
        private static final DateTimeFormatter DATE_FORMATTER = DateTimeFormatter.ofPattern("yyyyMMdd");

        LocalDateType() {
        }

        @Override
        public LocalDate deserialize(String value) {
            return parseISODate(value);
        }

        private LocalDate parseISODate(String value) {
            try {
                return ISO_LOCAL_DATE.parse(value, LocalDate::from);
            } catch (DateTimeParseException e) {
                return parseDate(value);
            }
        }

        private LocalDate parseDate(String value) {
            try {
                return DATE_FORMATTER.parse(value, LocalDate::from);
            } catch (DateTimeParseException e) {
                throw new PpcPropertyParseException(UNSUPPORTED_TYPE, e);
            }
        }

        @Override
        public String serialize(LocalDate value) {
            return value.format(ISO_LOCAL_DATE);
        }
    }

    class LongListType implements PpcPropertyType<List<Long>> {
        LongListType() {
        }

        @Override
        public List<Long> deserialize(String value) {
            if (value.isEmpty()) {
                return Collections.emptyList();
            }
            try {
                return Arrays.stream(value.split(","))
                        .map(String::trim)
                        .map(Long::valueOf)
                        .collect(Collectors.toList());
            } catch (NumberFormatException e) {
                throw new PpcPropertyParseException(UNSUPPORTED_TYPE, e);
            }
        }

        @Override
        public String serialize(List<Long> value) {
            return value.stream().map(Objects::toString).collect(Collectors.joining(","));
        }
    }

    class IntegerListType implements PpcPropertyType<List<Integer>> {
        IntegerListType() {
        }

        @Override
        public List<Integer> deserialize(String value) {
            if (value.isEmpty()) {
                return Collections.emptyList();
            }
            try {
                return Arrays.stream(value.split(","))
                        .map(String::trim)
                        .map(Integer::valueOf)
                        .collect(Collectors.toList());
            } catch (NumberFormatException e) {
                throw new PpcPropertyParseException(UNSUPPORTED_TYPE, e);
            }
        }

        @Override
        public String serialize(List<Integer> value) {
            return value.stream().map(Objects::toString).collect(Collectors.joining(","));
        }
    }

    class LongSet implements PpcPropertyType<Set<Long>> {
        LongSet() {
        }

        @Override
        public Set<Long> deserialize(String value) {
            if (value.isEmpty()) {
                return Collections.emptySet();
            }
            try {
                return Arrays.stream(value.split(","))
                        .map(String::trim)
                        .map(Long::valueOf)
                        .collect(Collectors.toSet());
            } catch (NumberFormatException e) {
                throw new PpcPropertyParseException(UNSUPPORTED_TYPE, e);
            }
        }

        @Override
        public String serialize(Set<Long> value) {
            return value.stream().map(Objects::toString).collect(Collectors.joining(","));
        }
    }

    class IntegerSet implements PpcPropertyType<Set<Integer>> {
        IntegerSet() {
        }

        @Override
        public Set<Integer> deserialize(String value) {
            if (value.isEmpty()) {
                return Collections.emptySet();
            }
            try {
                return Arrays.stream(value.split(","))
                        .map(String::trim)
                        .map(Integer::valueOf)
                        .collect(Collectors.toSet());
            } catch (NumberFormatException e) {
                throw new PpcPropertyParseException(UNSUPPORTED_TYPE, e);
            }
        }

        @Override
        public String serialize(Set<Integer> value) {
            return value.stream().map(Objects::toString).collect(Collectors.joining(","));
        }
    }

    class ShardSet extends IntegerSet {
        ShardSet() {
        }
    }

    class StringSet implements PpcPropertyType<Set<String>> {
        StringSet() {
        }

        @Override
        public Set<String> deserialize(String value) {
            if (value.isEmpty()) {
                return Collections.emptySet();
            }
            return Arrays.stream(value.split(","))
                    .map(String::trim)
                    .collect(Collectors.toSet());
        }

        @Override
        public String serialize(Set<String> value) {
            return String.join(",", value);
        }
    }

    class StringList implements PpcPropertyType<List<String>> {
        StringList() {
        }

        @Override
        public List<String> deserialize(String value) {
            if (value.isEmpty()) {
                return Collections.emptyList();
            }
            return Arrays.stream(value.split(","))
                    .map(String::trim)
                    .collect(Collectors.toList());
        }

        @Override
        public String serialize(List<String> value) {
            return String.join(",", value);
        }
    }

    class LongToLongMap implements PpcPropertyType<Map<Long, Long>> {
        LongToLongMap() {
        }

        @Override
        public Map<Long, Long> deserialize(String value) {
            if (value.isEmpty()) {
                return Collections.emptyMap();
            }
            return StreamEx.of(value.split(","))
                    .map(keyValue -> keyValue.trim().split("="))
                    .toMap(kv -> Long.valueOf(kv[0].trim()), kv -> Long.valueOf(kv[1].trim()));
        }

        @Override
        public String serialize(Map<Long, Long> value) {
            return String.join(",", EntryStream.of(value).mapKeyValue((k, v) -> k + "=" + v));
        }
    }

    class StringToStringMap implements PpcPropertyType<Map<String, String>> {
        StringToStringMap() {
        }

        @Override
        public Map<String, String> deserialize(String value) {
            if (value.isEmpty()) {
                return Collections.emptyMap();
            }
            return StreamEx.of(value.split(","))
                    .map(keyValue -> keyValue.trim().split("="))
                    .toMap(kv -> kv[0].trim(), kv -> kv[1].trim());
        }

        @Override
        public String serialize(Map<String, String> value) {
            return String.join(",", EntryStream.of(value).mapKeyValue((k, v) -> k + "=" + v));
        }
    }

    class StringToStringMapJson implements PpcPropertyType<Map<String, String>> {
        StringToStringMapJson() {
        }

        @Override
        public Map<String, String> deserialize(String value) {
            if (value.isEmpty()) {
                return Collections.emptyMap();
            }
            return fromJson(value, Map.class);
        }

        @Override
        public String serialize(Map<String, String> value) {
            return toJson(value);
        }
    }


    class StringToIntegerMap implements PpcPropertyType<Map<String, Integer>> {
        @Override
        public Map<String, Integer> deserialize(String value) {
            if (value.isEmpty()) {
                return Collections.emptyMap();
            }
            return StreamEx.of(value.split(","))
                    .map(keyValue -> keyValue.trim().split("="))
                    .toMap(kv -> kv[0].trim(), kv -> Integer.parseInt(kv[1].trim()));
        }

        @Override
        public String serialize(Map<String, Integer> value) {
            return String.join(",", EntryStream.of(value).mapKeyValue((k, v) -> k + "=" + v));
        }
    }

    static Map<String, Double> deserializeMapStringToDouble(String value) {
        if (value.isEmpty()) {
            return Collections.emptyMap();
        }
        return StreamEx.of(value.split(","))
                .map(keyValue -> keyValue.trim().split("="))
                .toMap(kv -> kv[0].trim(), kv -> Double.parseDouble(kv[1].trim()));
    }

    static String serializeMapStringToDouble(Map<String, Double> value) {
        return String.join(",", EntryStream.of(value).mapKeyValue((k, v) -> k + "=" + v));
    }

    class StringToDoubleMap implements PpcPropertyType<Map<String, Double>> {
        @Override
        public Map<String, Double> deserialize(String value) {
            return deserializeMapStringToDouble(value);
        }

        @Override
        public String serialize(Map<String, Double> value) {
            return serializeMapStringToDouble(value);
        }
    }

    class IntegerToStringToDoubleMap implements PpcPropertyType<Map<Integer, Map<String, Double>>> {
        @Override
        public Map<Integer, Map<String, Double>> deserialize(String value) {
            if (value.isEmpty()) {
                return Collections.emptyMap();
            }
            return StreamEx.of(value.split(";"))
                    .map(keyValue -> keyValue.trim().split(":"))
                    .toMap(kv -> Integer.parseInt(kv[0].trim()), kv -> deserializeMapStringToDouble(kv[1].trim()));
        }

        @Override
        public String serialize(Map<Integer, Map<String, Double>> value) {
            return String.join(";",
                    EntryStream.of(value).mapKeyValue((k, v) -> k + ":" + serializeMapStringToDouble(v)));
        }
    }
}
