package ru.yandex.direct.common.jackson.jaxbmodule;

import java.io.IOException;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonDeserializer;
import com.fasterxml.jackson.databind.deser.BeanDeserializerBase;
import com.fasterxml.jackson.databind.deser.std.DelegatingDeserializer;
import com.fasterxml.jackson.dataformat.xml.deser.WrapperHandlingDeserializer;
import org.apache.commons.lang3.StringUtils;

/**
 * Принимает, при сериализации бина, строку из одних только пробелов как null значение.
 * <p>
 * Можно было бы сделать над {@link BeanDeserializerBase}, но оригинальный {@link WrapperHandlingDeserializer}
 * становится поверх {@link BeanDeserializerBase} и перенастраивает парсер для unwrapped списков, но
 * похоже эта перенастройка ломается на {@code VALUE_STRING}, поэтому в этом десериализаторе мы сами управляем
 * {@link WrapperHandlingDeserializer} и отдаём ему десериализацию, только в случае,
 * если токен не пробельная строка
 */
public class IgnoreWhitespaceWrapperHandlingDeserializer extends DelegatingDeserializer {
    public IgnoreWhitespaceWrapperHandlingDeserializer(WrapperHandlingDeserializer delegate) {
        super(delegate);
    }

    @Override
    protected JsonDeserializer<?> newDelegatingInstance(JsonDeserializer<?> newDelegatee) {
        if (newDelegatee instanceof WrapperHandlingDeserializer) {
            return new IgnoreWhitespaceWrapperHandlingDeserializer((WrapperHandlingDeserializer) newDelegatee);
        }
        return newDelegatee;
    }

    @Override
    public Object deserialize(JsonParser p, DeserializationContext ctxt) throws IOException {
        if (p.getCurrentToken() == JsonToken.VALUE_STRING && StringUtils.isBlank(p.getText())) {
            return getNullValue(ctxt);
        }
        return _delegatee.deserialize(p, ctxt);
    }
}
