package ru.yandex.direct.common.jackson.jaxbmodule;

import java.io.IOException;

import javax.xml.bind.JAXBElement;
import javax.xml.namespace.QName;
import javax.xml.stream.XMLStreamReader;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;
import com.fasterxml.jackson.databind.BeanProperty;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.JsonDeserializer;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.deser.ContextualDeserializer;
import com.fasterxml.jackson.databind.deser.std.StdDeserializer;
import com.fasterxml.jackson.databind.jsontype.TypeDeserializer;
import com.fasterxml.jackson.dataformat.xml.deser.FromXmlParser;

// CHECKSTYLE:OFF
/**
 * Десериализатор JAXBElement.
 * <p>
 * Источники:
 * <ul>
 * <li><a href="https://github.com/FasterXML/jackson-datatype-jdk8/blob/master/src/main/java/com/fasterxml/jackson/datatype/jdk8/OptionalDeserializer.java">https://github.com/FasterXML/jackson-datatype-jdk8/blob/master/src/main/java/com/fasterxml/jackson/datatype/jdk8/OptionalDeserializer.java</a></li>
 * <li><a href="https://stackoverflow.com/questions/21002930/jackson-xml-binding-handling-null-vs-empty-string/21662427#21662427">https://stackoverflow.com/questions/21002930/jackson-xml-binding-handling-null-vs-empty-string/21662427#21662427</a></li>
 * </ul>
 */
// CHECKSTYLE:ON
public class JaxbElementDeserializer extends StdDeserializer<JAXBElement<?>> implements ContextualDeserializer {
    public static final String XSI_NAMESPACE_URI = "http://www.w3.org/2001/XMLSchema-instance";

    private final JavaType fullType;
    private final JavaType referenceType;
    private final TypeDeserializer valueTypeDeserializer;
    private final JsonDeserializer<?> valueDeserializer;

    public JaxbElementDeserializer(JavaType fullType, JavaType refType, TypeDeserializer typeDeser,
                                   JsonDeserializer<?> valueDeser) {
        super(fullType);
        this.fullType = fullType;
        referenceType = refType;
        valueTypeDeserializer = typeDeser;
        valueDeserializer = valueDeser;
    }

    @SuppressWarnings("ReferenceEquality")
    protected JaxbElementDeserializer withResolved(JavaType refType,
                                                   TypeDeserializer typeDeser, JsonDeserializer<?> valueDeser) {
        if ((refType == referenceType) && (valueDeser == valueDeserializer)
                && (typeDeser == valueTypeDeserializer)) {
            return this;
        }
        return new JaxbElementDeserializer(fullType, refType, typeDeser, valueDeser);
    }


    @Override
    public JAXBElement<?> getNullValue(DeserializationContext ctxt) {
        JAXBElement jaxbElement = new JAXBElement<>(new QName(""), referenceType.getRawClass(), null);
        jaxbElement.setNil(true);
        return jaxbElement;
    }

    @Override
    public JsonDeserializer<?> createContextual(DeserializationContext ctxt, BeanProperty property)
            throws JsonMappingException {
        JsonDeserializer<?> deserializer = valueDeserializer;
        TypeDeserializer typeDeserializer = valueTypeDeserializer;
        JavaType refType = referenceType;

        if (deserializer == null) {
            deserializer = ctxt.findContextualValueDeserializer(refType, property);
        } else {
            deserializer = ctxt.handleSecondaryContextualization(deserializer, property, refType);
        }
        if (typeDeserializer != null) {
            typeDeserializer = typeDeserializer.forProperty(property);
        }
        return withResolved(refType, typeDeserializer, deserializer);
    }

    @Override
    @SuppressWarnings("unchecked")
    public JAXBElement<?> deserialize(JsonParser jp, DeserializationContext ctxt) throws IOException {
        boolean isNil = false;

        if (jp instanceof FromXmlParser) {
            FromXmlParser fxp = (FromXmlParser) jp;

            XMLStreamReader reader = fxp.getStaxReader();
            if (reader.isStartElement() && reader.getAttributeCount() > 0) {
                String atVal = reader.getAttributeValue(XSI_NAMESPACE_URI, "nil");
                if (atVal != null && Boolean.parseBoolean(atVal)) {
                    isNil = true;
                }
            }
            // игнорируем весь оставшийся элемент {"nil": "true"}
            if (isNil && jp.getCurrentToken() == JsonToken.START_OBJECT) {
                jp.skipChildren();
            }
            if (isNil) {
                return getNullValue(ctxt);
            }
        }

        Object value = (valueTypeDeserializer == null)
                ? valueDeserializer.deserialize(jp, ctxt)
                : valueDeserializer.deserializeWithType(jp, ctxt, valueTypeDeserializer);

        return new JAXBElement(new QName(""), referenceType.getRawClass(), value);
    }
}
