package ru.yandex.direct.common.jetty;

import org.apache.commons.lang3.builder.ReflectionToStringBuilder;

import ru.yandex.direct.config.DirectConfig;

/**
 * Configuration for embedded jetty server
 */
@SuppressWarnings("checkstyle:visibilitymodifier")
public class JettyConfig {
    /**
     * network interface (IP or hostname) to bind to
     * if null, bind to all interfaces
     */
    public final String host;

    /**
     * http port
     */
    public final int port;

    /**
     * sslPort, no ssl initizlization if sslPort is 0
     */
    public final int sslPort;

    /**
     * The time in milliseconds that the connection can be idle before it is closed.
     */
    public final int idleTimeout;

    /**
     * The time in milliseconds for the graceful shutdown
     */
    public final int stopTimeout;

    /**
     * The time in milliseconds, after that time process will be halted.
     * Should be greater than stopTimeout.
     */
    public final long hardStopTimeout;

    /**
     * Settings for SSL keystore
     */
    public final Keystore keystore;

    /**
     * If enabled, log recorded to slf4j logger with name ACCESS_LOG.log
     */
    public final boolean requestLogEnabled;

    /**
     * If enabled, stack traces are shown in the error pages
     */
    public final boolean showStacks;

    /**
     * If enabled, print jetty logo at start
     */
    public final boolean logAsciiArt;

    public static class Keystore {
        /**
         * Path to resource with ssl keystore
         */
        public final String resourcePath;

        /**
         * Password for ssl keystore
         */
        public final String pass;

        public Keystore(DirectConfig conf) {
            resourcePath = conf.findString("resource_path").orElse(null);
            pass = conf.findString("pass").orElse(null);
        }

        @Override
        public String toString() {
            return ReflectionToStringBuilder.toString(this);
        }
    }

    /**
     * Thread pool settings
     */
    public final ThreadPool threadPool;

    public static class ThreadPool {
        public final int minThreads;
        public final int maxThreads;
        public final int idleTimeout;
        public final int acceptors;
        public final int selectors;

        public ThreadPool(DirectConfig conf) {
            minThreads = conf.findInt("min_threads").orElse(10);
            maxThreads = conf.findInt("max_threads").orElse(200);
            idleTimeout = conf.findInt("idle_timeout").orElse(60_000);

            // defaults from jetty source code
            int cores = Runtime.getRuntime().availableProcessors();
            acceptors = conf.findInt("acceptors").orElse(Math.max(1, Math.min(4, cores / 8)));
            selectors = conf.findInt("selectors").orElse(Math.max(1, Math.min(4, cores / 2)));
        }

        @Override
        public String toString() {
            return ReflectionToStringBuilder.toString(this);
        }
    }

    /**
     * Initialize Jetty config from part of DirectConfig
     *
     * @param conf
     */
    public JettyConfig(DirectConfig conf) {
        host = conf.findString("host").orElse(null);
        port = conf.findInt("port").orElse(80);
        sslPort = conf.findInt("ssl_port").orElse(0);
        idleTimeout = conf.findInt("idle_timeout").orElse(60_000);
        stopTimeout = conf.findInt("stop_timeout").orElse(20_000);
        hardStopTimeout = conf.findInt("hard_stop_timeout").orElse(stopTimeout + 30_000);
        keystore = new Keystore(conf.getBranch("keystore"));
        threadPool = new ThreadPool(conf.getBranch("thread_pool"));
        requestLogEnabled = conf.findBoolean("request_log_enabled").orElse(true);
        logAsciiArt = conf.findBoolean("log_ascii_art").orElse(true);
        showStacks = conf.findBoolean("show_stacks").orElse(false);
    }

    @Override
    public String toString() {
        return ReflectionToStringBuilder.toString(this);
    }
}
