package ru.yandex.direct.common.jetty;

import java.lang.management.ManagementFactory;
import java.util.ArrayList;
import java.util.List;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.atomic.AtomicLong;

import javax.annotation.Nonnull;
import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import javax.servlet.Servlet;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.core.LoggerContext;
import org.eclipse.jetty.jmx.MBeanContainer;
import org.eclipse.jetty.server.ConnectionFactory;
import org.eclipse.jetty.server.HttpConfiguration;
import org.eclipse.jetty.server.HttpConnectionFactory;
import org.eclipse.jetty.server.SecureRequestCustomizer;
import org.eclipse.jetty.server.Server;
import org.eclipse.jetty.server.ServerConnector;
import org.eclipse.jetty.server.Slf4jRequestLog;
import org.eclipse.jetty.server.SslConnectionFactory;
import org.eclipse.jetty.server.handler.ErrorHandler;
import org.eclipse.jetty.server.handler.HandlerCollection;
import org.eclipse.jetty.server.handler.RequestLogHandler;
import org.eclipse.jetty.server.handler.StatisticsHandler;
import org.eclipse.jetty.servlet.ErrorPageErrorHandler;
import org.eclipse.jetty.servlet.ServletHandler;
import org.eclipse.jetty.servlet.ServletHolder;
import org.eclipse.jetty.util.resource.Resource;
import org.eclipse.jetty.util.ssl.SslContextFactory;
import org.eclipse.jetty.util.thread.QueuedThreadPool;
import org.eclipse.jetty.webapp.WebAppContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.context.support.WebApplicationContextUtils;

import ru.yandex.direct.env.DevEnvironment;
import ru.yandex.direct.liveresource.LiveResourceFactory;
import ru.yandex.misc.sunMiscSignal.SunMiscSignal;

import static com.google.common.base.Preconditions.checkState;

public class JettyLauncher {
    private static final Logger logger = LoggerFactory.getLogger(JettyLauncher.class);
    private static final AtomicLong POOL_NUMBER = new AtomicLong();

    public static final int MAX_REQUEST_SIZE = 16 * 1024 * 1024;
    public static final String ACCESS_LOGGER_NAME = "ACCESS_LOG.log";

    protected final JettyConfig conf;

    protected List<WebAppContext> webAppContexts = new ArrayList<>();
    protected List<Servlet> servlets = new ArrayList<>();
    protected List<String> servletPaths = new ArrayList<>();
    private volatile Server server;

    public JettyLauncher(JettyConfig conf) {
        this.conf = conf;
        logger.debug("JettyLauncher initialized with config {}", conf);
    }

    public static JettyLauncher server(JettyConfig conf) {
        return new JettyLauncher(conf);
    }

    public JettyLauncher withWebApp(String webAppRoot, String path) {
        return withWebAppContext(new WebAppContext(webAppRoot, path));
    }

    public JettyLauncher withWebAppContext(WebAppContext context) {
        webAppContexts.add(context);
        return this;
    }

    public JettyLauncher withServlet(Servlet servlet, String path) {
        servlets.add(servlet);
        servletPaths.add(path);
        return this;
    }

    public JettyLauncher withDefaultWebApp(ClassLoader classLoader, String path) {
        WebAppContext webapp = new WebAppContext();
        webapp.setResourceBase(classLoader.getResource("webapp").toExternalForm());
        webapp.setContextPath(path);

        //Не отдаём stacktrace наружу в продакшен и ТС окружениях
        ErrorHandler errorHandler = new ErrorPageErrorHandler();
        errorHandler.setShowStacks(conf.showStacks);
        webapp.setErrorHandler(errorHandler);

        webAppContexts.add(webapp);
        return this;
    }

    /**
     * start jetty and wait for finish
     */
    public void run() throws Exception {
        setTermHandler();
        start();

        try {
            server.join(); // IS-NOT-COMPLETABLE-FUTURE-JOIN
            logger.info("jetty stopped");
        } finally {
            stop();
        }
    }

    /**
     * start jetty
     */
    @PostConstruct
    public void start() {
        checkState(server == null, "Double jetty initialization");
        server = configureServer();

        HandlerCollection handlers = new HandlerCollection();
        for (WebAppContext ctx : webAppContexts) {
            // если не можем запустить приложение - не запускаем и jetty
            ctx.setThrowUnavailableOnStartupException(true);
            ctx.setInitParameter("org.eclipse.jetty.servlet.Default.dirAllowed", "false");
            ctx.setMaxFormContentSize(MAX_REQUEST_SIZE);
            handlers.addHandler(ctx);
        }
        if (!servlets.isEmpty()) {
            ServletHandler servletHandler = new ServletHandler();
            for (int i = 0; i < servlets.size(); i++) {
                servletHandler.addServletWithMapping(new ServletHolder(servlets.get(i)), servletPaths.get(i));
            }
            handlers.addHandler(servletHandler);
        }

        if (conf.requestLogEnabled) {
            handlers.addHandler(createRequestLogHandler());
        }

        // Needed for graceful shutdown, see Server.setStopTimeout
        StatisticsHandler statsHandler = new StatisticsHandler();
        statsHandler.setHandler(handlers);
        server.setHandler(statsHandler);

        try {
            server.start();

            processStartHooks();

            if (conf.logAsciiArt) {
                logAsciiArt();
            }
            logger.info("jetty started, ports={}, {}", conf.port, conf.sslPort);
        } catch (Exception e) {
            stop();
            throw new JettyLaunchException("Can't start Jetty", e);
        }
    }

    @PreDestroy
    public void stop() {
        if (server != null) {
            try {
                server.stop();
            } catch (Exception e) {
                throw new JettyLaunchException("Stop failed", e);
            }
        }
    }

    /**
     * Вызов хуков, подвешенных на старт jetty
     */
    private void processStartHooks() {
        for (var jettyCtx : webAppContexts) {
            var appCtx = WebApplicationContextUtils.getWebApplicationContext(jettyCtx.getServletContext());
            if (appCtx != null) {
                appCtx.getBeansOfType(JettyLauncherStartListener.class)
                        .values()
                        .forEach(JettyLauncherStartListener::onStart);
            }
        }
    }

    private void logAsciiArt() {
        if (DevEnvironment.isDeveloperLaptop()) {
            String art = LiveResourceFactory.get("classpath:///jetty-started-art.txt").getContent();
            logger.info(art);
        }
    }

    protected RequestLogHandler createRequestLogHandler() {
        RequestLogHandler requestLogHandler = new RequestLogHandler();
        Slf4jRequestLog requestLog = new Slf4jRequestLog();
        requestLog.setLoggerName(ACCESS_LOGGER_NAME);
        requestLog.setLogLatency(true);
        requestLogHandler.setRequestLog(requestLog);
        return requestLogHandler;
    }

    private Server configureServer() {
        QueuedThreadPool threadPool = configureThreadPool();
        Server srv = new Server(threadPool);

        // выставляем статистики наружу через JMX
        MBeanContainer mbContainer = new MBeanContainer(ManagementFactory.getPlatformMBeanServer());
        srv.addEventListener(mbContainer);
        srv.addBean(mbContainer);

        final ServerConnector connector = new ServerConnector(srv,
                conf.threadPool.acceptors, conf.threadPool.selectors,
                new HttpConnectionFactory(getHttpConfiguration()));
        connector.setHost(conf.host);
        connector.setPort(conf.port);
        connector.setIdleTimeout(conf.idleTimeout);

        srv.addConnector(connector);

        if (conf.sslPort > 0) {
            SslContextFactory ctx = new SslContextFactory();
            ctx.setKeyStoreResource(Resource.newClassPathResource(conf.keystore.resourcePath));
            ctx.setKeyStorePassword(conf.keystore.pass);
            ConnectionFactory sslFactory = new SslConnectionFactory(ctx, "http/1.1");

            HttpConfiguration sslHttpConfig = getHttpConfiguration();
            sslHttpConfig.addCustomizer(new SecureRequestCustomizer());
            ConnectionFactory httpFactory = new HttpConnectionFactory(sslHttpConfig);

            ServerConnector sslConnector = new ServerConnector(srv,
                    conf.threadPool.acceptors, conf.threadPool.selectors,
                    sslFactory, httpFactory);
            sslConnector.setPort(conf.sslPort);
            sslConnector.setIdleTimeout(conf.idleTimeout);

            srv.addConnector(sslConnector);
        }

        // graceful shutdown
        srv.setStopTimeout(conf.stopTimeout);
        srv.setStopAtShutdown(true);

        return srv;
    }

    @Nonnull
    protected QueuedThreadPool configureThreadPool() {
        final QueuedThreadPool threadPool = new QueuedThreadPool();
        threadPool.setMaxThreads(conf.threadPool.maxThreads);
        threadPool.setMinThreads(conf.threadPool.minThreads);
        threadPool.setIdleTimeout(conf.threadPool.idleTimeout);
        threadPool.setName("jetty-worker-" + POOL_NUMBER.incrementAndGet());
        return threadPool;
    }

    protected void setTermHandler() {
        SunMiscSignal.handle("TERM", signalName -> {
            if (this.server != null) {
                logger.info("Signal {} catched, ", signalName);

                startHardTimeoutThread();

                logger.info("Let's try to graceful shutdown");
                try {
                    this.server.stop();
                } catch (Exception e) {
                    logger.error("Shutdown error", e);
                }

                LoggerContext context = (LoggerContext) LogManager.getContext(false);
                context.stop();
            } else {
                logger.info("Signal {} catched, but server is not inited yet", signalName);
            }
        });
    }

    private void startHardTimeoutThread() {
        logger.info("Start thread for hard halt after {} ms", conf.hardStopTimeout);
        var timer = new Timer("JettyTermHandlerTimer", true);
        timer.schedule(new TimerTask() {
            @Override
            public void run() {
                try {
                    logger.error("Hard stop timeout expired, process will be halted");
                    // логгера может уже не быть - пишем ещё в stderr
                    System.err.println("Hard stop timeout expired, process will be stopped");
                } finally {
                    Runtime.getRuntime().halt(1);
                }
            }
        }, conf.hardStopTimeout);
    }

    protected HttpConfiguration getHttpConfiguration() {
        final HttpConfiguration httpConfiguration = new HttpConfiguration();
        httpConfiguration.setRequestHeaderSize(65536);
        httpConfiguration.setResponseHeaderSize(65536);
        httpConfiguration.setSendServerVersion(false);
        httpConfiguration.setSendDateHeader(false);
        return httpConfiguration;
    }

    public static class JettyLaunchException extends RuntimeException {
        public JettyLaunchException(String message) {
            super(message);
        }

        public JettyLaunchException(String message, Throwable t) {
            super(message, t);
        }
    }

}
