package ru.yandex.direct.common.jooqmapper;

import java.util.function.BiConsumer;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Record;
import org.jooq.Table;
import org.jooq.TableField;

import static com.google.common.base.Preconditions.checkState;

/**
 * Этот класс отвечает за чтение из базы и запись в базу
 * одного конкретного поля таблицы.
 *
 * @param <R> Тип записи в базе
 * @param <M> Тип модели, из которой записывается значение в поле таблицы
 * @param <V> Тип поля таблицы
 * @param <X> Тип поля модели
 */
@ParametersAreNonnullByDefault
public final class FieldMapper<R extends Record, M, V, X> {

    /**
     * Поле в базе данных, значение которого читается и записывается
     * с помощью заданных {@link FieldReader} и {@link FieldWriter}
     */
    private final TableField<R, V> tableField;

    /**
     * Отвечает за чтение значения из модели,
     * конвертацию этого значения в формат базы
     * и запись в Jooq-билдер для последующей записи
     * в базу сконвертированного значения
     */
    private FieldWriter<R, M, V, X> fieldWriter;

    /**
     * Отвечает за чтение значения из записи в таблице,
     * конвертацию его в формат модели
     * и запись данных непосредственно в поле/поля модели.
     */
    private FieldReader<R, M, V, X> fieldReader;

    /**
     * Параметр управляет возможностью записи в базу с помощью данного маппера
     */
    private boolean writeToDbEnabled = true;

    /**
     * Параметр управляет возможностью чтения из базы с помощью данного маппера
     */
    private boolean readFromDbEnabled = true;

    FieldMapper(TableField<R, V> tableField,
                FieldWriter<R, M, V, X> fieldWriter,
                FieldReader<R, M, V, X> fieldReader) {
        this.tableField = tableField;
        this.fieldWriter = fieldWriter;
        this.fieldReader = fieldReader;
    }

    /**
     * @return поле таблицы, с которым работает данный маппер
     */
    public TableField<R, V> getTableField() {
        return tableField;
    }

    /**
     * @return таблица, с полем которой работает данный маппер
     */
    public Table<R> getTable() {
        return tableField.getTable();
    }

    /**
     * Вычисляет значение заданного поля в бд
     */
    public Object getDbFieldValue(M model) {
        assertWritingEnabled();
        return fieldWriter.getDbFieldValue(model);
    }

    /**
     * Читает из переданной записи в БД значение заданного поля
     * и выставляет в модели одно или несколько полей
     */
    public void fromDb(M model, Record record) {
        assertReadingEnabled();
        fieldReader.apply(model, record);
    }

    /**
     * @return true, если записи в базу включена
     */
    public boolean isWritingToDbEnabled() {
        return writeToDbEnabled;
    }

    /**
     * @return true, если чтение из базы включено
     */
    public boolean isReadingFromDbEnabled() {
        return readFromDbEnabled;
    }

    /**
     * Устанавливает функцию, получающую из модели значение для записи в поле таблицы в базе.
     * Тип получаемого значения может отличаться от типа поля в базе, для установки конвертера
     * используйте {@link #convertToDbBy(Function)}
     *
     * @param getter функция, получающая из модели значение для записи в поле таблицы в базе
     * @return возвращает себя
     */
    public FieldMapper<R, M, V, X> toDbBy(Function<M, X> getter) {
        fieldWriter.by(getter);
        return this;
    }

    /**
     * Устанавливает функцию, которая выставляет в модели значение, полученное из записи в БД.
     * <p>
     * Функция принимает уже сконвертированное значение из типа данных поля в базе во внутренний тип данных.
     * Потенциально функция на основе полученного значения может производить с моделью любые действия,
     * например, выставить несколько полей одновременно. Однако такой подход не рекомендуется, так как
     * усложняет понимание кода. Чтобы установить несколько полей модели на основе одного поля в базе,
     * создавайте для каждого поля собственные мапперы и регистрируйте в {@link OldJooqMapper}. При регистрации
     * нескольких мапперов для чтения из одного поля в базе, может быть полезным запретить им запись в базу
     * с помощью метода {@link #disableWritingToDb}
     *
     * @param setter функция, принимающая экземпляр модели и сконвертированное во внутренний формат значение
     *               из базы, которая должна выставить поля модели на основе этого значения.
     * @return возвращает себя
     */
    public FieldMapper<R, M, V, X> fromDbBy(BiConsumer<M, X> setter) {
        fieldReader.by(setter);
        return this;
    }

    /**
     * Устанавливает функцию-конвертер, которая принимает значение, полученное из модели,
     * и возвращает значение для записи в базу. В основном используется для приведения/конвертации типов,
     * когда тип данных в модели не соответствует типу данных в базе.
     *
     * @param converter функция-конвертер
     * @return возвращает себя
     */
    public FieldMapper<R, M, V, X> convertToDbBy(Function<X, V> converter) {
        fieldWriter.convertBy(converter);
        return this;
    }

    /**
     * Устанавливает функцию-конвертер, которая принимает значение, полученное из базы,
     * и возвращает значение для записи в модель. В основном используется для приведения/конвертации типов,
     * когда тип данных в модели не соответствует типу данных в базе.
     *
     * @param converter функция-конвертер
     * @return возвращает себя
     */
    public FieldMapper<R, M, V, X> convertFromDbBy(Function<V, X> converter) {
        fieldReader.convertBy(converter);
        return this;
    }

    /**
     * Устанавливает значение, которое будет записано в базу в случае,
     * если из модели извлечено значение {@code null}.
     *
     * @param defaultValue дефолтное значение для записи в базу
     * @return возвращает себя
     */
    public FieldMapper<R, M, V, X> withDefaultValueForDb(X defaultValue) {
        fieldWriter.withDefaultValue(defaultValue);
        return this;
    }

    /**
     * Устанавливает значение, которое будет записано в модель в случае,
     * если извлеченное из базы значение равно {@code null} (после применения конвертера).
     *
     * @param defaultValue дефолтное значение для записи в модель
     * @return возвращает себя
     */
    public FieldMapper<R, M, V, X> withDefaultValueForModel(X defaultValue) {
        fieldReader.withDefaultValue(defaultValue);
        return this;
    }

    /**
     * Устанавливает поведение, при котором, если из модели извлечено значение {@code null},
     * то в базу будет записано дефолтное значение, определенное в самой таблице.
     *
     * @return возвращает себя
     */
    public FieldMapper<R, M, V, X> withDatabaseDefault() {
        fieldWriter.withDatabaseDefault(true);
        return this;
    }

    /**
     * Отключает запись в базу для данного маппера
     *
     * @return возвращает себя
     */
    public FieldMapper<R, M, V, X> disableWritingToDb() {
        writeToDbEnabled = false;
        return this;
    }

    /**
     * Отключает чтение из таблицы для данного маппера
     *
     * @return возвращает себя
     */
    public FieldMapper<R, M, V, X> disableReadingFromDb() {
        readFromDbEnabled = false;
        return this;
    }

    private void assertWritingEnabled() {
        checkState(writeToDbEnabled, "something went wrong: writing to database is disabled for this mapper, "
                + "check isWritingToDbEnabled() first");
    }

    private void assertReadingEnabled() {
        checkState(readFromDbEnabled, "something went wrong: reading from database is disabled for this mapper, "
                + "check isReadingFromDbEnabled() first");
    }
}
