package ru.yandex.direct.common.jooqmapper;

import java.util.Map;
import java.util.Set;
import java.util.function.BiConsumer;
import java.util.function.Function;

import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import org.apache.commons.lang3.StringUtils;
import org.jooq.Record;
import org.jooq.TableField;

import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;

import static java.util.Collections.emptySet;
import static java.util.function.Function.identity;

/**
 * Здесь содержатся удобные методы для создания мапперов {@link FieldMapper}
 * для маппинга данных из базы на поля моделей.
 */
public class FieldMapperFactory {
    private FieldMapperFactory() {
    }

    /**
     * Возвращает маппер, у которого типы поля модели и поля в базе данных отличаются.
     * В возвращаемом маппере уже установлены геттер и сеттер на основе переданной
     * {@link ModelProperty}.
     * <p>
     * Требует явной установки конвертеров! Если чтение или запись для маппера будут запрещены
     * (см. {@link FieldMapper#disableReadingFromDb()} и {@link FieldMapper#disableWritingToDb()}),
     * то соответствующие конвертеры устанавливать необязательно.
     *
     * @param tableField поле базы данных
     * @param property   поле модели
     * @param <R>        тип записи, к которой относится поле таблицы БД
     * @param <M>        тип модели
     * @param <V>        тип поля базы данных
     * @param <X>        тип поля модели
     * @return экземпляр {@link FieldMapper} с установленными геттером и сеттером, но без конвертеров
     */
    public static <R extends Record, M extends Model, V, X> FieldMapper<R, M, V, X> convertibleField(
            TableField<R, V> tableField, ModelProperty<? super M, X> property) {
        return new FieldMapper<>(tableField,
                new FieldWriter<R, M, V, X>(tableField)
                        .by(property::get),
                new FieldReader<R, M, V, X>(tableField)
                        .by(property::set));
    }

    /**
     * Аналог метода {@link #convertibleField(TableField, ModelProperty)}
     * с подсказкой компилятору в виде указателя на класс – для явного определения типа
     * в местах, где другого способа нет.
     *
     * @param modelType игнорируется
     * @see #convertibleField(TableField, ModelProperty)
     */
    public static <R extends Record, M extends Model, V, X> FieldMapper<R, M, V, X> convertibleField(
            TableField<R, V> tableField, ModelProperty<? super M, X> property,
            @SuppressWarnings("unused") Class<M> modelType) {
        return new FieldMapper<>(tableField,
                new FieldWriter<R, M, V, X>(tableField)
                        .by(property::get),
                new FieldReader<R, M, V, X>(tableField)
                        .by(property::set));
    }

    /**
     * Возвращает маппер, у которого типы поля модели и поля в базе данных совпадают.
     * В возвращаемом маппере уже установлены геттер и сеттер на основе переданной
     * {@link ModelProperty} и Null-конвертеры, которые не производят никаких действий над значениями.
     *
     * @param tableField поле базы данных
     * @param property   поле модели
     * @param <R>        тип записи, к которой относится поле таблицы БД
     * @param <M>        тип модели
     * @param <V>        тип поля базы данных и модели
     * @return готовый к применению экземпляр {@link FieldMapper} с Null-конвертерами
     */
    public static <R extends Record, M extends Model, V> FieldMapper<R, M, V, V> field(
            TableField<R, V> tableField, ModelProperty<M, V> property) {
        return convertibleField(tableField, property)
                .convertToDbBy(identity())
                .convertFromDbBy(identity());
    }

    /**
     * Возвращает маппер множества в базе данных, которое представлено отдельными полями в модели
     *
     * @param tableField поле базы данных
     * @param mapping    отображение значения в сете поля базы данных на поле в модели
     * @param <R>        тип записи, к которой относится поле таблицы БД
     * @param <M>        тип модели
     * @return готовый к применению экземпляр {@link FieldMapper} для Integer
     */
    public static <R extends Record, M extends Model> FieldMapper<R, M, String, String> setField(
            TableField<R, String> tableField, Map<String, ModelProperty<? super M, Boolean>> mapping) {
        Function<M, String> getter = model -> EntryStream.of(mapping)
                .mapValues(prop -> prop.get(model))
                .nonNullValues()
                .filterValues(v -> v)
                .keys()
                .joining(",");

        BiConsumer<M, String> setter = (model, setValue) -> {
            Set<String> entries = setValue != null ?
                    Sets.newHashSet(StringUtils.split(setValue, ",")) : emptySet();
            EntryStream.of(mapping).forKeyValue((entry, prop) -> {
                if (entries.contains(entry)) {
                    prop.set(model, Boolean.TRUE);
                } else {
                    prop.set(model, Boolean.FALSE);
                }
            });
        };

        return new FieldMapper<>(tableField,
                new FieldWriter<R, M, String, String>(tableField)
                        .convertBy(identity())
                        .by(getter),
                new FieldReader<R, M, String, String>(tableField)
                        .convertBy(identity())
                        .by(setter));
    }

    /**
     * Возвращает маппер, у которого типы поля модели и поля в базе данных отличаются.
     * <p>
     * Требует явной установки геттера значения из модели, сеттера значения в модели
     * и конвертеров в формат базы данных и обратно! Если чтение или запись для маппера будут запрещены
     * (см. {@link FieldMapper#disableReadingFromDb()} и {@link FieldMapper#disableWritingToDb()}),
     * то соответствующие геттер или сеттер, а так же конвертеры устанавливать необязательно.
     *
     * @param tableField      поле базы данных
     * @param modelClass      класс модели, нужен только для определения типов в месте вызова
     * @param modelFieldClass класс поля модели, нужен только для определения типов в месте вызова
     * @param <R>             тип записи, к которой относится поле таблицы БД
     * @param <M>             тип модели
     * @param <V>             тип поля базы данных
     * @param <X>             тип поля модели
     * @return экземпляр {@link FieldMapper} без геттера, сеттера и конвертеров
     */
    public static <R extends Record, M, V, X> FieldMapper<R, M, V, X> convertibleField(
            TableField<R, V> tableField, @SuppressWarnings("unused") Class<M> modelClass,
            @SuppressWarnings("unused") Class<X> modelFieldClass) {
        return new FieldMapper<>(tableField,
                new FieldWriter<>(tableField),
                new FieldReader<>(tableField));
    }

    /**
     * Возвращает маппер, у которого типы поля модели и поля в базе данных совпадают.
     * В возвращаемом маппере уже установлены Null-конвертеры, которые не производят
     * никаких действий над значениями.
     * <p>
     * Требует явной установки геттера значения из модели и сеттера значения в модели.
     * Если чтение или запись для маппера будут запрещены
     * (см. {@link FieldMapper#disableReadingFromDb()} и {@link FieldMapper#disableWritingToDb()}),
     * то соответствующие геттер или сеттер устанавливать необязательно.
     *
     * @param tableField поле базы данных
     * @param modelClass класс модели, нужен только для определения типов в месте вызова
     * @param <R>        тип записи, к которой относится поле таблицы БД
     * @param <M>        тип модели
     * @param <V>        тип поля базы данных и модели
     * @return экземпляр {@link FieldMapper} без геттера и сеттера c установленными Null-конвертерами
     */
    public static <R extends Record, M, V> FieldMapper<R, M, V, V> field(
            TableField<R, V> tableField, @SuppressWarnings("unused") Class<M> modelClass) {
        return new FieldMapper<>(tableField,
                new FieldWriter<R, M, V, V>(tableField)
                        .convertBy(identity()),
                new FieldReader<R, M, V, V>(tableField)
                        .convertBy(identity()));
    }
}
